import { restCall, restNewPostCall, userGenerate } from '../../../tests/_utils/index'
import { dbManager } from '../../../tests/db-manager'
import { expect, test, describe, afterAll, beforeAll, vi } from 'vitest'
import type { Pool } from 'pg'
import { HTTP_STATUS } from 'camino-common/src/http'
import { TestUser } from 'camino-common/src/tests-utils'
import { ADMINISTRATION_IDS } from 'camino-common/src/static/administrations'
import { AdministrationActiviteTypeEmail } from 'camino-common/src/administrations'
import { toCaminoDate } from 'camino-common/src/date'

console.info = vi.fn()
console.error = vi.fn()
let dbPool: Pool
beforeAll(async () => {
  const { knex: knexInstance, pool } = await dbManager.populateDb()
  dbPool = pool

  await knexInstance('utilisateurs').insert({
    id: 'deletedUserId',
    prenom: 'test',
    nom: 'test',
    email: null,
    keycloak_id: null,
    role: 'lecteur',
    administrationId: 'dea-guyane-01',
    dateCreation: toCaminoDate('2022-05-12'),
  })
})

afterAll(async () => {
  await dbManager.closeKnex()
})

describe('getAdministrationUtilisateurs', () => {
  test.each<[TestUser | undefined, boolean]>([
    [{ role: 'super' }, true],
    [
      {
        role: 'admin',
        administrationId: ADMINISTRATION_IDS['DREAL - AUVERGNE-RHÔNE-ALPES - SIÈGE DE LYON'],
      },
      true,
    ],
    [{ role: 'entreprise', entrepriseIds: [] }, false],
    [{ role: "bureau d'études", entrepriseIds: [] }, false],
    [{ role: 'defaut' }, false],
    [undefined, false],
  ])('utilisateur %s peur voir les utilisateurs: %s', async (user, lecture) => {
    await userGenerate(dbPool, { role: 'defaut' })
    await userGenerate(dbPool, { role: 'admin', administrationId: 'dea-guyane-01' })
    await userGenerate(dbPool, { role: 'admin', administrationId: 'dea-reunion-01' })

    const tested = await restCall(dbPool, '/rest/administrations/:administrationId/utilisateurs', { administrationId: 'dea-guyane-01' }, user)

    if (lecture) {
      expect(tested.statusCode).toBe(HTTP_STATUS.OK)
      expect(tested.body).toEqual([
        {
          administrationId: 'dea-guyane-01',
          email: 'admin-user-dea-guyane-01@camino.local',
          id: 'admin-user-dea-guyane-01',
          nom: 'nom-admin',
          prenom: 'prenom-admin',
          role: 'admin',
          telephone_fixe: null,
          telephone_mobile: null,
        },
      ])
    } else {
      expect(tested.statusCode).toBe(HTTP_STATUS.FORBIDDEN)
    }
  })
})

describe('administrationActiviteTypeEmails', () => {
  test.each<[TestUser | undefined, boolean]>([
    [{ role: 'super' }, true],
    [
      {
        role: 'admin',
        administrationId: ADMINISTRATION_IDS['DREAL - AUVERGNE-RHÔNE-ALPES - SIÈGE DE LYON'],
      },
      true,
    ],
    [{ role: 'entreprise', entrepriseIds: [] }, false],
    [{ role: "bureau d'études", entrepriseIds: [] }, false],
    [{ role: 'defaut' }, false],
    [undefined, false],
  ])('utilisateur %s peur gérer les emails associés à un type d’activité: %s', async (user, canEdit) => {
    let tested = await restCall(dbPool, '/rest/administrations/:administrationId/activiteTypeEmails', { administrationId: 'dea-guyane-01' }, user)

    if (canEdit) {
      expect(tested.statusCode).toBe(HTTP_STATUS.OK)
      expect(tested.body).toEqual([])

      const newActiviteTypeEmail: AdministrationActiviteTypeEmail = { activite_type_id: 'gra', email: 'toto@toto.com' }
      await restNewPostCall(dbPool, '/rest/administrations/:administrationId/activiteTypeEmails', { administrationId: 'dea-guyane-01' }, user, newActiviteTypeEmail)

      tested = await restCall(dbPool, '/rest/administrations/:administrationId/activiteTypeEmails', { administrationId: 'dea-guyane-01' }, user)
      expect(tested.statusCode).toBe(HTTP_STATUS.OK)
      expect(tested.body).toEqual([newActiviteTypeEmail])

      await restNewPostCall(dbPool, '/rest/administrations/:administrationId/activiteTypeEmails/delete', { administrationId: 'dea-guyane-01' }, user, newActiviteTypeEmail)
      tested = await restCall(dbPool, '/rest/administrations/:administrationId/activiteTypeEmails', { administrationId: 'dea-guyane-01' }, user)
      expect(tested.statusCode).toBe(HTTP_STATUS.OK)
      expect(tested.body).toEqual([])
    } else {
      expect(tested.statusCode).toBe(HTTP_STATUS.FORBIDDEN)
    }
  })
})
