import { Definition } from '../definition'
import { RegionId } from './region'
import { DepartementId } from './departement'
import { z } from 'zod'
import { map, toSorted } from '../typescript-tools'

const ADMINISTRATION_TYPE_IDS_ARRAY = ['aut', 'dre', 'min', 'ope', 'pre'] as const

export const ADMINISTRATION_TYPE_IDS = {
  AUTORITE: 'aut',
  DREAL: 'dre',
  MINISTERE: 'min',
  OPERATEUR: 'ope',
  PREFECTURE: 'pre',
} as const satisfies Record<string, AdministrationTypeId>

export const administrationTypeIdValidator = z.enum(ADMINISTRATION_TYPE_IDS_ARRAY)

export type AdministrationTypeId = z.infer<typeof administrationTypeIdValidator>

export type AdministrationType<T = AdministrationTypeId> = Omit<Definition<T>, 'description'>
export const ADMINISTRATION_TYPES: {
  [key in AdministrationTypeId]: AdministrationType<key>
} = {
  aut: { id: 'aut', nom: 'Autorité' },
  dre: { id: 'dre', nom: 'Dréal' },
  min: { id: 'min', nom: 'Ministère' },
  ope: { id: 'ope', nom: 'Opérateur' },
  pre: { id: 'pre', nom: 'Préfecture' },
}

export const sortedAdministrationTypes = Object.values(ADMINISTRATION_TYPES).sort((a, b) => a.nom.localeCompare(b.nom))
// prettier-ignore
export const IDS = [ "aut-97300-01", "aut-mrae-guyane-01", "dea-guadeloupe-01", "dea-guyane-01", "dea-martinique-01", "dea-mayotte-01", "dea-reunion-01", "dre-aura-01", "dre-bfc-01", "dre-bretagne-01", "dre-centre-val-de-loire-01", "dre-corse-01", "dre-grand-est-01", "dre-hauts-de-france-01", "dre-ile-de-france-01", "dre-normandie-01", "dre-nouvelle-aquitaine-01", "dre-occitanie-01", "dre-paca-01", "dre-pays-de-la-loire-01", "min-dajb-01", "min-mctrct-dgcl-01", "min-mtes-dgaln-01", "min-mtes-dgec-01", "min-mtes-dgpr-01", "ope-brgm-01", "ope-cacem-01", "ope-onf-973-01", "ope-ptmg-973-01", "pre-01053-01", "pre-02408-01", "pre-03190-01", "pre-04070-01", "pre-05061-01", "pre-06088-01", "pre-07186-01", "pre-08105-01", "pre-09122-01", "pre-10387-01", "pre-11069-01", "pre-12202-01", "pre-13203-01", "pre-14118-01", "pre-15014-01", "pre-16015-01", "pre-17300-01", "pre-18033-01", "pre-19272-01", "pre-21231-01", "pre-22278-01", "pre-23096-01", "pre-24322-01", "pre-25056-01", "pre-26362-01", "pre-27229-01", "pre-28085-01", "pre-29232-01", "pre-2A004-01", "pre-2B033-01", "pre-30189-01", "pre-31555-01", "pre-32013-01", "pre-33063-01", "pre-34172-01", "pre-35238-01", "pre-36044-01", "pre-37261-01", "pre-38185-01", "pre-39300-01", "pre-40192-01", "pre-41018-01", "pre-42218-01", "pre-43157-01", "pre-44109-01", "pre-45234-01", "pre-46042-01", "pre-47001-01", "pre-48095-01", "pre-49007-01", "pre-50502-01", "pre-51108-01", "pre-52121-01", "pre-53130-01", "pre-54395-01", "pre-55029-01", "pre-56260-01", "pre-57463-01", "pre-58194-01", "pre-59350-01", "pre-60057-01", "pre-61001-01", "pre-62041-01", "pre-63113-01", "pre-64445-01", "pre-65440-01", "pre-66136-01", "pre-67482-01", "pre-68066-01", "pre-69383-01", "pre-70550-01", "pre-71270-01", "pre-72181-01", "pre-73065-01", "pre-74010-01", "pre-75104-01", "pre-76540-01", "pre-77288-01", "pre-78646-01", "pre-79191-01", "pre-80021-01", "pre-81004-01", "pre-82121-01", "pre-83137-01", "pre-84007-01", "pre-85191-01", "pre-86194-01", "pre-87085-01", "pre-88160-01", "pre-89024-01", "pre-90010-01", "pre-91228-01", "pre-92050-01", "pre-93008-01", "pre-94028-01", "pre-95127-01", "pre-97105-01", "pre-97209-01", "pre-97302-01", "pre-97411-01", "pre-97611-01", ] as const

export const ADMINISTRATION_IDS = {
  'GENDARMERIE NATIONALE - GUYANE': 'aut-97300-01',
  'MRAE GUYANE': 'aut-mrae-guyane-01',
  'DEAL - GUADELOUPE': 'dea-guadeloupe-01',
  'DGTM - GUYANE': 'dea-guyane-01',
  'DEAL - MARTINIQUE': 'dea-martinique-01',
  'DEAL - MAYOTTE': 'dea-mayotte-01',
  'DEAL - LA RÉUNION': 'dea-reunion-01',
  'DREAL - AUVERGNE-RHÔNE-ALPES - SIÈGE DE LYON': 'dre-aura-01',
  'DREAL - BOURGOGNE-FRANCHE-COMTÉ - SIÈGE DE BESANÇON': 'dre-bfc-01',
  'DREAL - BRETAGNE': 'dre-bretagne-01',
  'DREAL - CENTRE-VAL-DE-LOIRE': 'dre-centre-val-de-loire-01',
  'DREAL - CORSE': 'dre-corse-01',
  'DREAL - GRAND EST - SIÈGE DE METZ': 'dre-grand-est-01',
  'DREAL - HAUTS-DE-FRANCE - SIÈGE DE LILLE': 'dre-hauts-de-france-01',
  'DRIEE - ÎLE-DE-FRANCE': 'dre-ile-de-france-01',
  'DREAL - NORMANDIE - SIÈGE DE ROUEN': 'dre-normandie-01',
  'DREAL - NOUVELLE-AQUITAINE - SIÈGE DE POITIERS': 'dre-nouvelle-aquitaine-01',
  'DREAL - OCCITANIE - SIÈGE DE TOULOUSE': 'dre-occitanie-01',
  "DREAL - PROVENCE-ALPES-CÔTE-D'AZUR": 'dre-paca-01',
  'DREAL - PAYS DE LA LOIRE': 'dre-pays-de-la-loire-01',
  "DAJ - MINISTÈRE DE L'ECONOMIE, DES FINANCES ET DE LA RELANCE": 'min-dajb-01',
  'DGCL/SDFLAE/FL1': 'min-mctrct-dgcl-01',
  'DGALN/DEB/EARM2': 'min-mtes-dgaln-01',
  'DGEC/DE/SD2/2A': 'min-mtes-dgec-01',
  'DGPR/SRT/SDRCP/BSSS': 'min-mtes-dgpr-01',
  BRGM: 'ope-brgm-01',
  CACEM: 'ope-cacem-01',
  'OFFICE NATIONAL DES FORÊTS': 'ope-onf-973-01',
  'PÔLE TECHNIQUE MINIER DE GUYANE': 'ope-ptmg-973-01',
  'PRÉFECTURE - AIN': 'pre-01053-01',
  'PRÉFECTURE - AISNE': 'pre-02408-01',
  'PRÉFECTURE - ALLIER': 'pre-03190-01',
  'PRÉFECTURE - ALPES-DE-HAUTE-PROVENCE': 'pre-04070-01',
  'PRÉFECTURE - HAUTES-ALPES': 'pre-05061-01',
  'PRÉFECTURE - ALPES-MARITIMES': 'pre-06088-01',
  'PRÉFECTURE - ARDÈCHE': 'pre-07186-01',
  'PRÉFECTURE - ARDENNES': 'pre-08105-01',
  'PRÉFECTURE - ARIÈGE': 'pre-09122-01',
  'PRÉFECTURE - AUBE': 'pre-10387-01',
  'PRÉFECTURE - AUDE': 'pre-11069-01',
  'PRÉFECTURE - AVEYRON': 'pre-12202-01',
  'PRÉFECTURE - BOUCHES-DU-RHÔNE': 'pre-13203-01',
  'PRÉFECTURE - CALVADOS': 'pre-14118-01',
  'PRÉFECTURE - CANTAL': 'pre-15014-01',
  'PRÉFECTURE - CHARENTE': 'pre-16015-01',
  'PRÉFECTURE - CHARENTE-MARITIME': 'pre-17300-01',
  'PRÉFECTURE - CHER': 'pre-18033-01',
  'PRÉFECTURE - CORRÈZE': 'pre-19272-01',
  "PRÉFECTURE - CÔTE-D'OR": 'pre-21231-01',
  "PRÉFECTURE - CÔTES-D'ARMOR": 'pre-22278-01',
  'PRÉFECTURE - CREUSE': 'pre-23096-01',
  'PRÉFECTURE - DORDOGNE': 'pre-24322-01',
  'PRÉFECTURE - DOUBS': 'pre-25056-01',
  'PRÉFECTURE - DRÔME': 'pre-26362-01',
  'PRÉFECTURE - EURE': 'pre-27229-01',
  'PRÉFECTURE - EURE-ET-LOIR': 'pre-28085-01',
  'PRÉFECTURE - FINISTÈRE': 'pre-29232-01',
  'PRÉFECTURE - CORSE-DU-SUD': 'pre-2A004-01',
  'PRÉFECTURE - HAUTE-CORSE': 'pre-2B033-01',
  'PRÉFECTURE - GARD': 'pre-30189-01',
  'PRÉFECTURE - HAUTE-GARONNE': 'pre-31555-01',
  'PRÉFECTURE - GERS': 'pre-32013-01',
  'PRÉFECTURE - GIRONDE': 'pre-33063-01',
  'PRÉFECTURE - HÉRAULT': 'pre-34172-01',
  'PRÉFECTURE - ILLE-ET-VILAINE': 'pre-35238-01',
  'PRÉFECTURE - INDRE': 'pre-36044-01',
  'PRÉFECTURE - INDRE-ET-LOIRE': 'pre-37261-01',
  'PRÉFECTURE - ISÈRE': 'pre-38185-01',
  'PRÉFECTURE - JURA': 'pre-39300-01',
  'PRÉFECTURE - LANDES': 'pre-40192-01',
  'PRÉFECTURE - LOIR-ET-CHER': 'pre-41018-01',
  'PRÉFECTURE - LOIRE': 'pre-42218-01',
  'PRÉFECTURE - HAUTE-LOIRE': 'pre-43157-01',
  'PRÉFECTURE - LOIRE-ATLANTIQUE': 'pre-44109-01',
  'PRÉFECTURE - LOIRET': 'pre-45234-01',
  'PRÉFECTURE - LOT': 'pre-46042-01',
  'PRÉFECTURE - LOT-ET-GARONNE': 'pre-47001-01',
  'PRÉFECTURE - LOZÈRE': 'pre-48095-01',
  'PRÉFECTURE - MAINE-ET-LOIRE': 'pre-49007-01',
  'PRÉFECTURE - MANCHE': 'pre-50502-01',
  'PRÉFECTURE - MARNE': 'pre-51108-01',
  'PRÉFECTURE - HAUTE-MARNE': 'pre-52121-01',
  'PRÉFECTURE - MAYENNE': 'pre-53130-01',
  'PRÉFECTURE - MEURTHE-ET-MOSELLE': 'pre-54395-01',
  'PRÉFECTURE - MEUSE': 'pre-55029-01',
  'PRÉFECTURE - MORBIHAN': 'pre-56260-01',
  'PRÉFECTURE - MOSELLE': 'pre-57463-01',
  'PRÉFECTURE - NIÈVRE': 'pre-58194-01',
  'PRÉFECTURE - NORD': 'pre-59350-01',
  'PRÉFECTURE - OISE': 'pre-60057-01',
  'PRÉFECTURE - ORNE': 'pre-61001-01',
  'PRÉFECTURE - PAS-DE-CALAIS': 'pre-62041-01',
  'PRÉFECTURE - PUY-DE-DÔME': 'pre-63113-01',
  'PRÉFECTURE - PYRÉNÉES-ATLANTIQUES': 'pre-64445-01',
  'PRÉFECTURE - HAUTES-PYRÉNÉES': 'pre-65440-01',
  'PRÉFECTURE - PYRÉNÉES-ORIENTALES': 'pre-66136-01',
  'PRÉFECTURE - BAS-RHIN': 'pre-67482-01',
  'PRÉFECTURE - HAUT-RHIN': 'pre-68066-01',
  'PRÉFECTURE - RHÔNE': 'pre-69383-01',
  'PRÉFECTURE - HAUTE-SAÔNE': 'pre-70550-01',
  'PRÉFECTURE - SAÔNE-ET-LOIRE': 'pre-71270-01',
  'PRÉFECTURE - SARTHE': 'pre-72181-01',
  'PRÉFECTURE - SAVOIE': 'pre-73065-01',
  'PRÉFECTURE - HAUTE-SAVOIE': 'pre-74010-01',
  'PRÉFECTURE - ÎLE-DE-FRANCE': 'pre-75104-01',
  'PRÉFECTURE - SEINE-MARITIME': 'pre-76540-01',
  'PRÉFECTURE - SEINE-ET-MARNE': 'pre-77288-01',
  'PRÉFECTURE - YVELINES': 'pre-78646-01',
  'PRÉFECTURE - DEUX-SÈVRES': 'pre-79191-01',
  'PRÉFECTURE - SOMME': 'pre-80021-01',
  'PRÉFECTURE - TARN': 'pre-81004-01',
  'PRÉFECTURE - TARN-ET-GARONNE': 'pre-82121-01',
  'PRÉFECTURE - VAR': 'pre-83137-01',
  'PRÉFECTURE - VAUCLUSE': 'pre-84007-01',
  'PRÉFECTURE - VENDÉE': 'pre-85191-01',
  'PRÉFECTURE - VIENNE': 'pre-86194-01',
  'PRÉFECTURE - HAUTE-VIENNE': 'pre-87085-01',
  'PRÉFECTURE - VOSGES': 'pre-88160-01',
  'PRÉFECTURE - YONNE': 'pre-89024-01',
  'PRÉFECTURE - TERRITOIRE DE BELFORT': 'pre-90010-01',
  'PRÉFECTURE - ESSONNE': 'pre-91228-01',
  'PRÉFECTURE - HAUTS-DE-SEINE': 'pre-92050-01',
  'PRÉFECTURE - SEINE-SAINT-DENIS': 'pre-93008-01',
  'PRÉFECTURE - VAL-DE-MARNE': 'pre-94028-01',
  "PRÉFECTURE - VAL-D'OISE": 'pre-95127-01',
  'PRÉFECTURE - GUADELOUPE': 'pre-97105-01',
  'PRÉFECTURE - MARTINIQUE': 'pre-97209-01',
  'PRÉFECTURE - GUYANE': 'pre-97302-01',
  'PRÉFECTURE - RÉUNION': 'pre-97411-01',
  'PRÉFECTURE - MAYOTTE': 'pre-97611-01',
} as const satisfies Record<string, AdministrationId>

export const isAdministrationId = (id: string | string[] | null | undefined): id is AdministrationId => {
  return typeof id !== 'object' && Object.values(ADMINISTRATION_IDS).includes(id)
}

export const administrationIdValidator = z.enum(IDS)

export type AdministrationId = z.infer<typeof administrationIdValidator>

export interface Administration<T = AdministrationId> {
  id: T
  typeId: AdministrationTypeId
  nom: string
  abreviation: string
  service?: string
  url?: string
  email?: string
  telephone?: string
  adresse1: string
  adresse2?: string
  codePostal: string
  commune: string
  cedex?: string
  regionId?: RegionId
  departementId?: DepartementId
}

export const Administrations: {
  [key in AdministrationId]: Administration<key>
} =
  // ----- ne pas supprimer cette ligne : début
  {
    'aut-97300-01': {
      id: 'aut-97300-01',
      typeId: 'aut',
      nom: 'Gendarmerie Nationale - Guyane',
      abreviation: 'Gendarmerie Nationale - Guyane',
      service: 'Centre de conduite des Opérations',
      url: 'https://www.gendarmerie.interieur.gouv.fr/',
      email: 'cco.comgendgf@gendarmerie.interieur.gouv.fr',
      telephone: '+594 29 28 00',
      adresse1: 'Caserne La Madeleine',
      adresse2: '1296 Route de la Madeleine',
      codePostal: '97300',
      commune: 'Cayenne',
    },
    'aut-mrae-guyane-01': {
      id: 'aut-mrae-guyane-01',
      typeId: 'aut',
      nom: 'Mission régionale autorité environnementale de Guyane',
      abreviation: 'MRAE Guyane',
      url: 'http://www.mrae.developpement-durable.gouv.fr/guyane-r27.html',
      telephone: '+594 (0)5 94 39 80 00',
      adresse1: 'Route du Vieux-Port\n97300 Cayenne',
      adresse2: 'Adresse postale\nRoute du Vieux-Port\nCS 76003\n97306 Cayenne',
      codePostal: '97300',
      commune: 'Cayenne',
      cedex: 'Cedex',
      regionId: '03',
    },
    'dea-guadeloupe-01': {
      id: 'dea-guadeloupe-01',
      typeId: 'dre',
      nom: "Direction de l'environnement de l'aménagement et du logement (DEAL) - Guadeloupe",
      abreviation: 'DEAL - Guadeloupe',
      url: 'http://www.guadeloupe.developpement-durable.gouv.fr',
      email: 'deal-guadeloupe@developpement-durable.gouv.fr',
      telephone: '+33 (0)5 90 99 46 46',
      adresse1: 'Route de St Phy\n97102 Basse-Terre',
      adresse2: 'Adresse postale\r\nDEAL Guadeloupe\nBP 54\n97102 Basse-Terre',
      codePostal: '97102',
      commune: 'Basse-Terre',
      cedex: 'Cedex',
      regionId: '01',
    },
    'dea-guyane-01': {
      id: 'dea-guyane-01',
      typeId: 'dre',
      nom: 'Direction Générale des Territoires et de la Mer de Guyane',
      abreviation: 'DGTM - Guyane',
      url: 'https://www.guyane.gouv.fr',
      telephone: '+594 5 94 39 80 00',
      adresse1: 'Route du Vieux-Port\n97300 Cayenne',
      adresse2: 'Adresse postale\nRoute du Vieux-Port\nCS 76003\n97306 Cayenne',
      codePostal: '97300',
      commune: 'Cayenne',
      cedex: 'Cedex',
      regionId: '03',
    },
    'dea-martinique-01': {
      id: 'dea-martinique-01',
      typeId: 'dre',
      nom: "Direction de l'environnement de l'aménagement et du logement (DEAL) - Martinique",
      abreviation: 'DEAL - Martinique',
      url: 'http://www.martinique.developpement-durable.gouv.fr',
      email: 'deal-martinique-usagers@developpement-durable.gouv.fr',
      telephone: '+33 (0)5 96 59 57 00',
      adresse1: 'B.P. 7212\n97274 Schœlcher',
      codePostal: '97274',
      commune: 'Schœlcher',
      cedex: 'Cedex',
      regionId: '02',
    },
    'dea-mayotte-01': {
      id: 'dea-mayotte-01',
      typeId: 'dre',
      nom: "Direction de l'environnement de l'aménagement et du logement (DEAL) - Mayotte",
      abreviation: 'DEAL - Mayotte',
      url: 'http://www.mayotte.gouv.fr/Services-de-l-Etat/Direction-de-l-Environnement-de-l-Amenagement-et-du-Logement-DEAL',
      email: 'de-mayotte@equipement.gouv.fr',
      telephone: '+262 269 61 12 54',
      adresse1: "Terre plein de M'Tsapéré\n97600 Mamoudzou",
      adresse2: 'Adresse postale\nBP 109\n97600 Mamoudzou',
      codePostal: '97600',
      commune: 'Mamoudzou',
      cedex: 'Cedex',
      regionId: '06',
    },
    'dea-reunion-01': {
      id: 'dea-reunion-01',
      typeId: 'dre',
      nom: "Direction de l'environnement de l'aménagement et du logement (DEAL) - La Réunion",
      abreviation: 'DEAL - La Réunion',
      url: 'http://www.reunion.developpement-durable.gouv.fr',
      email: 'deal-reunion@developpement-durable.gouv.fr',
      telephone: '+33 (0)2 62 40 26 26',
      adresse1: '2 rue Juliette Dodu\nCS 41009\r\n97743 Saint-Denis',
      codePostal: '97743',
      commune: 'Saint-Denis',
      cedex: 'Cedex',
      regionId: '04',
    },
    'dre-aura-01': {
      id: 'dre-aura-01',
      typeId: 'dre',
      nom: "Direction régionale de l'environnement, de l'aménagement et du logement (DREAL) - Auvergne-Rhône-Alpes - Siège de Lyon",
      abreviation: 'DREAL - Auvergne-Rhône-Alpes - Siège de Lyon',
      url: 'http://www.auvergne-rhone-alpes.developpement-durable.gouv.fr',
      email: 'dir.dreal-auvergne-rhone-alpes@developpement-durable.gouv.fr',
      telephone: '+33 (0)4 26 28 60 00',
      adresse1: '5 place Jules-Ferry\n69006 Lyon',
      adresse2: 'Adresse postale\nDreal Rhône-Alpes\n69453 Lyon Cedex 6',
      codePostal: '69006',
      commune: 'Lyon',
      cedex: 'Cedex',
      regionId: '84',
    },
    'dre-bfc-01': {
      id: 'dre-bfc-01',
      typeId: 'dre',
      nom: "Direction régionale de l'environnement, de l'aménagement et du logement (DREAL) - Bourgogne-Franche-Comté - Siège de Besançon",
      abreviation: 'DREAL - Bourgogne-Franche-Comté - Siège de Besançon',
      url: 'http://www.bourgogne-franche-comte.developpement-durable.gouv.fr',
      telephone: '+33 (0)3 39 59 62 00',
      adresse1: '5 voie Gisèle Halimi',
      adresse2: 'BP 31269 ',
      codePostal: '25005',
      commune: 'Besançon',
      cedex: 'Cedex',
      regionId: '27',
    },
    'dre-bretagne-01': {
      id: 'dre-bretagne-01',
      typeId: 'dre',
      nom: "Direction régionale de l'environnement, de l'aménagement et du logement (DREAL) - Bretagne",
      abreviation: 'DREAL - Bretagne',
      url: 'http://www.bretagne.developpement-durable.gouv.fr',
      email: 'dreal-bretagne@developpement-durable.gouv.fr',
      telephone: '+33 (0)2 99 33 45 55',
      adresse1: '10 rue Maurice-Fabre\nCS 96515\n35065 Rennes',
      codePostal: '35065',
      commune: 'Rennes',
      cedex: 'Cedex',
      regionId: '53',
    },
    'dre-centre-val-de-loire-01': {
      id: 'dre-centre-val-de-loire-01',
      typeId: 'dre',
      nom: "Direction régionale de l'environnement, de l'aménagement et du logement (DREAL) - Centre-Val-de-Loire",
      abreviation: 'DREAL - Centre-Val-de-Loire',
      url: 'http://www.centre.developpement-durable.gouv.fr',
      email: 'dreal-centre@developpement-durable.gouv.fr',
      telephone: '+33 (0)2 36 17 41 41',
      adresse1: '5 avenue Buffon\nCS 96407\n45064 Orléans',
      codePostal: '45064',
      commune: 'Orléans',
      cedex: 'Cedex',
      regionId: '24',
    },
    'dre-corse-01': {
      id: 'dre-corse-01',
      typeId: 'dre',
      nom: "Direction régionale de l'environnement, de l'aménagement et du logement (DREAL) - Corse",
      abreviation: 'DREAL - Corse',
      url: 'http://www.corse.developpement-durable.gouv.fr',
      email: 'dreal-corse@developpement-durable.gouv.fr',
      telephone: '+33 (0)4 95 51 79 70',
      adresse1: '19 cours Napoléon\nBâtiment D\nCS 10006\n20704 Ajaccio',
      codePostal: '20704',
      commune: 'Ajaccio',
      cedex: 'Cedex',
      regionId: '94',
    },
    'dre-grand-est-01': {
      id: 'dre-grand-est-01',
      typeId: 'dre',
      nom: "Direction régionale de l'environnement, de l'aménagement et du logement (DREAL) - Grand Est - Siège de Metz",
      abreviation: 'DREAL - Grand Est - Siège de Metz',
      url: 'http://www.grand-est.developpement-durable.gouv.fr',
      telephone: '+33 (0)3 87 62 81 00',
      adresse1: 'Greenpark\n2 rue Augustin-Fresnel\nBP 95038\n57071 Metz',
      codePostal: '57071',
      commune: 'Metz',
      cedex: 'Cedex',
      regionId: '44',
    },
    'dre-hauts-de-france-01': {
      id: 'dre-hauts-de-france-01',
      typeId: 'dre',
      nom: "Direction régionale de l'environnement, de l'aménagement et du logement (DREAL) - Hauts-de-France - Siège de Lille",
      abreviation: 'DREAL - Hauts-de-France - Siège de Lille',
      url: 'http://www.hauts-de-france.developpement-durable.gouv.fr',
      telephone: '+33 (0)3 20 13 48 48',
      adresse1: '44 rue de Tournai\nCS 40259\n59019 Lille',
      codePostal: '59019',
      commune: 'Lille',
      cedex: 'Cedex',
      regionId: '32',
    },
    'dre-ile-de-france-01': {
      id: 'dre-ile-de-france-01',
      typeId: 'dre',
      nom: "Direction régionale et interdépartementale de l'environnement et de l'énergie (DRIEE) - Île-de-France",
      abreviation: 'DRIEE - Île-de-France',
      url: 'http://www.driee.ile-de-france.developpement-durable.gouv.fr',
      email: 'driee-if@developpement-durable.gouv.fr',
      telephone: '+33 (0)1 87 36 45 00',
      adresse1: '12 cours Louis Lumière, CS 70027',
      codePostal: '94307',
      commune: 'Vincennes',
      cedex: 'cedex',
      regionId: '11',
    },
    'dre-normandie-01': {
      id: 'dre-normandie-01',
      typeId: 'dre',
      nom: "Direction régionale de l'environnement, de l'aménagement et du logement (DREAL) - Normandie - Siège de Rouen",
      abreviation: 'DREAL - Normandie - Siège de Rouen',
      url: 'http://www.normandie.developpement-durable.gouv.fr',
      email: 'dreal-hnormandie@developpement-durable.gouv.fr',
      telephone: '+33 (0)2 35 58 53 27',
      adresse1: 'BP 86002\nCité administrative\n2 rue Saint-Sever\n76032 Rouen',
      codePostal: '76032',
      commune: 'Rouen',
      cedex: 'Cedex',
      regionId: '28',
    },
    'dre-nouvelle-aquitaine-01': {
      id: 'dre-nouvelle-aquitaine-01',
      typeId: 'dre',
      nom: "Direction régionale de l'environnement, de l'aménagement et du logement (DREAL) - Nouvelle-Aquitaine - Siège de Poitiers",
      abreviation: 'DREAL - Nouvelle-Aquitaine - Siège de Poitiers',
      url: 'http://www.nouvelle-aquitaine.developpement-durable.gouv.fr',
      telephone: '+33 (0)5 49 55 63 63',
      adresse1: '15 rue Arthur-Ranc\nBP 60539\n86020 Poitiers',
      codePostal: '86020',
      commune: 'Poitiers',
      cedex: 'Cedex',
      regionId: '75',
    },
    'dre-occitanie-01': {
      id: 'dre-occitanie-01',
      typeId: 'dre',
      nom: "Direction régionale de l'environnement, de l'aménagement et du logement (DREAL) - Occitanie - Siège de Toulouse",
      abreviation: 'DREAL - Occitanie - Siège de Toulouse',
      url: 'http://www.occitanie.developpement-durable.gouv.fr',
      telephone: '+33 (0)5 61 58 50 00',
      adresse1: '1 rue de la Cité-Administrative\nBâtiment G\nCS 80002\n31074 Toulouse',
      codePostal: '31074',
      commune: 'Toulouse',
      cedex: 'Cedex',
      regionId: '76',
    },
    'dre-paca-01': {
      id: 'dre-paca-01',
      typeId: 'dre',
      nom: "Direction régionale de l'environnement, de l'aménagement et du logement (DREAL) - Provence-Alpes-Côte-d'Azur",
      abreviation: "DREAL - Provence-Alpes-Côte-d'Azur",
      url: 'http://www.paca.developpement-durable.gouv.fr',
      telephone: '+33 (0)4 91 28 40 40',
      adresse1: '16 rue Antoine-Zattara\nCS 70248\n13331 Marseille',
      codePostal: '13331',
      commune: 'Marseille',
      cedex: 'Cedex',
      regionId: '93',
    },
    'dre-pays-de-la-loire-01': {
      id: 'dre-pays-de-la-loire-01',
      typeId: 'dre',
      nom: "Direction régionale de l'environnement, de l'aménagement et du logement (DREAL) - Pays de la Loire",
      abreviation: 'DREAL - Pays de la Loire',
      url: 'http://www.pays-de-la-loire.developpement-durable.gouv.fr',
      email: 'dreal-pays-de-la-loire@developpement-durable.gouv.fr',
      telephone: '+33 (0)2 72 74 73 00',
      adresse1: '5 rue Françoise-Giroud\nCS 16326\n44263 Nantes',
      codePostal: '44263',
      commune: 'Nantes',
      cedex: 'Cedex',
      regionId: '52',
    },
    'min-dajb-01': {
      id: 'min-dajb-01',
      typeId: 'min',
      nom: "Ministère de l'Economie, des Finances et de la Relance",
      abreviation: "DAJ - Ministère de l'Economie, des Finances et de la Relance",
      service: "Bureau 4C. Droit de l'industrie, de l'énergie et des réseaux de communication",
      url: 'http://www.economie.gouv.fr',
      telephone: '01 44 97 03 32',
      adresse1: '6 rue Louise-Weiss',
      adresse2: 'Télédoc 342',
      codePostal: '75703',
      commune: 'Paris',
      cedex: 'Cedex 13',
    },
    'min-mctrct-dgcl-01': {
      id: 'min-mctrct-dgcl-01',
      typeId: 'min',
      nom: 'Ministère de la Cohésion des Territoires et des Relations avec les Collectivités Territoriales',
      abreviation: 'DGCL/SDFLAE/FL1',
      service: 'Bureau de la fiscalité locale',
      url: 'https://www.cohesion-territoires.gouv.fr/direction-generale-des-collectivites-locales',
      telephone: '01 49 27 31 59',
      adresse1: '2 place des Saussaies',
      codePostal: '75800',
      commune: 'Paris',
    },
    'min-mtes-dgaln-01': {
      id: 'min-mtes-dgaln-01',
      typeId: 'min',
      nom: "Ministère de l'Economie, des Finances et de la Relance & Ministère de la Transition écologique",
      abreviation: 'DGALN/DEB/EARM2',
      service: "Bureau de la politique des ressources minérales non énergétiques - Direction générale de l'aménagement, du logement et de la nature (DGALN)",
      url: 'http://www.mineralinfo.fr',
      email: 'earm2.deb.dgaln@developpement-durable.gouv.fr',
      telephone: '+33 (0)1 40 81 21 22',
      adresse1: 'Tour Séquoia',
      adresse2: '1 place Carpeaux',
      codePostal: '92055',
      commune: 'Paris-La-Défense',
      cedex: 'Cedex',
    },
    'min-mtes-dgec-01': {
      id: 'min-mtes-dgec-01',
      typeId: 'min',
      nom: "Ministère de l'Economie, des Finances et de la Relance & Ministère de la Transition écologique",
      abreviation: 'DGEC/DE/SD2/2A',
      service: "Bureau Ressources énergétiques du sous-sol (2A) - Direction générale de l'énergie et du climat (DGEC)",
      url: 'http://www.minergies.fr',
      email: '2a.sd2.de.dgec@developpement-durable.gouv.fr',
      telephone: '+33 (0)1 40 81 95 63',
      adresse1: 'Tour Séquoia\n1 place Carpeaux\n92800 Puteaux',
      adresse2: 'Adresse postale\r\n92055 Paris-La-Défense Cedex',
      codePostal: '92800',
      commune: 'Puteaux',
      cedex: 'Cedex',
    },
    'min-mtes-dgpr-01': {
      id: 'min-mtes-dgpr-01',
      typeId: 'min',
      nom: "Ministère de l'Economie, des Finances et de la Relance & Ministère de la Transition écologique",
      abreviation: 'DGPR/SRT/SDRCP/BSSS',
      service: 'Bureau du sol et du sous-sol - Direction générale de la prévention des risques (DGPR)',
      email: 'bsss.sdrcp.srt.dgpr@developpement-durable.gouv.fr',
      telephone: '+33 (0)1 40 81 21 22',
      adresse1: 'Tour Séquoia\n1 place Carpeaux\n92800 Puteaux',
      adresse2: 'Adresse postale\n92055 Paris-La-Défense Cedex',
      codePostal: '92800',
      commune: 'Puteaux',
      cedex: 'Cedex',
    },
    'ope-brgm-01': {
      id: 'ope-brgm-01',
      typeId: 'ope',
      nom: 'Bureau de recherches géologiques et minières',
      abreviation: 'BRGM',
      url: 'https://www.brgm.fr/',
      telephone: '02 38 64 34 34',
      adresse1: '3 avenue Claude-Guillemin',
      adresse2: 'BP 36009',
      codePostal: '45060',
      commune: 'Orléans',
      cedex: 'Cedex 2',
    },
    'ope-cacem-01': {
      id: 'ope-cacem-01',
      typeId: 'ope',
      nom: "Centre d'appui au contrôle de l'environnement marin",
      abreviation: 'Cacem',
      service: 'DIRM NAMO/DOSM/CROSS Etel/CACEM',
      url: 'http://extranet.legicem.metier.developpement-durable.gouv.fr',
      email: 'cacem@developpement-durable.gouv.fr',
      adresse1: 'Centre régional opérationnel de surveillance et de sauvetage Atlantique',
      adresse2: '40 avenue L. Bougo',
      codePostal: '56410',
      commune: 'Etel',
    },
    'ope-onf-973-01': {
      id: 'ope-onf-973-01',
      typeId: 'ope',
      nom: 'Office national des forêts',
      abreviation: 'Office national des forêts',
      service: 'Direction territoriale Guyane',
      url: 'http://www1.onf.fr/guyane/@@index.html',
      email: 'pole.minier@onf.fr',
      telephone: '+594 (0)5 94 25 53 78',
      adresse1: '541 route de Montabo',
      adresse2: 'Adresse postale\r\n541 route de Montabo\nCS87002',
      codePostal: '97300',
      commune: 'Cayenne',
      regionId: '03',
    },
    'ope-ptmg-973-01': {
      id: 'ope-ptmg-973-01',
      typeId: 'ope',
      nom: 'Pôle Technique Minier de Guyane',
      abreviation: 'Pôle Technique Minier de Guyane',
      telephone: '+594 (0)5 94 30 06 00',
      adresse1: 'Hôtel de la Collectivité Territoriale de Guyane, Carrefour de Suzini',
      adresse2: '4179 route de Montabo',
      codePostal: '97307',
      commune: 'Cayenne',
    },
    'pre-01053-01': {
      id: 'pre-01053-01',
      typeId: 'pre',
      nom: 'Préfecture - Ain',
      abreviation: 'Préfecture - Ain',
      adresse1: "45, avenue d'Alsace-Lorraine, CS 80400, Quartier Bourg centre",
      codePostal: '01012',
      commune: 'Bourg-en-Bresse Cedex',
      telephone: '04 74 32 30 00',
      departementId: '01',
      email: 'prefecture@ain.gouv.fr',
      url: 'http://www.ain.gouv.fr',
    },
    'pre-02408-01': {
      id: 'pre-02408-01',
      typeId: 'pre',
      nom: 'Préfecture - Aisne',
      abreviation: 'Préfecture - Aisne',
      adresse1: '2, rue Paul-Doumer',
      codePostal: '02010',
      commune: 'Laon Cedex',
      telephone: '03 23 21 82 82',
      departementId: '02',
      email: 'prefecture@aisne.gouv.fr',
      url: 'http://www.aisne.gouv.fr',
    },
    'pre-03190-01': {
      id: 'pre-03190-01',
      typeId: 'pre',
      nom: 'Préfecture - Allier',
      abreviation: 'Préfecture - Allier',
      adresse1: "2, rue Michel-de-L'Hospital, CS 31649",
      codePostal: '03016',
      commune: 'Moulins Cedex',
      telephone: '04 70 48 30 00',
      departementId: '03',
      email: 'prefecture@allier.gouv.fr',
      url: 'http://www.allier.gouv.fr',
    },
    'pre-04070-01': {
      id: 'pre-04070-01',
      typeId: 'pre',
      nom: 'Préfecture - Alpes-de-Haute-Provence',
      abreviation: 'Préfecture - Alpes-de-Haute-Provence',
      adresse1: '8, rue Docteur-Romieu',
      codePostal: '04016',
      commune: 'Digne-les-Bains Cedex',
      telephone: '04 92 36 72 00',
      departementId: '04',
      url: 'http://www.alpes-de-haute-provence.gouv.fr',
    },
    'pre-05061-01': {
      id: 'pre-05061-01',
      typeId: 'pre',
      nom: 'Préfecture - Hautes-Alpes',
      abreviation: 'Préfecture - Hautes-Alpes',
      adresse1: '28, rue Saint-Arey, CS 66002',
      codePostal: '05011',
      commune: 'Gap Cedex',
      telephone: '04 92 40 48 00',
      departementId: '05',
      email: 'prefecture@hautes-alpes.gouv.fr',
      url: 'http://www.hautes-alpes.gouv.fr',
    },
    'pre-06088-01': {
      id: 'pre-06088-01',
      typeId: 'pre',
      nom: 'Préfecture - Alpes-Maritimes',
      abreviation: 'Préfecture - Alpes-Maritimes',
      adresse1: 'CADAM, 147, route de Grenoble',
      codePostal: '06000',
      commune: 'Nice',
      telephone: '04 93 72 20 00',
      departementId: '06',
      adresse2: "Services de l'État dans les Alpes-Maritimes, Préfecture, CADAM, 147, route de Grenoble",
      url: 'http://www.alpes-maritimes.gouv.fr',
    },
    'pre-07186-01': {
      id: 'pre-07186-01',
      typeId: 'pre',
      nom: 'Préfecture - Ardèche',
      abreviation: 'Préfecture - Ardèche',
      adresse1: '4, boulevard de Vernon',
      codePostal: '07000',
      commune: 'Privas',
      telephone: '04 75 66 50 00',
      departementId: '07',
      email: 'pref-courrier@ardeche.gouv.fr',
      url: 'http://www.ardeche.gouv.fr',
    },
    'pre-08105-01': {
      id: 'pre-08105-01',
      typeId: 'pre',
      nom: 'Préfecture - Ardennes',
      abreviation: 'Préfecture - Ardennes',
      adresse1: '1, place de la Préfecture, BP 60002',
      codePostal: '08005',
      commune: 'Charleville-Mézières Cedex',
      telephone: '03 24 59 66 00',
      departementId: '08',
      email: 'prefecture@ardennes.gouv.fr',
      url: 'http://www.ardennes.gouv.fr',
    },
    'pre-09122-01': {
      id: 'pre-09122-01',
      typeId: 'pre',
      nom: 'Préfecture - Ariège',
      abreviation: 'Préfecture - Ariège',
      adresse1: '2, rue de la Préfecture - Préfet Claude Erignac, BP 4087',
      codePostal: '09007',
      commune: 'Foix Cedex',
      telephone: '05 61 02 10 00',
      departementId: '09',
      email: 'prefecture@ariege.gouv.fr',
      url: 'http://www.ariege.gouv.fr',
    },
    'pre-10387-01': {
      id: 'pre-10387-01',
      typeId: 'pre',
      nom: 'Préfecture - Aube',
      abreviation: 'Préfecture - Aube',
      adresse1: 'CS 20372, 2, rue Pierre-Labonde',
      codePostal: '10025',
      commune: 'Troyes Cedex',
      telephone: '03 25 42 35 00',
      departementId: '10',
      email: 'prefecture@aube.gouv.fr',
      url: 'http://www.aube.gouv.fr',
    },
    'pre-11069-01': {
      id: 'pre-11069-01',
      typeId: 'pre',
      nom: 'Préfecture - Aude',
      abreviation: 'Préfecture - Aude',
      adresse1: 'CS 20001, 52, rue Jean-Bringer',
      codePostal: '11836',
      commune: 'Carcassonne Cedex 9',
      telephone: '04 68 10 27 00',
      departementId: '11',
      email: 'prefecture@aude.gouv.fr',
      url: 'http://www.aude.gouv.fr',
    },
    'pre-12202-01': {
      id: 'pre-12202-01',
      typeId: 'pre',
      nom: 'Préfecture - Aveyron',
      abreviation: 'Préfecture - Aveyron',
      adresse1: 'Centre administratif Foch, Place Foch',
      codePostal: '12000',
      commune: 'Rodez',
      telephone: '05 65 75 71 71',
      departementId: '12',
      email: 'prefecture@aveyron.fr',
      url: 'http://www.aveyron.gouv.fr',
    },
    'pre-13203-01': {
      id: 'pre-13203-01',
      typeId: 'pre',
      nom: 'Préfecture - Bouches-du-Rhône',
      abreviation: 'Préfecture - Bouches-du-Rhône',
      adresse1: 'Boulevard Paul-Peytral',
      codePostal: '13282',
      commune: 'Marseille Cedex 20',
      telephone: '04 84 35 40 00',
      departementId: '13',
      email: 'contact@bouches-du-rhone.pref.gouv.fr',
      url: 'http://www.bouches-du-rhone.gouv.fr',
    },
    'pre-14118-01': {
      id: 'pre-14118-01',
      typeId: 'pre',
      nom: 'Préfecture - Calvados',
      abreviation: 'Préfecture - Calvados',
      adresse1: 'Rue Daniel-Huet',
      codePostal: '14038',
      commune: 'Caen Cedex 9',
      telephone: '02 31 30 64 00',
      departementId: '14',
      email: 'prefecture@calvados.gouv.fr',
      url: 'http://www.calvados.gouv.fr',
    },
    'pre-15014-01': {
      id: 'pre-15014-01',
      typeId: 'pre',
      nom: 'Préfecture - Cantal',
      abreviation: 'Préfecture - Cantal',
      adresse1: 'Cours Monthyon, BP 529',
      codePostal: '15005',
      commune: 'Aurillac Cedex',
      telephone: '04 71 46 23 00',
      departementId: '15',
      email: 'courrier@cantal.pref.gouv.fr',
      url: 'http://www.cantal.gouv.fr',
    },
    'pre-16015-01': {
      id: 'pre-16015-01',
      typeId: 'pre',
      nom: 'Préfecture - Charente',
      abreviation: 'Préfecture - Charente',
      adresse1: '7-9, rue de la Préfecture, CS 92301',
      codePostal: '16023',
      commune: 'Angoulême Cedex',
      telephone: '05 45 97 61 00',
      departementId: '16',
      email: 'communication@charente.gouv.fr',
      url: 'http://www.charente.gouv.fr',
    },
    'pre-17300-01': {
      id: 'pre-17300-01',
      typeId: 'pre',
      nom: 'Préfecture - Charente-Maritime',
      abreviation: 'Préfecture - Charente-Maritime',
      adresse1: 'CS 70000, 38, rue Réaumur',
      codePostal: '17017',
      commune: 'La Rochelle Cedex 1',
      telephone: '05 46 27 43 00',
      departementId: '17',
      email: 'prefecture@charente-maritime.gouv.fr',
      url: 'http://www.charente-maritime.gouv.fr',
    },
    'pre-18033-01': {
      id: 'pre-18033-01',
      typeId: 'pre',
      nom: 'Préfecture - Cher',
      abreviation: 'Préfecture - Cher',
      adresse1: 'Place Marcel-Plaisant, CS 60022',
      codePostal: '18020',
      commune: 'Bourges Cedex',
      telephone: '02 48 67 18 18',
      departementId: '18',
      email: 'prefecture@cher.gouv.fr',
      url: 'http://www.cher.gouv.fr',
    },
    'pre-19272-01': {
      id: 'pre-19272-01',
      typeId: 'pre',
      nom: 'Préfecture - Corrèze',
      abreviation: 'Préfecture - Corrèze',
      adresse1: '1, rue Souham, BP 250',
      codePostal: '19012',
      commune: 'Tulle Cedex',
      telephone: '05 55 20 55 20',
      departementId: '19',
      email: 'prefecture@correze.gouv.fr',
      url: 'http://www.correze.gouv.fr',
    },
    'pre-21231-01': {
      id: 'pre-21231-01',
      typeId: 'pre',
      nom: "Préfecture - Côte-d'Or",
      abreviation: "Préfecture - Côte-d'Or",
      adresse1: '53, rue de la Préfecture',
      codePostal: '21041',
      commune: 'Dijon Cedex',
      telephone: '03 80 44 64 00',
      departementId: '21',
      url: 'http://www.cote-dor.gouv.fr',
    },
    'pre-22278-01': {
      id: 'pre-22278-01',
      typeId: 'pre',
      nom: "Préfecture - Côtes-d'Armor",
      abreviation: "Préfecture - Côtes-d'Armor",
      adresse1: '1, place du Général-de-Gaulle, BP 2370',
      codePostal: '22023',
      commune: 'Saint-Brieuc Cedex 1',
      telephone: '02 96 62 44 22',
      departementId: '22',
      email: 'prefecture@cotes-darmor.gouv.fr',
      url: 'http://www.cotes-darmor.gouv.fr',
    },
    'pre-23096-01': {
      id: 'pre-23096-01',
      typeId: 'pre',
      nom: 'Préfecture - Creuse',
      abreviation: 'Préfecture - Creuse',
      adresse1: 'BP 79, Place Louis-Lacrocq',
      codePostal: '23011',
      commune: 'Guéret Cedex',
      telephone: '05 55 51 59 00',
      departementId: '23',
      url: 'http://www.creuse.gouv.fr',
    },
    'pre-24322-01': {
      id: 'pre-24322-01',
      typeId: 'pre',
      nom: 'Préfecture - Dordogne',
      abreviation: 'Préfecture - Dordogne',
      adresse1: '2, rue Paul-Louis-Courier',
      codePostal: '24000',
      commune: 'Périgueux',
      telephone: '05 53 02 24 24',
      departementId: '24',
      adresse2: "Services de l'État - Préfecture de la Dordogne, Cité adminsitrative",
      email: 'prefecture@dordogne.gouv.fr',
      url: 'http://www.dordogne.gouv.fr',
    },
    'pre-25056-01': {
      id: 'pre-25056-01',
      typeId: 'pre',
      nom: 'Préfecture - Doubs',
      abreviation: 'Préfecture - Doubs',
      adresse1: '8 bis, rue Charles-Nodier',
      codePostal: '25035',
      commune: 'Besançon Cedex',
      telephone: '03 81 25 10 00',
      departementId: '25',
      email: 'pref-courrier@doubs.gouv.fr',
      url: 'http://www.doubs.gouv.fr',
    },
    'pre-26362-01': {
      id: 'pre-26362-01',
      typeId: 'pre',
      nom: 'Préfecture - Drôme',
      abreviation: 'Préfecture - Drôme',
      adresse1: '3, boulevard Vauban',
      codePostal: '26030',
      commune: 'Valence Cedex 9',
      telephone: '04 75 79 28 00',
      departementId: '26',
      email: 'prefecture@drome.gouv.fr',
      url: 'http://www.drome.gouv.fr',
    },
    'pre-27229-01': {
      id: 'pre-27229-01',
      typeId: 'pre',
      nom: 'Préfecture - Eure',
      abreviation: 'Préfecture - Eure',
      adresse1: 'Boulevard Georges-Chauvin',
      codePostal: '27022',
      commune: 'Evreux Cedex',
      telephone: '02 32 78 27 27',
      departementId: '27',
      email: 'courrier-web@eure.gouv.fr',
      url: 'http://www.seine-maritime.gouv.fr/Outils/Horaires-et-coordonnees/Prefecture-de-l-Eure',
    },
    'pre-28085-01': {
      id: 'pre-28085-01',
      typeId: 'pre',
      nom: 'Préfecture - Eure-et-Loir',
      abreviation: 'Préfecture - Eure-et-Loir',
      adresse1: 'Place de la République, CS 80537',
      codePostal: '28019',
      commune: 'Chartres Cedex',
      telephone: '02 37 27 72 00',
      departementId: '28',
      url: 'http://www.eure-et-loir.gouv.fr',
    },
    'pre-29232-01': {
      id: 'pre-29232-01',
      typeId: 'pre',
      nom: 'Préfecture - Finistère',
      abreviation: 'Préfecture - Finistère',
      adresse1: '42, boulevard Dupleix',
      codePostal: '29320',
      commune: 'Quimper Cedex',
      telephone: '02 98 76 29 29',
      departementId: '29',
      email: 'prefecture@finistere.gouv.fr',
      url: 'http://www.finistere.gouv.fr',
    },
    'pre-2A004-01': {
      id: 'pre-2A004-01',
      typeId: 'pre',
      nom: 'Préfecture - Corse-du-Sud',
      abreviation: 'Préfecture - Corse-du-Sud',
      adresse1: 'Palais Lantivy, Cours Napoléon',
      codePostal: '20188',
      commune: 'Ajaccio Cedex 9',
      telephone: '04 95 11 12 13',
      departementId: '2A',
      email: 'prefecture@corse-du-sud.gouv.fr',
      url: 'http://www.corse-du-sud.gouv.fr',
    },
    'pre-2B033-01': {
      id: 'pre-2B033-01',
      typeId: 'pre',
      nom: 'Préfecture - Haute-Corse',
      abreviation: 'Préfecture - Haute-Corse',
      adresse1: 'CS 60007, Rond-Point du Maréchal-Leclerc-de-Hautecloque',
      codePostal: '20401',
      commune: 'Bastia Cedex 09',
      telephone: '04 95 34 50 00',
      departementId: '2B',
      email: 'prefecture.haute-corse@haute-corse.pref.gouv.fr',
      url: 'http://www.haute-corse.gouv.fr',
    },
    'pre-30189-01': {
      id: 'pre-30189-01',
      typeId: 'pre',
      nom: 'Préfecture - Gard',
      abreviation: 'Préfecture - Gard',
      adresse1: '2, rue Guillemette, 19, avenue Feuchères (étrangers)',
      codePostal: '30000',
      commune: 'Nîmes',
      telephone: '0820 09 11 72',
      departementId: '30',
      adresse2: '10, avenue Feuchères',
      email: 'prefecture@gard.gouv.fr',
      url: 'http://www.gard.gouv.fr',
    },
    'pre-31555-01': {
      id: 'pre-31555-01',
      typeId: 'pre',
      nom: 'Préfecture - Haute-Garonne',
      abreviation: 'Préfecture - Haute-Garonne',
      adresse1: '1 place Saint-Étienne',
      codePostal: '31000',
      commune: 'Toulouse',
      telephone: '05 34 45 34 45',
      departementId: '31',
      adresse2: 'Place Saint-Étienne',
      url: 'http://www.haute-garonne.gouv.fr',
    },
    'pre-32013-01': {
      id: 'pre-32013-01',
      typeId: 'pre',
      nom: 'Préfecture - Gers',
      abreviation: 'Préfecture - Gers',
      adresse1: '3, place du Préfet Claude-Erignac',
      codePostal: '32007',
      commune: 'Auch Cedex',
      telephone: '05 62 61 44 00',
      departementId: '32',
      email: 'prefecture@gers.gouv.fr',
      url: 'http://www.gers.gouv.fr',
    },
    'pre-33063-01': {
      id: 'pre-33063-01',
      typeId: 'pre',
      nom: 'Préfecture - Gironde',
      abreviation: 'Préfecture - Gironde',
      adresse1: 'Rue Claude-Bonnier',
      codePostal: '33000',
      commune: 'Bordeaux',
      telephone: '05 56 90 60 60',
      departementId: '33',
      adresse2: '2, esplanade Charles-de-Gaulle, CS 41397',
      url: 'http://www.gironde.gouv.fr',
    },
    'pre-34172-01': {
      id: 'pre-34172-01',
      typeId: 'pre',
      nom: 'Préfecture - Hérault',
      abreviation: 'Préfecture - Hérault',
      adresse1: '34, place des Martyrs-de-la-Résistance',
      codePostal: '34062',
      commune: 'Montpellier Cedex 2',
      telephone: '04 67 61 61 61',
      departementId: '34',
      url: 'http://www.herault.gouv.fr',
    },
    'pre-35238-01': {
      id: 'pre-35238-01',
      typeId: 'pre',
      nom: 'Préfecture - Ille-et-Vilaine',
      abreviation: 'Préfecture - Ille-et-Vilaine',
      adresse1: '3, avenue de la Préfecture',
      codePostal: '35026',
      commune: 'Rennes Cedex 9',
      telephone: '02 99 02 10 35',
      departementId: '35',
      email: 'prefecture@ille-et-vilaine.gouv.fr',
      url: 'https://www.ille-et-vilaine.gouv.fr',
    },
    'pre-36044-01': {
      id: 'pre-36044-01',
      typeId: 'pre',
      nom: 'Préfecture - Indre',
      abreviation: 'Préfecture - Indre',
      adresse1: 'Rue des Jeux marins',
      codePostal: '36000',
      commune: 'Châteauroux',
      telephone: '02 54 29 50 00',
      departementId: '36',
      adresse2: 'Place de la Victoire et des Alliés, CS 80583',
      url: 'http://www.indre.gouv.fr',
    },
    'pre-37261-01': {
      id: 'pre-37261-01',
      typeId: 'pre',
      nom: 'Préfecture - Indre-et-Loire',
      abreviation: 'Préfecture - Indre-et-Loire',
      adresse1: '15, rue Bernard-Palissy',
      codePostal: '37925',
      commune: 'Tours Cedex 9',
      telephone: '+ 33 2 47 64 37 37',
      departementId: '37',
      email: 'prefecture@indre-et-loire.gouv.fr',
      url: 'http://www.indre-et-loire.gouv.fr',
    },
    'pre-38185-01': {
      id: 'pre-38185-01',
      typeId: 'pre',
      nom: 'Préfecture - Isère',
      abreviation: 'Préfecture - Isère',
      adresse1: 'CS 71046, Place de Verdun',
      codePostal: '38021',
      commune: 'Grenoble Cedex 1',
      telephone: '04 76 60 34 00',
      departementId: '38',
      url: 'http://www.isere.gouv.fr',
    },
    'pre-39300-01': {
      id: 'pre-39300-01',
      typeId: 'pre',
      nom: 'Préfecture - Jura',
      abreviation: 'Préfecture - Jura',
      adresse1: '8, rue de la Préfecture',
      codePostal: '39030',
      commune: 'Lons-le-Saunier Cedex',
      telephone: '03 84 86 84 00',
      departementId: '39',
      email: 'prefecture@jura.gouv.fr',
      url: 'http://www.jura.gouv.fr',
    },
    'pre-40192-01': {
      id: 'pre-40192-01',
      typeId: 'pre',
      nom: 'Préfecture - Landes',
      abreviation: 'Préfecture - Landes',
      adresse1: '24-26, rue Victor-Hugo',
      codePostal: '40021',
      commune: 'Mont-de-Marsan Cedex',
      telephone: '05 58 06 58 06',
      departementId: '40',
      email: 'prefecture@landes.gouv.fr',
      url: 'http://www.landes.gouv.fr',
    },
    'pre-41018-01': {
      id: 'pre-41018-01',
      typeId: 'pre',
      nom: 'Préfecture - Loir-et-Cher',
      abreviation: 'Préfecture - Loir-et-Cher',
      adresse1: 'BP 40299, 1, place de la République',
      codePostal: '41006',
      commune: 'Blois Cedex',
      telephone: '0810 02 41 41',
      departementId: '41',
      url: 'http://www.loir-et-cher.gouv.fr',
    },
    'pre-42218-01': {
      id: 'pre-42218-01',
      typeId: 'pre',
      nom: 'Préfecture - Loire',
      abreviation: 'Préfecture - Loire',
      adresse1: '2, rue Charles-de-Gaulle',
      codePostal: '42022',
      commune: 'Saint-Etienne Cedex 1',
      telephone: '04 77 48 48 48',
      departementId: '42',
      email: 'pref-loire@loire.pref.gouv.fr',
      url: 'http://www.loire.gouv.fr',
    },
    'pre-43157-01': {
      id: 'pre-43157-01',
      typeId: 'pre',
      nom: 'Préfecture - Haute-Loire',
      abreviation: 'Préfecture - Haute-Loire',
      adresse1: '6, avenue du Général-de-Gaulle, CS 40321',
      codePostal: '43009',
      commune: 'Le Puy-en-Velay Cedex',
      telephone: '04 71 09 43 43',
      departementId: '43',
      email: 'prefecture@haute-loire.gouv.fr',
      url: 'http://www.haute-loire.gouv.fr',
    },
    'pre-44109-01': {
      id: 'pre-44109-01',
      typeId: 'pre',
      nom: 'Préfecture - Loire-Atlantique',
      abreviation: 'Préfecture - Loire-Atlantique',
      adresse1: '6, quai Ceineray, BP 33515',
      codePostal: '44035',
      commune: 'Nantes Cedex 1',
      telephone: '02 40 41 20 20',
      departementId: '44',
      url: 'http://www.loire-atlantique.gouv.fr',
    },
    'pre-45234-01': {
      id: 'pre-45234-01',
      typeId: 'pre',
      nom: 'Préfecture - Loiret',
      abreviation: 'Préfecture - Loiret',
      adresse1: '181, rue de Bourgogne',
      codePostal: '45042',
      commune: 'Orléans Cedex 1',
      telephone: '0821 80 30 45',
      departementId: '45',
      url: 'http://www.loiret.gouv.fr',
    },
    'pre-46042-01': {
      id: 'pre-46042-01',
      typeId: 'pre',
      nom: 'Préfecture - Lot',
      abreviation: 'Préfecture - Lot',
      adresse1: 'Cité Bessières',
      codePostal: '46009',
      commune: 'Cahors Cedex 9',
      telephone: '05 65 23 10 00',
      departementId: '46',
      email: 'prefecture@lot.gouv.fr',
      url: 'http://www.lot.gouv.fr',
    },
    'pre-47001-01': {
      id: 'pre-47001-01',
      typeId: 'pre',
      nom: 'Préfecture - Lot-et-Garonne',
      abreviation: 'Préfecture - Lot-et-Garonne',
      adresse1: 'Place de Verdun',
      codePostal: '47920',
      commune: 'Agen Cedex 9',
      telephone: '05 53 77 60 47',
      departementId: '47',
      url: 'http://www.lot-et-garonne.gouv.fr',
    },
    'pre-48095-01': {
      id: 'pre-48095-01',
      typeId: 'pre',
      nom: 'Préfecture - Lozère',
      abreviation: 'Préfecture - Lozère',
      adresse1: 'Rue du Faubourg-Montbel',
      codePostal: '48000',
      commune: 'Mende',
      telephone: '04 66 49 60 00',
      departementId: '48',
      adresse2: 'BP 130',
      email: 'prefecture@lozere.gouv.fr',
      url: 'http://www.lozere.gouv.fr',
    },
    'pre-49007-01': {
      id: 'pre-49007-01',
      typeId: 'pre',
      nom: 'Préfecture - Maine-et-Loire',
      abreviation: 'Préfecture - Maine-et-Loire',
      adresse1: 'Place Michel-Debré',
      codePostal: '49934',
      commune: 'Angers Cedex 9',
      telephone: '02 41 81 81 81',
      departementId: '49',
      email: 'modernisation@maine-et-loire.pref.gouv.fr',
      url: 'http://www.maine-et-loire.gouv.fr',
    },
    'pre-50502-01': {
      id: 'pre-50502-01',
      typeId: 'pre',
      nom: 'Préfecture - Manche',
      abreviation: 'Préfecture - Manche',
      adresse1: 'Place de la Préfecture, BP 70522',
      codePostal: '50002',
      commune: 'Saint-Lô Cedex',
      telephone: '02 33 75 49 50',
      departementId: '50',
      email: 'prefecture@manche.gouv.fr',
      url: 'http://www.manche.gouv.fr',
    },
    'pre-51108-01': {
      id: 'pre-51108-01',
      typeId: 'pre',
      nom: 'Préfecture - Marne',
      abreviation: 'Préfecture - Marne',
      adresse1: '38, rue Carnot',
      codePostal: '51036',
      commune: 'Châlons-en-Champagne Cedex',
      telephone: '03 26 26 10 10',
      departementId: '51',
      email: 'contact@marne.gouv.fr',
      url: 'http://www.marne.gouv.fr',
    },
    'pre-52121-01': {
      id: 'pre-52121-01',
      typeId: 'pre',
      nom: 'Préfecture - Haute-Marne',
      abreviation: 'Préfecture - Haute-Marne',
      adresse1: '89, rue de la Victoire-de-la-Marne',
      codePostal: '52011',
      commune: 'Chaumont Cedex',
      telephone: '03 25 30 52 52',
      departementId: '52',
      email: 'prefecture@haute-marne.gouv.fr',
      url: 'http://www.haute-marne.gouv.fr',
    },
    'pre-53130-01': {
      id: 'pre-53130-01',
      typeId: 'pre',
      nom: 'Préfecture - Mayenne',
      abreviation: 'Préfecture - Mayenne',
      adresse1: '46, rue Mazagran, CS 91507',
      codePostal: '53015',
      commune: 'Laval Cedex',
      telephone: '02 43 01 50 00',
      departementId: '53',
      email: 'pref-communication@mayenne.gouv.fr',
      url: 'http://www.mayenne.gouv.fr',
    },
    'pre-54395-01': {
      id: 'pre-54395-01',
      typeId: 'pre',
      nom: 'Préfecture - Meurthe-et-Moselle',
      abreviation: 'Préfecture - Meurthe-et-Moselle',
      adresse1: '6, rue Sainte-Catherine',
      codePostal: '54000',
      commune: 'Nancy',
      telephone: '03 83 34 26 26',
      departementId: '54',
      adresse2: 'CS 60031, 1, rue du Préfet-Claude-Érignac',
      email: 'pref-informations@meurthe-et-moselle.gouv.fr',
      url: 'http://www.meurthe-et-moselle.gouv.fr',
    },
    'pre-55029-01': {
      id: 'pre-55029-01',
      typeId: 'pre',
      nom: 'Préfecture - Meuse',
      abreviation: 'Préfecture - Meuse',
      adresse1: '40, rue du Bourg, CS 30512',
      codePostal: '55012',
      commune: 'Bar-le-Duc Cedex',
      telephone: '03 29 77 55 55',
      departementId: '55',
      url: 'http://www.meuse.gouv.fr',
    },
    'pre-56260-01': {
      id: 'pre-56260-01',
      typeId: 'pre',
      nom: 'Préfecture - Morbihan',
      abreviation: 'Préfecture - Morbihan',
      adresse1: '24, place de la République',
      codePostal: '56000',
      commune: 'Vannes',
      telephone: '02 97 54 84 00',
      departementId: '56',
      adresse2: 'Préfecture du Morbihan, BP 501',
      url: 'http://www.morbihan.gouv.fr',
    },
    'pre-57463-01': {
      id: 'pre-57463-01',
      typeId: 'pre',
      nom: 'Préfecture - Moselle',
      abreviation: 'Préfecture - Moselle',
      adresse1: '9, place de la Préfecture, BP 71014',
      codePostal: '57034',
      commune: 'Metz Cedex 01',
      telephone: '03 87 34 87 34',
      departementId: '57',
    },
    'pre-58194-01': {
      id: 'pre-58194-01',
      typeId: 'pre',
      nom: 'Préfecture - Nièvre',
      abreviation: 'Préfecture - Nièvre',
      adresse1: '40, rue de la Préfecture',
      codePostal: '58026',
      commune: 'Nevers Cedex',
      telephone: '03 86 60 70 80',
      departementId: '58',
      email: 'courrier@nievre.pref.gouv.fr',
      url: 'http://www.nievre.gouv.fr',
    },
    'pre-59350-01': {
      id: 'pre-59350-01',
      typeId: 'pre',
      nom: 'Préfecture - Nord',
      abreviation: 'Préfecture - Nord',
      adresse1: 'CS 20003, 12, rue Jean-Sans-Peur',
      codePostal: '59039',
      commune: 'Lille Cedex',
      telephone: '03 20 30 59 59',
      departementId: '59',
      url: 'http://www.nord.gouv.fr',
    },
    'pre-60057-01': {
      id: 'pre-60057-01',
      typeId: 'pre',
      nom: 'Préfecture - Oise',
      abreviation: 'Préfecture - Oise',
      adresse1: '1, place de la Préfecture',
      codePostal: '60022',
      commune: 'Beauvais Cedex',
      telephone: '03 44 06 12 60',
      departementId: '60',
      url: 'http://www.oise.gouv.fr',
    },
    'pre-61001-01': {
      id: 'pre-61001-01',
      typeId: 'pre',
      nom: 'Préfecture - Orne',
      abreviation: 'Préfecture - Orne',
      adresse1: 'BP 529, 39, rue Saint-Blaise',
      codePostal: '61018',
      commune: 'Alençon Cedex',
      telephone: '02 33 80 61 61',
      departementId: '61',
      email: 'pref-communication@orne.gouv.fr',
      url: 'http://www.orne.gouv.fr',
    },
    'pre-62041-01': {
      id: 'pre-62041-01',
      typeId: 'pre',
      nom: 'Préfecture - Pas-de-Calais',
      abreviation: 'Préfecture - Pas-de-Calais',
      adresse1: 'Rue Ferdinand-Buisson',
      codePostal: '62020',
      commune: 'Arras Cedex 9',
      telephone: '03 21 21 20 00',
      departementId: '62',
      url: 'http://www.pas-de-calais.gouv.fr',
    },
    'pre-63113-01': {
      id: 'pre-63113-01',
      typeId: 'pre',
      nom: 'Préfecture - Puy-de-Dôme',
      abreviation: 'Préfecture - Puy-de-Dôme',
      adresse1: "1, rue d'Assas",
      codePostal: '63033',
      commune: 'Clermont-Ferrand Cedex 1',
      telephone: '04 73 98 63 63',
      departementId: '63',
      url: 'http://www.puy-de-dome.gouv.fr',
    },
    'pre-64445-01': {
      id: 'pre-64445-01',
      typeId: 'pre',
      nom: 'Préfecture - Pyrénées-Atlantiques',
      abreviation: 'Préfecture - Pyrénées-Atlantiques',
      adresse1: '2, rue du Maréchal-Joffre',
      codePostal: '64021',
      commune: 'Pau Cedex',
      telephone: '05 59 98 24 24',
      departementId: '64',
      email: 'prefecture@pyrenees-atlantiques.gouv.fr',
      url: 'http://www.pyrenees-atlantiques.gouv.fr',
    },
    'pre-65440-01': {
      id: 'pre-65440-01',
      typeId: 'pre',
      nom: 'Préfecture - Hautes-Pyrénées',
      abreviation: 'Préfecture - Hautes-Pyrénées',
      adresse1: 'CS 61350, Place Charles-de-Gaulle',
      codePostal: '65013',
      commune: 'Tarbes Cedex 9',
      telephone: '05 62 56 65 65',
      departementId: '65',
      email: 'prefecture@hautes-pyrenees.gouv.fr',
      url: 'http://www.hautes-pyrenees.gouv.fr',
    },
    'pre-66136-01': {
      id: 'pre-66136-01',
      typeId: 'pre',
      nom: 'Préfecture - Pyrénées-Orientales',
      abreviation: 'Préfecture - Pyrénées-Orientales',
      adresse1: '24, quai Sadi-Carnot, BP 951',
      codePostal: '66951',
      commune: 'Perpignan Cedex',
      telephone: '04 68 51 66 66',
      departementId: '66',
      url: 'http://www.pyrenees-orientales.gouv.fr',
    },
    'pre-67482-01': {
      id: 'pre-67482-01',
      typeId: 'pre',
      nom: 'Préfecture - Bas-Rhin',
      abreviation: 'Préfecture - Bas-Rhin',
      adresse1: '5 place de la République, BP 1047',
      codePostal: '67073',
      commune: 'Strasbourg Cedex',
      telephone: '03 88 21 67 68',
      departementId: '67',
      url: 'http://www.bas-rhin.gouv.fr',
    },
    'pre-68066-01': {
      id: 'pre-68066-01',
      typeId: 'pre',
      nom: 'Préfecture - Haut-Rhin',
      abreviation: 'Préfecture - Haut-Rhin',
      adresse1: '11, avenue de la République',
      codePostal: '68000',
      commune: 'Colmar',
      telephone: '03 89 29 20 00',
      departementId: '68',
      adresse2: '7, rue Bruat, BP 10489',
      email: 'pref-courrier@haut-rhin.gouv.fr',
      url: 'http://www.haut-rhin.gouv.fr',
    },
    'pre-69383-01': {
      id: 'pre-69383-01',
      typeId: 'pre',
      nom: 'Préfecture - Rhône',
      abreviation: 'Préfecture - Rhône',
      adresse1: '97, rue Molière',
      codePostal: '69003',
      commune: 'Lyon',
      telephone: '0821 80 30 69',
      departementId: '69',
      adresse2: '106, rue Pierre-Corneille',
      url: 'http://www.rhone.gouv.fr',
    },
    'pre-70550-01': {
      id: 'pre-70550-01',
      typeId: 'pre',
      nom: 'Préfecture - Haute-Saône',
      abreviation: 'Préfecture - Haute-Saône',
      adresse1: '1, rue de la Préfecture, BP 429',
      codePostal: '70013',
      commune: 'Vesoul Cedex',
      telephone: '03 84 77 70 00',
      departementId: '70',
      email: 'prefecture@haute-saone.pref.gouv.fr',
      url: 'http://www.haute-saone.gouv.fr',
    },
    'pre-71270-01': {
      id: 'pre-71270-01',
      typeId: 'pre',
      nom: 'Préfecture - Saône-et-Loire',
      abreviation: 'Préfecture - Saône-et-Loire',
      adresse1: '196, rue de Strasbourg',
      codePostal: '71021',
      commune: 'Mâcon Cedex 9',
      telephone: '03 85 21 81 00',
      departementId: '71',
      url: 'http://www.saone-et-loire.gouv.fr/prefecture-et-sous-prefectures-r1071.html',
    },
    'pre-72181-01': {
      id: 'pre-72181-01',
      typeId: 'pre',
      nom: 'Préfecture - Sarthe',
      abreviation: 'Préfecture - Sarthe',
      adresse1: 'Place Aristide-Briand',
      codePostal: '72041',
      commune: 'Le Mans Cedex 9',
      telephone: '02 43 39 72 72',
      departementId: '72',
      email: 'pref-mail@sarthe.gouv.fr',
      url: 'http://www.sarthe.gouv.fr',
    },
    'pre-73065-01': {
      id: 'pre-73065-01',
      typeId: 'pre',
      nom: 'Préfecture - Savoie',
      abreviation: 'Préfecture - Savoie',
      adresse1: 'Château des Ducs de Savoie, BP 1801',
      codePostal: '73018',
      commune: 'Chambéry Cedex',
      telephone: '04 79 75 50 50',
      departementId: '73',
      email: 'prefecture@savoie.gouv.fr',
      url: 'http://www.savoie.gouv.fr',
    },
    'pre-74010-01': {
      id: 'pre-74010-01',
      typeId: 'pre',
      nom: 'Préfecture - Haute-Savoie',
      abreviation: 'Préfecture - Haute-Savoie',
      adresse1: "30, rue du 30e-Régiment-d'Infanterie, BP 2332",
      codePostal: '74034',
      commune: 'Annecy Cedex',
      telephone: '04 50 33 60 00',
      departementId: '74',
      email: 'prefecture@haute-savoie.gouv.fr',
      url: 'http://www.haute-savoie.gouv.fr',
    },
    'pre-75104-01': {
      id: 'pre-75104-01',
      typeId: 'pre',
      nom: 'Préfecture de police de Paris',
      abreviation: 'Préfecture de police de Paris',
      adresse1: '1 bis rue de Lutèce',
      codePostal: '75004',
      commune: 'Paris',
      telephone: '3430',
      departementId: '75',
      url: 'http://www.prefecturedepolice.interieur.gouv.fr',
    },
    'pre-76540-01': {
      id: 'pre-76540-01',
      typeId: 'pre',
      nom: 'Préfecture - Seine-Maritime',
      abreviation: 'Préfecture - Seine-Maritime',
      adresse1: '7, place de la Madeleine',
      codePostal: '76036',
      commune: 'Rouen Cedex',
      telephone: '02 32 76 50 00',
      departementId: '76',
      url: 'http://www.seine-maritime.gouv.fr',
    },
    'pre-77288-01': {
      id: 'pre-77288-01',
      typeId: 'pre',
      nom: 'Préfecture - Seine-et-Marne',
      abreviation: 'Préfecture - Seine-et-Marne',
      adresse1: '12 rue des Saints-Pères',
      codePostal: '77010',
      commune: 'Melun Cedex',
      telephone: '01 64 71 76 77',
      departementId: '77',
      url: 'http://www.seine-et-marne.gouv.fr',
    },
    'pre-78646-01': {
      id: 'pre-78646-01',
      typeId: 'pre',
      nom: 'Préfecture - Yvelines',
      abreviation: 'Préfecture - Yvelines',
      adresse1: '1, rue Jean-Houdon',
      codePostal: '78000',
      commune: 'Versailles',
      telephone: '01 39 49 78 00',
      departementId: '78',
      email: 'pref-communication78@yvelines.gouv.fr',
      url: 'http://www.yvelines.gouv.fr',
    },
    'pre-79191-01': {
      id: 'pre-79191-01',
      typeId: 'pre',
      nom: 'Préfecture - Deux-Sèvres',
      abreviation: 'Préfecture - Deux-Sèvres',
      adresse1: '4, rue Duguesclin, BP 70000',
      codePostal: '79099',
      commune: 'Niort Cedex 9',
      telephone: '05 49 08 68 68',
      departementId: '79',
      email: 'courrier@deux-sevres.pref.gouv.fr',
      url: 'http://www.deux-sevres.gouv.fr',
    },
    'pre-80021-01': {
      id: 'pre-80021-01',
      typeId: 'pre',
      nom: 'Préfecture - Somme',
      abreviation: 'Préfecture - Somme',
      adresse1: '51, rue de la République',
      codePostal: '80020',
      commune: 'Amiens Cedex 9',
      telephone: '0821 80 30 80',
      departementId: '80',
      email: 'pref-courrier@somme.gouv.fr',
      url: 'http://www.somme.gouv.fr',
    },
    'pre-81004-01': {
      id: 'pre-81004-01',
      typeId: 'pre',
      nom: 'Préfecture - Tarn',
      abreviation: 'Préfecture - Tarn',
      adresse1: 'Place de la Préfecture',
      codePostal: '81013',
      commune: 'Albi Cedex 9',
      telephone: '05 63 45 61 61',
      departementId: '81',
      email: 'courrier@tarn.pref.gouv.fr',
      url: 'http://www.tarn.gouv.fr',
    },
    'pre-82121-01': {
      id: 'pre-82121-01',
      typeId: 'pre',
      nom: 'Préfecture - Tarn-et-Garonne',
      abreviation: 'Préfecture - Tarn-et-Garonne',
      adresse1: "2, allée de l'Empereur, BP 779",
      codePostal: '82013',
      commune: 'Montauban Cedex',
      telephone: '05 63 22 82 00',
      departementId: '82',
      url: 'http://www.tarn-et-garonne.gouv.fr',
    },
    'pre-83137-01': {
      id: 'pre-83137-01',
      typeId: 'pre',
      nom: 'Préfecture - Var',
      abreviation: 'Préfecture - Var',
      adresse1: "CS 31209, Boulevard du 112e Régiment-d'Infanterie",
      codePostal: '83070',
      commune: 'Toulon Cedex',
      telephone: '04 94 18 83 83',
      departementId: '83',
      url: 'http://www.var.gouv.fr',
    },
    'pre-84007-01': {
      id: 'pre-84007-01',
      typeId: 'pre',
      nom: 'Préfecture - Vaucluse',
      abreviation: 'Préfecture - Vaucluse',
      adresse1: '2, avenue de la Folie',
      codePostal: '84000',
      commune: 'Avignon',
      telephone: '04 88 17 84 84',
      departementId: '84',
      adresse2: "Services de l'Etat en Vaucluse Préfecture",
      email: 'pref-contact@vaucluse.gouv.fr',
      url: 'http://www.vaucluse.gouv.fr',
    },
    'pre-85191-01': {
      id: 'pre-85191-01',
      typeId: 'pre',
      nom: 'Préfecture - Vendée',
      abreviation: 'Préfecture - Vendée',
      adresse1: '29, rue Delille',
      codePostal: '85922',
      commune: 'La Roche-sur-Yon Cedex 9',
      telephone: '02 51 36 70 85',
      departementId: '85',
      email: 'prefecture@vendee.gouv.fr',
      url: 'http://www.vendee.gouv.fr',
    },
    'pre-86194-01': {
      id: 'pre-86194-01',
      typeId: 'pre',
      nom: 'Préfecture - Vienne',
      abreviation: 'Préfecture - Vienne',
      adresse1: 'BP 589, 7, place Aristide-Briand',
      codePostal: '86021',
      commune: 'Poitiers Cedex',
      telephone: '05 49 55 70 00',
      departementId: '86',
      email: 'pref-courrier@vienne.gouv.fr',
      url: 'http://www.vienne.gouv.fr',
    },
    'pre-87085-01': {
      id: 'pre-87085-01',
      typeId: 'pre',
      nom: 'Préfecture - Haute-Vienne',
      abreviation: 'Préfecture - Haute-Vienne',
      adresse1: '12 rue des Combes',
      codePostal: '87000',
      commune: 'Limoges',
      telephone: '05 55 44 18 00',
      departementId: '87',
      url: 'http://www.haute-vienne.gouv.fr',
    },
    'pre-88160-01': {
      id: 'pre-88160-01',
      typeId: 'pre',
      nom: 'Préfecture - Vosges',
      abreviation: 'Préfecture - Vosges',
      adresse1: '1, place Foch',
      codePostal: '88026',
      commune: 'Epinal Cedex',
      telephone: '03 29 69 88 88',
      departementId: '88',
      email: 'prefecture@vosges.gouv.fr',
      url: 'http://www.vosges.gouv.fr',
    },
    'pre-89024-01': {
      id: 'pre-89024-01',
      typeId: 'pre',
      nom: 'Préfecture - Yonne',
      abreviation: 'Préfecture - Yonne',
      adresse1: 'Place de la Préfecture',
      codePostal: '89016',
      commune: 'Auxerre Cedex',
      telephone: '03 86 72 79 89',
      departementId: '89',
      email: 'prefecture@yonne.gouv.fr',
      url: 'http://www.yonne.gouv.fr',
    },
    'pre-90010-01': {
      id: 'pre-90010-01',
      typeId: 'pre',
      nom: 'Préfecture - Territoire de Belfort',
      abreviation: 'Préfecture - Territoire de Belfort',
      adresse1: 'Place de la République',
      codePostal: '90020',
      commune: 'Belfort Cedex',
      telephone: '03 84 57 00 07',
      departementId: '90',
      email: 'courrier90@territoire-de-belfort.pref.gouv.fr',
      url: 'http://www.territoire-belfort.gouv.fr',
    },
    'pre-91228-01': {
      id: 'pre-91228-01',
      typeId: 'pre',
      nom: 'Préfecture - Essonne',
      abreviation: 'Préfecture - Essonne',
      adresse1: 'Boulevard de France',
      codePostal: '91010',
      commune: 'Évry Courcouronnes cedex',
      telephone: '01 69 91 91 91',
      departementId: '91',
      email: 'prefecture@essonne.gouv.fr',
      url: 'http://www.essonne.gouv.fr',
    },
    'pre-92050-01': {
      id: 'pre-92050-01',
      typeId: 'pre',
      nom: 'Préfecture - Hauts-de-Seine',
      abreviation: 'Préfecture - Hauts-de-Seine',
      adresse1: '167-177, avenue Joliot-Curie',
      codePostal: '92013',
      commune: 'Nanterre Cedex',
      telephone: '01 40 97 20 00',
      departementId: '92',
      email: 'prefecture@hauts-de-seine.gouv.fr',
      url: 'http://www.hauts-de-seine.gouv.fr',
    },
    'pre-93008-01': {
      id: 'pre-93008-01',
      typeId: 'pre',
      nom: 'Préfecture - Seine-Saint-Denis',
      abreviation: 'Préfecture - Seine-Saint-Denis',
      adresse1: '1, esplanade Jean-Moulin',
      codePostal: '93007',
      commune: 'Bobigny Cedex',
      telephone: '01 41 60 60 60',
      departementId: '93',
      email: 'prefecture@seine-saint-denis.gouv.fr',
      url: 'http://www.seine-saint-denis.gouv.fr',
    },
    'pre-94028-01': {
      id: 'pre-94028-01',
      typeId: 'pre',
      nom: 'Préfecture - Val-de-Marne',
      abreviation: 'Préfecture - Val-de-Marne',
      adresse1: '21-29, avenue du Général-de-Gaulle',
      codePostal: '94038',
      commune: 'Créteil Cedex',
      telephone: '01 49 56 60 00',
      departementId: '94',
      email: 'prefecture@val-de-marne.gouv.fr',
      url: 'http://www.val-de-marne.gouv.fr',
    },
    'pre-95127-01': {
      id: 'pre-95127-01',
      typeId: 'pre',
      nom: "Préfecture - Val-d'Oise",
      abreviation: "Préfecture - Val-d'Oise",
      adresse1: 'CS 20105, 5, avenue Bernard-Hirsch',
      codePostal: '95010',
      commune: 'Cergy-Pontoise Cedex',
      telephone: '01 34 20 95 95',
      departementId: '95',
      email: 'prefecture@val-doise.gouv.fr',
      url: 'http://www.val-doise.gouv.fr',
    },
    'pre-97105-01': {
      id: 'pre-97105-01',
      typeId: 'pre',
      nom: 'Préfecture - Guadeloupe',
      abreviation: 'Préfecture - Guadeloupe',
      adresse1: 'Avenue Paul-Lacavé',
      codePostal: '97109',
      commune: 'Basse-Terre Cedex',
      telephone: '+590 590 99 39 00',
      departementId: '971',
      url: 'http://www.guadeloupe.pref.gouv.fr',
    },
    'pre-97209-01': {
      id: 'pre-97209-01',
      typeId: 'pre',
      nom: 'Préfecture - Martinique',
      abreviation: 'Préfecture - Martinique',
      adresse1: '82, rue Victor-Sévère',
      codePostal: '97262',
      commune: 'Fort-de-France Cedex',
      telephone: '+596 596 39 36 00',
      departementId: '972',
      email: 'contact.prefecture@martinique.pref.gouv.fr',
      url: 'http://www.martinique.pref.gouv.fr',
    },
    'pre-97302-01': {
      id: 'pre-97302-01',
      typeId: 'pre',
      nom: 'Préfecture - Guyane',
      abreviation: 'Préfecture - Guyane',
      url: 'http://www.guyane.pref.gouv.fr',
      email: 'courrier@guyane.pref.gouv.fr',
      telephone: '+594 594 39 45 00',
      adresse1: 'BP 7008, Rue Fiedmond',
      codePostal: '97307',
      commune: 'Cayenne Cedex',
      departementId: '973',
    },
    'pre-97411-01': {
      id: 'pre-97411-01',
      typeId: 'pre',
      nom: 'Préfecture - La Réunion',
      abreviation: 'Préfecture - La Réunion',
      adresse1: '1, rue de la Messagerie, CS 51079',
      codePostal: '97404',
      commune: 'Saint-Denis Cedex',
      telephone: '+262 262 40 77 77',
      departementId: '974',
      url: 'http://www.reunion.gouv.fr',
    },
    'pre-97611-01': {
      id: 'pre-97611-01',
      typeId: 'pre',
      nom: 'Préfecture - Mayotte',
      abreviation: 'Préfecture - Mayotte',
      adresse1: 'BP 676, Kawéni, Avenue de la Préfecture',
      codePostal: '97600',
      commune: 'Mamoudzou',
      telephone: '+262 269 63 50 00',
      departementId: '976',
      url: 'http://www.mayotte.pref.gouv.fr/',
    },
  } // ----- ne pas supprimer cette ligne : fin

export const sortedAdministrations = toSorted(
  map(IDS, id => Administrations[id]),
  (a, b) => a.abreviation.localeCompare(b.abreviation)
)
