import { Meta, StoryFn, StoryObj } from '@storybook/vue3'
import { EtapeDocumentsEdit } from './etape-documents-edit'
import { ETAPE_IS_BROUILLON, ETAPE_IS_NOT_BROUILLON, EtapeDocument, GetEtapeDocumentsByEtapeId, etapeDocumentIdValidator, etapeIdValidator } from 'camino-common/src/etape'
import { ApiClient } from '../../api/api-client'
import { action } from '@storybook/addon-actions'
import { tempDocumentNameValidator } from 'camino-common/src/document'
import { testBlankUser } from 'camino-common/src/tests-utils'
import { useArgs } from '@storybook/preview-api'
import { entrepriseIdValidator } from 'camino-common/src/entreprise'
import { demarcheIdValidator } from 'camino-common/src/demarche'
import { firstEtapeDateValidator } from 'camino-common/src/date'

const meta: Meta = {
  title: 'Components/Etape/EtapeDocumentsEdit',
  // @ts-ignore
  component: EtapeDocumentsEdit,
}
export default meta

const documents: EtapeDocument[] = [
  {
    type: 'DESCRIPTION_OPTIONNELLE',
    id: etapeDocumentIdValidator.parse('id'),
    etape_document_type_id: 'dep',
    description: 'Une description',
    public_lecture: false,
    entreprises_lecture: false,
  },
  {
    type: 'DESCRIPTION_OPTIONNELLE',
    id: etapeDocumentIdValidator.parse('id-car'),
    etape_document_type_id: 'car',
    description: 'Une description',
    public_lecture: false,
    entreprises_lecture: false,
  },
  {
    type: 'DESCRIPTION_OPTIONNELLE',
    id: etapeDocumentIdValidator.parse('id2'),
    etape_document_type_id: 'doe',
    description: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  {
    type: 'DESCRIPTION_OBLIGATOIRE',
    id: etapeDocumentIdValidator.parse('idAutre'),
    etape_document_type_id: 'aut',
    description: 'description autre document',
    public_lecture: true,
    entreprises_lecture: true,
  },
]

const uploadTempDocumentAction = action('uploadTempDocument')
const getEtapeDocumentsByEtapeIdAction = action('getEtapeDocumentsByEtapeId')

const apiClient: Pick<ApiClient, 'uploadTempDocument' | 'getEtapeDocumentsByEtapeId'> = {
  getEtapeDocumentsByEtapeId: etapeId => {
    getEtapeDocumentsByEtapeIdAction(etapeId)

    return Promise.resolve({ etapeDocuments: documents })
  },
  uploadTempDocument: document => {
    uploadTempDocumentAction(document)

    return Promise.resolve(tempDocumentNameValidator.parse(new Date().toISOString()))
  },
}

const completeUpdateAction = action('completeUpdate')

const demarcheId = demarcheIdValidator.parse('demarcheId')
const firstEtapeDate = firstEtapeDateValidator.parse('2022-01-01')

export const Empty: StoryFn = () => (
  <EtapeDocumentsEdit
    apiClient={{ ...apiClient, getEtapeDocumentsByEtapeId: () => Promise.resolve({ etapeDocuments: [] }) }}
    contenu={{}}
    etapeId={etapeIdValidator.parse('etapeId')}
    sdomZoneIds={[]}
    tde={{ titreTypeId: 'arm', demarcheTypeId: 'oct', etapeTypeId: 'mfr', demarcheId, firstEtapeDate }}
    isBrouillon={ETAPE_IS_NOT_BROUILLON}
    completeUpdate={completeUpdateAction}
    user={{ ...testBlankUser, role: 'super' }}
  />
)
export const Rempli: StoryFn = () => (
  <EtapeDocumentsEdit
    apiClient={apiClient}
    contenu={{}}
    etapeId={etapeIdValidator.parse('etapeId')}
    sdomZoneIds={[]}
    tde={{ titreTypeId: 'arm', demarcheTypeId: 'oct', etapeTypeId: 'mfr', demarcheId, firstEtapeDate }}
    isBrouillon={ETAPE_IS_NOT_BROUILLON}
    completeUpdate={completeUpdateAction}
    user={{ ...testBlankUser, role: 'super' }}
  />
)

export const Complet: StoryFn = () => (
  <EtapeDocumentsEdit
    apiClient={{
      ...apiClient,
      getEtapeDocumentsByEtapeId: () =>
        Promise.resolve({
          etapeDocuments: [
            {
              type: 'DESCRIPTION_OPTIONNELLE',
              id: etapeDocumentIdValidator.parse('id'),
              etape_document_type_id: 'dep',
              description: 'Une description',
              public_lecture: false,
              entreprises_lecture: false,
            },
            {
              type: 'DESCRIPTION_OPTIONNELLE',
              id: etapeDocumentIdValidator.parse('id2'),
              etape_document_type_id: 'dom',
              description: 'Une autre description',
              public_lecture: false,
              entreprises_lecture: false,
            },
            {
              type: 'DESCRIPTION_OPTIONNELLE',
              id: etapeDocumentIdValidator.parse('id3'),
              etape_document_type_id: 'for',
              description: null,
              public_lecture: false,
              entreprises_lecture: false,
            },
            {
              type: 'DESCRIPTION_OPTIONNELLE',
              id: etapeDocumentIdValidator.parse('id4'),
              etape_document_type_id: 'jpa',
              description: null,
              public_lecture: false,
              entreprises_lecture: false,
            },
            {
              type: 'DESCRIPTION_OPTIONNELLE',
              id: etapeDocumentIdValidator.parse('id5'),
              etape_document_type_id: 'car',
              description: null,
              public_lecture: false,
              entreprises_lecture: false,
            },
          ],
        }),
    }}
    contenu={{}}
    etapeId={etapeIdValidator.parse('etapeId')}
    sdomZoneIds={[]}
    tde={{ titreTypeId: 'arm', demarcheTypeId: 'oct', etapeTypeId: 'mfr', demarcheId, firstEtapeDate }}
    isBrouillon={ETAPE_IS_NOT_BROUILLON}
    completeUpdate={completeUpdateAction}
    user={{ ...testBlankUser, role: 'super' }}
  />
)

export const CompletAvecDocumentNonRenseigne: StoryFn = () => (
  <EtapeDocumentsEdit
    apiClient={{
      ...apiClient,
      getEtapeDocumentsByEtapeId: () =>
        Promise.resolve({
          etapeDocuments: [
            {
              type: 'DESCRIPTION_OPTIONNELLE',
              id: etapeDocumentIdValidator.parse('id'),
              etape_document_type_id: 'dep',
              description: 'Une description',
              public_lecture: false,
              entreprises_lecture: false,
            },
            {
              type: 'NON_RENSEIGNE',
              id: etapeDocumentIdValidator.parse('id2'),
              etape_document_type_id: 'dom',
              description: 'Une autre description',
              public_lecture: false,
              entreprises_lecture: false,
            },
            {
              type: 'DESCRIPTION_OPTIONNELLE',
              id: etapeDocumentIdValidator.parse('id3'),
              etape_document_type_id: 'for',
              description: null,
              public_lecture: false,
              entreprises_lecture: false,
            },
            {
              type: 'DESCRIPTION_OPTIONNELLE',
              id: etapeDocumentIdValidator.parse('id4'),
              etape_document_type_id: 'jpa',
              description: null,
              public_lecture: false,
              entreprises_lecture: false,
            },
            {
              type: 'DESCRIPTION_OPTIONNELLE',
              id: etapeDocumentIdValidator.parse('id5'),
              etape_document_type_id: 'car',
              description: null,
              public_lecture: false,
              entreprises_lecture: false,
            },
          ],
        }),
    }}
    contenu={{}}
    etapeId={etapeIdValidator.parse('etapeId')}
    sdomZoneIds={[]}
    tde={{ titreTypeId: 'arm', demarcheTypeId: 'oct', etapeTypeId: 'mfr', demarcheId, firstEtapeDate }}
    isBrouillon={ETAPE_IS_NOT_BROUILLON}
    completeUpdate={completeUpdateAction}
    user={{ ...testBlankUser, role: 'super' }}
  />
)

export const ArmMecanise: StoryFn = () => (
  <EtapeDocumentsEdit
    apiClient={apiClient}
    contenu={{ arm: { mecanise: { value: true, heritee: false, etapeHeritee: null } } }}
    etapeId={etapeIdValidator.parse('etapeId')}
    sdomZoneIds={[]}
    tde={{ titreTypeId: 'arm', demarcheTypeId: 'oct', etapeTypeId: 'mfr', demarcheId, firstEtapeDate }}
    isBrouillon={ETAPE_IS_NOT_BROUILLON}
    completeUpdate={completeUpdateAction}
    user={{ ...testBlankUser, role: 'admin', administrationId: 'dea-guyane-01' }}
  />
)

export const ArmMecaniseDynamicNoSnapshot: StoryObj<{ mecanise: boolean }> = {
  render: function Component(args: { mecanise: boolean }) {
    const [, setArgs] = useArgs()

    return () => (
      <div>
        <button onClick={() => setArgs({ mecanise: !args.mecanise })}> Change la mécanisation </button>
        <EtapeDocumentsEdit
          apiClient={apiClient}
          contenu={{ arm: { mecanise: { value: args.mecanise, heritee: false, etapeHeritee: null } } }}
          etapeId={etapeIdValidator.parse('etapeId')}
          sdomZoneIds={[]}
          tde={{ titreTypeId: 'arm', demarcheTypeId: 'oct', etapeTypeId: 'mfr', demarcheId, firstEtapeDate }}
          isBrouillon={ETAPE_IS_NOT_BROUILLON}
          completeUpdate={completeUpdateAction}
          user={{ ...testBlankUser, role: 'super' }}
        />
      </div>
    )
  },
  args: {
    mecanise: true,
  },
}

export const EnConstruction: StoryFn = () => (
  <EtapeDocumentsEdit
    apiClient={apiClient}
    contenu={{ arm: { mecanise: { value: true, heritee: false, etapeHeritee: null } } }}
    etapeId={etapeIdValidator.parse('etapeId')}
    sdomZoneIds={[]}
    tde={{ titreTypeId: 'arm', demarcheTypeId: 'oct', etapeTypeId: 'mfr', demarcheId, firstEtapeDate }}
    isBrouillon={ETAPE_IS_BROUILLON}
    completeUpdate={completeUpdateAction}
    user={{ ...testBlankUser, role: 'super' }}
  />
)
export const OctroiAxmUtilisateurSuper: StoryFn = () => (
  <EtapeDocumentsEdit
    apiClient={apiClient}
    contenu={{}}
    etapeId={etapeIdValidator.parse('etapeId')}
    sdomZoneIds={[]}
    tde={{ titreTypeId: 'axm', demarcheTypeId: 'oct', etapeTypeId: 'mfr', demarcheId, firstEtapeDate }}
    isBrouillon={ETAPE_IS_BROUILLON}
    completeUpdate={completeUpdateAction}
    user={{ ...testBlankUser, role: 'super' }}
  />
)

export const OctroiAxmUtilisateurEntreprise: StoryFn = () => (
  <EtapeDocumentsEdit
    apiClient={apiClient}
    contenu={{}}
    etapeId={etapeIdValidator.parse('etapeId')}
    sdomZoneIds={[]}
    tde={{ titreTypeId: 'axm', demarcheTypeId: 'oct', etapeTypeId: 'mfr', demarcheId, firstEtapeDate }}
    isBrouillon={ETAPE_IS_BROUILLON}
    completeUpdate={completeUpdateAction}
    user={{ ...testBlankUser, role: 'entreprise', entrepriseIds: [entrepriseIdValidator.parse('idEntreprise1')] }}
  />
)

export const OctroiAxmUtilisateurEntrepriseComplet: StoryFn = () => (
  <EtapeDocumentsEdit
    apiClient={{
      ...apiClient,
      getEtapeDocumentsByEtapeId(etapeId) {
        getEtapeDocumentsByEtapeIdAction(etapeId)

        return Promise.resolve<GetEtapeDocumentsByEtapeId>({
          etapeDocuments: [
            {
              type: 'DESCRIPTION_OPTIONNELLE',
              id: etapeDocumentIdValidator.parse('id'),
              etape_document_type_id: 'dep',
              description: 'Une description',
              public_lecture: false,
              entreprises_lecture: false,
            },
            {
              type: 'DESCRIPTION_OPTIONNELLE',
              id: etapeDocumentIdValidator.parse('id2'),
              etape_document_type_id: 'dom',
              description: 'Une autre description',
              public_lecture: false,
              entreprises_lecture: false,
            },
            {
              type: 'DESCRIPTION_OPTIONNELLE',
              id: etapeDocumentIdValidator.parse('id3'),
              etape_document_type_id: 'for',
              description: null,
              public_lecture: false,
              entreprises_lecture: false,
            },
            {
              type: 'DESCRIPTION_OPTIONNELLE',
              id: etapeDocumentIdValidator.parse('id4'),
              etape_document_type_id: 'jpa',
              description: null,
              public_lecture: false,
              entreprises_lecture: false,
            },
            {
              type: 'DESCRIPTION_OPTIONNELLE',
              id: etapeDocumentIdValidator.parse('id5'),
              etape_document_type_id: 'car',
              description: null,
              public_lecture: false,
              entreprises_lecture: false,
            },
          ],
        })
      },
    }}
    contenu={{}}
    etapeId={etapeIdValidator.parse('etapeId')}
    sdomZoneIds={[]}
    tde={{ titreTypeId: 'axm', demarcheTypeId: 'oct', etapeTypeId: 'mfr', demarcheId, firstEtapeDate }}
    isBrouillon={ETAPE_IS_BROUILLON}
    completeUpdate={completeUpdateAction}
    user={{ ...testBlankUser, role: 'entreprise', entrepriseIds: [entrepriseIdValidator.parse('idEntreprise1')] }}
  />
)

export const SdomZone: StoryFn = () => (
  <EtapeDocumentsEdit
    apiClient={apiClient}
    contenu={{ arm: { mecanise: { value: true, heritee: false, etapeHeritee: null } } }}
    etapeId={etapeIdValidator.parse('etapeId')}
    sdomZoneIds={['1', '2']}
    tde={{ titreTypeId: 'axm', demarcheTypeId: 'oct', etapeTypeId: 'mfr', demarcheId, firstEtapeDate }}
    isBrouillon={ETAPE_IS_NOT_BROUILLON}
    completeUpdate={completeUpdateAction}
    user={{ ...testBlankUser, role: 'super' }}
  />
)

export const PasDeDocumentsObligatoires: StoryFn = () => (
  <EtapeDocumentsEdit
    apiClient={apiClient}
    contenu={{}}
    etapeId={etapeIdValidator.parse('etapeId')}
    sdomZoneIds={['1', '2']}
    tde={{ titreTypeId: 'prm', demarcheTypeId: 'pro', etapeTypeId: 'mfr', demarcheId, firstEtapeDate }}
    isBrouillon={ETAPE_IS_NOT_BROUILLON}
    completeUpdate={completeUpdateAction}
    user={{ ...testBlankUser, role: 'super' }}
  />
)

export const Loading: StoryFn = () => (
  <EtapeDocumentsEdit
    apiClient={{ ...apiClient, getEtapeDocumentsByEtapeId: () => new Promise(() => ({})) }}
    contenu={{}}
    etapeId={etapeIdValidator.parse('etapeId')}
    sdomZoneIds={[]}
    tde={{ titreTypeId: 'arm', demarcheTypeId: 'oct', etapeTypeId: 'mfr', demarcheId, firstEtapeDate }}
    isBrouillon={ETAPE_IS_NOT_BROUILLON}
    completeUpdate={completeUpdateAction}
    user={{ ...testBlankUser, role: 'super' }}
  />
)
export const WithError: StoryFn = () => (
  <EtapeDocumentsEdit
    apiClient={{ ...apiClient, getEtapeDocumentsByEtapeId: () => Promise.resolve({ message: 'Une erreur est survenue' }) }}
    contenu={{}}
    etapeId={etapeIdValidator.parse('etapeId')}
    sdomZoneIds={[]}
    tde={{ titreTypeId: 'arm', demarcheTypeId: 'oct', etapeTypeId: 'mfr', demarcheId, firstEtapeDate }}
    isBrouillon={ETAPE_IS_NOT_BROUILLON}
    completeUpdate={completeUpdateAction}
    user={{ ...testBlankUser, role: 'super' }}
  />
)
