import { Domaine } from '@/components/_common/domaine'
import { TitresTypesTypes } from 'camino-common/src/static/titresTypesTypes'
import { getDomaineId, getTitreTypeType } from 'camino-common/src/static/titresTypes'
import { createDebounce } from '@/utils/debounce'
import { useRouter } from 'vue-router'
import { ref, FunctionalComponent, defineComponent } from 'vue'
import { titreApiClient } from '../titre/titre-api-client'
import { capitalize } from 'camino-common/src/strings'
import { isNotNullNorUndefined } from 'camino-common/src/typescript-tools'
import { CaminoAnnee, getAnnee } from 'camino-common/src/date'
import { TypeAheadSingle } from '../_ui/typeahead-single'
import { useState } from '@/utils/vue-tsx-utils'
import { QuickAccessResult } from 'camino-common/src/titres'

export const QuickAccessTitre = defineComponent<{ id: string; onSelectTitre: () => void }>(props => {
  const router = useRouter()
  const [titres, setTitres] = useState<QuickAccessResult[]>([])

  const search = async (searchTerm: string): Promise<void> => {
    const searchTitres = await titreApiClient.quickAccess(searchTerm)
    if ('message' in searchTitres) {
      console.error(searchTitres)
      setTitres([])
    } else {
      setTitres(searchTitres)
    }
  }

  const onSelectedTitre = (titre: QuickAccessResult | undefined) => {
    if (titre) {
      router.push({ name: 'titre', params: { id: titre.id } })
      props.onSelectTitre()
    }
  }

  return () => <PureQuickAccessTitre titres={titres.value} onSearch={search} onSelectedTitre={onSelectedTitre} id={props.id} />
})

// @ts-ignore waiting for https://github.com/vuejs/core/issues/7833
QuickAccessTitre.props = ['id', 'onSelectTitre']

interface Props {
  id: string
  titres: QuickAccessResult[]
  onSelectedTitre: (titre: QuickAccessResult | undefined) => void
  alwaysOpen?: boolean
  onSearch: (searchTerm: string) => void
}
interface DisplayTitreProps {
  titre: Pick<QuickAccessResult, 'nom' | 'typeId' | 'titreDateDebut'>
}
export const DisplayTitre: FunctionalComponent<DisplayTitreProps> = props => {
  let annee: CaminoAnnee | null = null
  if (isNotNullNorUndefined(props.titre.titreDateDebut)) {
    annee = getAnnee(props.titre.titreDateDebut)
  }

  return (
    <div style={{ display: 'flex', flexDirection: 'row', justifyContent: 'flex-start', alignItems: 'flex-start', textAlign: 'left' }}>
      <Domaine domaineId={getDomaineId(props.titre.typeId)} />
      <div style={{ display: 'flex', flexDirection: 'column' }} class="fr-pl-2w">
        <span class="fr-text--bold">
          {capitalize(props.titre.nom)}
          {annee ? ` - ${annee}` : null}
        </span>
        <span class="fr-text">{capitalize(TitresTypesTypes[getTitreTypeType(props.titre.typeId)].nom)}</span>
      </div>
    </div>
  )
}

export const PureQuickAccessTitre = defineComponent<Props>(props => {
  const display = (titre: QuickAccessResult) => {
    return <DisplayTitre titre={titre} />
  }

  const overrideItem = ref<QuickAccessResult | null>(null)
  const selectItem = (item: QuickAccessResult | undefined) => {
    overrideItem.value = null
    props.onSelectedTitre(item)
  }

  const debounce = createDebounce()

  return () => (
    <div class="fr-search-bar" id="search-quick-access-titre" role="search">
      <label class="fr-label" for={props.id}>
        Rechercher
      </label>
      <TypeAheadSingle
        overrideItem={overrideItem.value}
        props={{
          id: props.id,
          itemKey: 'id',
          placeholder: 'Rechercher un titre',
          items: props.titres,
          minInputLength: 3,
          itemChipLabel: item => item.nom,
          onSelectItem: selectItem,
          onInput: event => debounce(() => props.onSearch(event)),
          displayItemInList: display,
          alwaysOpen: props.alwaysOpen ?? false,
          clearAfterSelect: true,
        }}
      />
      <button class="fr-btn" title="Rechercher">
        Rechercher
      </button>
    </div>
  )
})

// @ts-ignore waiting for https://github.com/vuejs/core/issues/7833
PureQuickAccessTitre.props = ['id', 'titres', 'onSelectedTitre', 'onSearch', 'alwaysOpen']
