import { defineComponent, FunctionalComponent } from 'vue'
import { Column, TableRow } from './_ui/table'
import { useRouter } from 'vue-router'
import { Liste, Params } from './_common/liste'
import { ApiClient, apiClient } from '@/api/api-client'

import { CaminoRouteLocation, routesDefinitions } from '@/router/routes'
import { CaminoRouter } from '@/typings/vue-router'
import { Mutation } from 'camino-common/src/mutations'
import { DeleteRestRoutes, NewDeleteRestRoutes, NewPostRestRoutes, NewPutRestRoutes } from 'camino-common/src/rest'
import { CaminoRouterLink } from '@/router/camino-router-link'
import { dateTimeFormat } from 'camino-common/src/date'

export const Mutations = defineComponent(() => {
  const router = useRouter()
  return () => <PureMutations apiClient={apiClient} updateUrlQuery={router} currentRoute={router.currentRoute.value} />
})

interface Props {
  apiClient: Pick<ApiClient, 'getMutations' | 'titresRechercherByNom' | 'getTitresByIds'>
  currentRoute: CaminoRouteLocation
  updateUrlQuery: Pick<CaminoRouter, 'push'>
}

const colonnesData = [
  { id: 'date', contentTitle: 'Date', noSort: true },
  { id: 'utilisateur', contentTitle: 'Utilisateur', noSort: true },
  { id: 'explication', contentTitle: 'Explication', noSort: true },
  { id: 'detail', contentTitle: 'Action', noSort: true },
] as const satisfies Column[]

type ColonneId = (typeof colonnesData)[number]['id']

// ici on veut avoir un match exhaustif de toutes les routes connues, mais les anciennes routes doivent quand même fonctionner sans faire planter l'application
const exhaustiveCheckPass = (_param: never): never => {
  return "Pas d'explication" as never
}

const PathToExplanation: FunctionalComponent<{ mutation: Mutation }> = props => {
  const baseText = props.mutation.method === 'delete' ? 'Suppression ' : props.mutation.method === 'put' ? 'Modification ' : 'Création '
  const caminoPath = props.mutation.camino_path as NewPostRestRoutes | NewPutRestRoutes | DeleteRestRoutes | NewDeleteRestRoutes
  switch (caminoPath) {
    case '/rest/activites/:activiteId':
      return (
        <>
          {baseText}{' '}
          <CaminoRouterLink
            title={`Aller vers l'activité ${props.mutation.camino_variables.activiteId}`}
            isDisabled={false}
            to={{ name: 'activite', params: { activiteId: props.mutation.camino_variables.activiteId } }}
          >
            d'une activité
          </CaminoRouterLink>
        </>
      )
    case '/rest/activites/:activiteId/reset-depot':
      return (
        <>
          Réinitialisation{' '}
          <CaminoRouterLink
            title={`Aller vers l'activité ${props.mutation.camino_variables.activiteId}`}
            isDisabled={false}
            to={{ name: 'activite', params: { activiteId: props.mutation.camino_variables.activiteId } }}
          >
            d'une activité
          </CaminoRouterLink>
        </>
      )
    case '/rest/demarches':
      return <>{baseText} d'une démarche</>
    case '/rest/etapes':
      return <>{baseText} d'une étape</>
    case '/rest/titres':
      return <>{baseText} d'un titre</>
    case '/rest/entreprises':
      return <>{baseText} d'une entreprise</>
    case '/rest/demarches/:demarcheIdOrSlug':
      return (
        <>
          {baseText}{' '}
          <CaminoRouterLink
            title={`Aller vers la démarche ${props.mutation.camino_variables.demarcheIdOrSlug}`}
            isDisabled={false}
            to={{ name: 'demarche', params: { demarcheId: props.mutation.camino_variables.demarcheIdOrSlug } }}
          >
            d'une démarche
          </CaminoRouterLink>
        </>
      )
    case '/rest/titres/:titreId':
      return (
        <>
          {baseText}{' '}
          <CaminoRouterLink title={`Aller vers le titre ${props.mutation.camino_variables.titreId}`} isDisabled={false} to={{ name: 'titre', params: { id: props.mutation.camino_variables.titreId } }}>
            d'un titre
          </CaminoRouterLink>
        </>
      )
    case '/rest/etapes/:etapeIdOrSlug':
      return (
        <>
          {baseText}{' '}
          <CaminoRouterLink
            title={`Aller vers l'étape ${props.mutation.camino_variables.etapeIdOrSlug}`}
            isDisabled={false}
            to={{ name: 'etape', params: { id: props.mutation.camino_variables.etapeIdOrSlug } }}
          >
            d'une étape
          </CaminoRouterLink>
        </>
      )
    case '/rest/etapes/:etapeId/depot':
      return (
        <>
          Finalisation{' '}
          <CaminoRouterLink title={`Aller vers l'étape ${props.mutation.camino_variables.etapeId}`} isDisabled={false} to={{ name: 'etape', params: { id: props.mutation.camino_variables.etapeId } }}>
            d'une étape
          </CaminoRouterLink>
        </>
      )
    case '/rest/entreprises/:entrepriseId':
      return (
        <>
          {baseText}{' '}
          <CaminoRouterLink
            title={`Aller vers l'entreprise ${props.mutation.camino_variables.entrepriseId}`}
            isDisabled={false}
            to={{ name: 'entreprise', params: { id: props.mutation.camino_variables.entrepriseId } }}
          >
            d'une entreprise
          </CaminoRouterLink>
        </>
      )
    case '/rest/entreprises/:entrepriseId/documents':
    case '/rest/entreprises/:entrepriseId/documents/:entrepriseDocumentId':
      return (
        <>
          {baseText}{' '}
          <CaminoRouterLink
            title={`Aller vers l'entreprise ${props.mutation.camino_variables.entrepriseId}`}
            isDisabled={false}
            to={{ name: 'entreprise', params: { id: props.mutation.camino_variables.entrepriseId } }}
          >
            d'un document d'entreprise
          </CaminoRouterLink>
        </>
      )
    case '/rest/titres/:id/titreLiaisons':
      return (
        <>
          {baseText}{' '}
          <CaminoRouterLink title={`Aller vers le titre ${props.mutation.camino_variables.id}`} isDisabled={false} to={{ name: 'titre', params: { id: props.mutation.camino_variables.id } }}>
            d'un lien entre deux titres
          </CaminoRouterLink>
        </>
      )
    case '/rest/administrations/:administrationId/activiteTypeEmails':
    case '/rest/administrations/:administrationId/activiteTypeEmails/delete':
      return (
        <>
          {baseText}{' '}
          <CaminoRouterLink
            title={`Aller vers l'administration ${props.mutation.camino_variables.administrationId}`}
            isDisabled={false}
            to={{ name: 'administration', params: { id: props.mutation.camino_variables.administrationId } }}
          >
            d'un email pour l'activité d'une administration
          </CaminoRouterLink>
        </>
      )
    case '/rest/titres/:titreId/abonne':
      return (
        <>
          {baseText}{' '}
          <CaminoRouterLink title={`Aller vers le titre ${props.mutation.camino_variables.titreId}`} isDisabled={false} to={{ name: 'titre', params: { id: props.mutation.camino_variables.titreId } }}>
            d'un abonnement sur un titre
          </CaminoRouterLink>
        </>
      )
    case '/rest/utilisateurs/:id/permission':
      return (
        <>
          {baseText}{' '}
          <CaminoRouterLink
            title={`Aller vers l'utilisateur ${props.mutation.camino_variables.id}`}
            isDisabled={false}
            to={{ name: 'utilisateur', params: { id: props.mutation.camino_variables.id } }}
          >
            des permissions d'un utilisateur
          </CaminoRouterLink>
        </>
      )
    case '/rest/utilisateur/generateQgisToken':
    case '/rest/geojson/import/:geoSystemeId':
    case '/rest/geojson_forages/import/:geoSystemeId':
    case '/rest/geojson_points/import/:geoSystemeId':
      return <></>
    default:
      return exhaustiveCheckPass(caminoPath)
  }
}

const lignes = (mutations: Mutation[]): TableRow<ColonneId>[] => {
  return mutations.map(mutation => {
    const columns: TableRow<ColonneId>['columns'] = {
      date: {
        type: 'jsx',
        jsxElement: (
          <CaminoRouterLink title={`Aller à la mutation ${mutation.id}`} isDisabled={false} to={{ name: 'mutation', params: { mutationId: mutation.id } }}>
            {dateTimeFormat(new Date(mutation.datetime))}
          </CaminoRouterLink>
        ),
        value: mutation.datetime,
      },
      utilisateur: {
        type: 'text',
        value: `${mutation.utilisateur_nom} ${mutation.utilisateur_prenom}`,
      },
      explication: {
        type: 'jsx',
        jsxElement: <PathToExplanation mutation={mutation} />,
        value: mutation.id,
      },
      detail: {
        type: 'text',
        value: `${mutation.method} -> ${mutation.path}`,
      },
    }

    return {
      id: mutation.id,
      link: null,
      columns,
    }
  })
}

const filteredPaths = [
  '/rest/utilisateur/generateQgisToken',
  '/rest/geojson/import/:geoSystemeId',
  '/rest/geojson_forages/import/:geoSystemeId',
  '/rest/geojson_points/import/:geoSystemeId',
] as const satisfies (NewPostRestRoutes | NewPutRestRoutes | DeleteRestRoutes | NewDeleteRestRoutes)[]
export const PureMutations = defineComponent<Props>(props => {
  const getData = async (event: Params<string>) => {
    const values = await props.apiClient.getMutations(event.filtres?.emails)
    if ('message' in values) {
      return values
    }

    const filteredValue = values.filter(value => !filteredPaths.includes(value.camino_path))

    return { total: filteredValue.length, values: lignes(filteredValue) }
  }

  return () => (
    <Liste
      listeFiltre={{
        filtres: routesDefinitions[props.currentRoute.name].meta.filtres,
        apiClient: props.apiClient,
        updateUrlQuery: props.updateUrlQuery,
        entreprises: [],
      }}
      renderButton={null}
      download={null}
      colonnes={colonnesData}
      route={props.currentRoute}
      getData={getData}
      nom="Mutations"
    />
  )
})
// @ts-ignore waiting for https://github.com/vuejs/core/issues/7833
PureMutations.props = ['apiClient', 'currentRoute', 'updateUrlQuery']
