import { z } from 'zod'
import { CaminoDate, toCaminoDate } from '../date'
import { Definition } from '../definition'
import { EtapeBrouillon } from '../etape'

// prettier-ignore
const IDS = ["abd", "aca","aco","and","anf","def","dex","dpu", "ihi", "mfr","mod","rca","rcb","rcd","rcm","rco","rcs","rif","rmc", "rpu","sco","acf","apd","ape","apo","app","apu","cac","css","dae","ddc","des","dma","epu","exp","ipc", "mca","mcb","mcd", "mci","mcm","mco","mcp","mcr","mcs","mec","men","meo","mie","mif","mim","mna","mnb","mnc","mnd","mni","mno","mns","mnv","mpb", "ncl","pfc","pfd","ppu","pqr","rde","rno","rpe","sas","sca","scg","spe","spo","spp","vfc","vfd","wae","wao","war","wau","wda","wdc","wdd","wde","wfa","wfd","wfo","wmt","wpb","wpc","wpo","wpp","wps","wrc","wrt","wse","wtp", "asc", "adc"] as const

// prettier-ignore
const FONDAMENTALES_IDS = ["abd", "aca","aco","and","anf","def","dex", "dpu","ihi", "mfr","mod","rca","rcb","rcd","rcm","rco","rcs","rif","rmc", "rpu","sco"] as const satisfies Readonly<EtapeTypeIdFondamentaleArray>

// prettier-ignore
const NON_FONDAMENTALES_IDS = ["acf","apd","ape","apo","app","apu","cac","css","dae","ddc","des","dma","epu","exp","ipc", "mca","mcb","mcd","mci", "mcm","mco","mcp","mcr", "mcs","mec","men","meo","mie","mif","mim","mna","mnb","mnc","mnd","mni","mno","mns","mnv","mpb", "ncl","pfc","pfd","ppu","pqr","rde","rno", "rpe","sas","sca","scg","spe","spo","spp","vfc","vfd","wae","wao","war","wau","wda","wdc","wdd","wde","wfa","wfd","wfo","wmt","wpb","wpc","wpo","wpp","wps","wrc","wrt","wse","wtp", "asc", "adc"] as const satisfies Readonly<EtapeTypeIdNonFondamentale[]>

// Ceci est un test :)
;[...FONDAMENTALES_IDS, ...NON_FONDAMENTALES_IDS] as const satisfies typeof IDS

export const ETAPES_TYPES = {
  abrogationDeLaDecision: 'abd',
  avisDeLaCommissionDesAutorisationsDeRecherchesMinieres_CARM_: 'aca',
  avenantALautorisationDeRechercheMiniere: 'aco',
  decisionDuJugeAdministratif: 'and',
  avisDeMiseEnConcurrenceAuJORF: 'anf',
  rapportEtAvisDeLaDREAL: 'apd',
  avisDuConseilDEtat: 'ape',
  avisDeLaCommissionDepartementaleDesMines_CDM_: 'apo',
  avisDuPrefet: 'app',
  publicationDeLavisDeDecisionImplicite: 'apu',
  consultationDesAdministrationsCentrales: 'cac',
  classementSansSuite: 'css',
  decisionDeLaMissionAutoriteEnvironnementale_ExamenAuCasParCasDuProjet_: 'dae',
  demandeDeConsentement: 'ddc',
  decisionDeLOfficeNationalDesForets: 'def',
  desistementDuDemandeur: 'des',
  decisionDeLAutoriteAdministrative: 'dex',
  publicationDeDecisionAuJORF: 'dpu',
  enquetePublique: 'epu',
  expertises: 'exp',
  informationsHistoriquesIncompletes: 'ihi',
  demandeDeComplements_RecevabiliteDeLaDemande_: 'mca',
  demandeDeComplements_RecepisseDeDeclarationLoiSurLeau_: 'mcb',
  demandeDeComplements_DecisionDeLaMissionAutoriteEnvironnementale_ExamenAuCasParCasDuProjet_: 'mcd',
  demandeDeComplements_CompletudeDeLaDemande_: 'mcm',
  demandeDeComplements: 'mco',
  completudeDeLaDemande: 'mcp',
  recevabiliteDeLaDemande: 'mcr',
  demandeDeComplements_SaisineDeLaCARM_: 'mcs',
  avisDeDemandeConcurrente: 'mec',
  enregistrementDeLaDemande: 'men',
  priseEnChargeParLOfficeNationalDesForets: 'meo',
  demande: 'mfr',
  demandeDinformations_AvisDuDREALDEALOuDGTM_: 'mie',
  demandeDinformations: 'mif',
  demandeDinformations_RecevabiliteDeLaDemande_: 'mim',
  notificationAuDemandeur_AjournementDeLaCARM_: 'mna',
  notificationAuDemandeur_AvisFavorableDeLaCARM_: 'mnb',
  notificationAuDemandeur_ClassementSansSuite_: 'mnc',
  notificationAuDemandeur_AvisDefavorable_: 'mnd',
  notificationAuDemandeur_InitiationDeLaDemarcheDeRetrait_: 'mni',
  notificationAuDemandeur: 'mno',
  notificationAuDemandeur_SignatureDeLautorisationDeRechercheMiniere_: 'mns',
  notificationAuDemandeur_SignatureDeLavenantALautorisationDeRechercheMiniere_: 'mnv',
  modificationDeLaDemande: 'mod',
  notificationDesCollectivitesLocales: 'ncl',
  paiementDesFraisDeDossierComplementaires: 'pfc',
  paiementDesFraisDeDossier: 'pfd',
  consultationDuPublic: 'ppu',
  publicationDansUnJournalLocalOuNational: 'pqr',
  receptionDeComplements_RecevabiliteDeLaDemande_: 'rca',
  receptionDeComplements_RecepisseDeDeclarationLoiSurLeau_: 'rcb',
  receptionDeComplements_DecisionDeLaMissionAutoriteEnvironnementale_ExamenAuCasParCasDuProjet__: 'rcd',
  receptionDeComplements_CompletudeDeLaDemande_: 'rcm',
  receptionDeComplements: 'rco',
  receptionDeComplements_SaisineDeLaCARM_: 'rcs',
  recepisseDeDeclarationLoiSurLeau: 'rde',
  receptionDinformation: 'rif',
  reponseDuDemandeur: 'rno',
  rapportDuConseilDEtat: 'rpe',
  publicationDeDecisionAuRecueilDesActesAdministratifs: 'rpu',
  saisineDeLautoriteSignataire: 'sas',
  saisineDeLaCommissionDesAutorisationsDeRecherchesMinieres_CARM_: 'sca',
  saisinesEtAvisCGE_AE: 'scg',
  resultatMiseEnConcurrence: 'rmc',
  signatureDeLautorisationDeRechercheMiniere: 'sco',
  saisineDuConseilDEtat: 'spe',
  saisineDeLaCommissionDepartementaleDesMines_CDM_: 'spo',
  saisineDuPrefet: 'spp',
  validationDuPaiementDesFraisDeDossierComplementaires: 'vfc',
  validationDuPaiementDesFraisDeDossier: 'vfd',
  avisDeLautoriteEnvironnementale: 'wae',
  arreteDouvertureDesTravauxMiniers: 'wao',
  avisDeReception: 'war',
  avisDuDemandeurSurLesPrescriptionsProposees: 'wau',
  donneActeDeLaDeclaration_DOTM_: 'wda',
  demandeDeComplements_AOTMOuDOTM_: 'wdc',
  depotDeLaDemande_wdd: 'wdd',
  demandeDeComplements_DADT_: 'wde',
  demandeDautorisationDouvertureDeTravauxMiniers_AOTM_: 'wfa',
  declarationDarretDefinitifDeTravaux_DADT_: 'wfd',
  declarationDouvertureDeTravauxMiniers_DOTM_: 'wfo',
  memoireDeFinDeTravaux: 'wmt',
  porterAConnaissance: 'wpb',
  arreteDePrescriptionsComplementaires: 'wpc',
  arreteDeSecondDonnerActe: 'wpo',
  arretePrefectoralDePremierDonnerActe_DADT_: 'wpp',
  arretePrefectoralDeSursisAStatuer: 'wps',
  receptionDeComplements_wrc: 'wrc',
  recolement: 'wrt',
  saisineDeLautoriteEnvironnementale: 'wse',
  transmissionDuProjetDePrescriptionsAuDemandeur: 'wtp',
  avisDesServicesEtCommissionsConsultatives: 'asc',
  avisDesCollectivites: 'adc',
  declarationDIrrecevabilite: 'mci',
  attestationDeConstitutionDeGarantiesFinancieres: 'acf',
  demandeDeModificationAES: 'dma',
  mesuresDePublicite: 'mpb',
  informationDuPrefetEtDesCollectivites: 'ipc',
} as const satisfies Record<string, EtapeTypeId>

export const etapeTypeIdValidator = z.enum(IDS)
export const etapeTypeIdFondamentaleValidator = z.enum(FONDAMENTALES_IDS)
export const etapeTypeIdNonFondamentaleValidator = z.enum(NON_FONDAMENTALES_IDS)

export const isEtapeTypeIdFondamentale = (e: EtapeTypeId): e is EtapeTypeIdFondamentale => {
  return etapeTypeIdFondamentaleValidator.safeParse(e).success
}

export type EtapeTypeId = z.infer<typeof etapeTypeIdValidator>

export type EtapeType<T = EtapeTypeId> = Definition<T> & { unique: boolean; fondamentale: boolean; date_fin: CaminoDate | null; public_lecture: boolean; entreprises_lecture: boolean }
export const EtapesTypes = {
  ipc: {
    id: 'ipc',
    nom: 'information du préfet et des collectivités',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  rcs: {
    id: 'rcs',
    nom: 'réponse à la demande de compléments (saisine de la CARM)',
    description: 'après une sca',
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  wpb: {
    id: 'wpb',
    nom: 'porter-à-connaissance',
    description: 'Porter-à-connaissance',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  rcb: {
    id: 'rcb',
    nom: "réponse à la demande de compléments (récépissé de déclaration loi sur l'eau)",
    description: 'après une rde',
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  mcs: {
    id: 'mcs',
    nom: 'demande de compléments (saisine de la CARM)',
    description: 'après une sca',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  rca: {
    id: 'rca',
    nom: 'réponse à la demande de compléments (recevabilité de la demande)',
    description: 'après une mcr',
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  rcd: {
    id: 'rcd',
    nom: 'réponse à la demande de compléments (décision de la mission autorité environnementale (examen au cas par cas du projet))',
    description: 'après une dae',
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  rco: {
    id: 'rco',
    nom: 'réponse à la demande de compléments',
    description:
      'Étape par laquelle le service instructeur accuse réception des compléments transmis par le demandeur. Cette étape permet la reprise du décompte du délai dans lequel une décision doit être prise.',
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  anf: {
    id: 'anf',
    nom: 'avis de mise en concurrence au JORF',
    description:
      "Pour les titres miniers H (hydrocarbures), étape par laquelle le ministre chargé des mines publie l'avis de mise en concurrence au Journal officiel de la République française au frais du demandeur. Pour les titres miniers M (métaux), étape par laquelle le préfet publie l'avis de mise en concurrence au Journal officiel de la République française au frais du demandeur.",
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  mim: {
    id: 'mim',
    nom: "demande d'informations (recevabilité de la demande)",
    description: 'après une mcr',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  mie: {
    id: 'mie',
    nom: "demande d'informations (avis du DREAL, DEAL ou DGTM)",
    description: 'après une adp',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  cac: {
    id: 'cac',
    nom: 'consultation des administrations centrales',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  rcm: {
    id: 'rcm',
    nom: 'réponse à la demande de compléments (complétude de la demande)',
    description: 'après une mcp',
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  mcd: {
    id: 'mcd',
    nom: 'demande de compléments (décision de la mission autorité environnementale (examen au cas par cas du projet)',
    description: 'après une dae',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  scg: {
    id: 'scg',
    nom: "saisines et avis du conseil général de l'économie (CGE) et de l'autorité environnementale (AE)",
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  sas: {
    id: 'sas',
    nom: "saisine de l'autorité signataire",
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wda: {
    id: 'wda',
    nom: 'Donné acte de la déclaration (DOTM)',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  epu: {
    id: 'epu',
    nom: 'Enquête publique',
    description:
      "L'enquête publique requise pour l'octroi des concessions minières et des permis d'exploitations dans les départements d'outre-mer a lieu après constat de recevabilité du dossier. L'enquête publique dure un mois mais il est possible de la prolonger.",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  rpe: {
    id: 'rpe',
    nom: "rapport du Conseil d'État",
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  aca: {
    id: 'aca',
    nom: 'avis de la commission des autorisations de recherches minières (CARM)',
    description: '',
    fondamentale: true,
    unique: true,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  apo: {
    id: 'apo',
    nom: 'avis de la commission départementale des mines (CDM)',
    description: "L'avis de cette commission est consultatif et donne lieu à des votes des différents membres des collèges. Les représentants des administrations participent également.",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  css: {
    id: 'css',
    nom: 'classement sans suite',
    description: '',
    fondamentale: false,
    unique: true,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  sca: {
    id: 'sca',
    nom: 'saisine de la commission des autorisations de recherches minières (CARM)',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  meo: {
    id: 'meo',
    nom: "prise en charge par l'Office national des forêts",
    description: '',
    fondamentale: false,
    unique: true,
    date_fin: toCaminoDate('2020-01-01'),
    public_lecture: false,
    entreprises_lecture: true,
  },
  mna: {
    id: 'mna',
    nom: 'notification au demandeur (ajournement de la CARM)',
    description: 'après une aca ajournée',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  pqr: {
    id: 'pqr',
    nom: 'publication dans un journal local ou national',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  sco: {
    id: 'sco',
    nom: "signature de l'autorisation de recherche minière",
    description:
      "Signature de l'autorisation par l'ONF seulement dans le cas d'ARM manuelles ou par les deux parties (ONF et demandeur) dans le cas d'ARM mécanisées pour acceptation de l'état des lieux par le demandeur et valant début de validité de celle-ci.",
    fondamentale: true,
    unique: true,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  spp: {
    id: 'spp',
    nom: 'saisine du préfet',
    description: "L'administration centrale transmet la demande au préfet pour qu'il engage son instruction locale.\n",
    fondamentale: false,
    unique: true,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  dex: {
    id: 'dex',
    nom: "décision de l'autorité administrative",
    description: '',
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  and: {
    id: 'and',
    nom: 'décision du juge administratif',
    description: 'Décision à un contentieux ou à un référé.',
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  mni: {
    id: 'mni',
    nom: 'notification au demandeur (initiation de la démarche de retrait)',
    description: 'après une ide',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  ihi: {
    id: 'ihi',
    nom: 'informations historiques incomplètes',
    description: 'Impossibilité de retracer la totalité de la vie administrative du titre.',
    fondamentale: true,
    unique: true,
    date_fin: toCaminoDate('2018-12-31'),
    public_lecture: true,
    entreprises_lecture: true,
  },
  mns: {
    id: 'mns',
    nom: "notification au demandeur (signature de l'autorisation de recherche minière)",
    description: 'après une sco',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  mnc: {
    id: 'mnc',
    nom: 'notification au demandeur (classement sans suite)',
    description: 'après une css',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  aco: {
    id: 'aco',
    nom: "avenant à l'autorisation de recherche minière",
    description: "Document (courrier) validant une demande de renouvellement et actant la prolongation de la durée de l'autorisation.",
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  mnd: {
    id: 'mnd',
    nom: 'notification au demandeur (avis défavorable)',
    description: 'après une aca défavorable',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  rpu: {
    id: 'rpu',
    nom: 'publication de décision au recueil des actes administratifs',
    description: 'Publication, par le préfet, au recueil des actes administratifs.',
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  ncl: {
    id: 'ncl',
    nom: 'notification des collectivités locales',
    description: 'Notification de la décision aux collectivités locales.',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  wrt: {
    id: 'wrt',
    nom: 'récolement',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  spe: {
    id: 'spe',
    nom: "saisine du Conseil d'Etat",
    description: "Le service instructeur transmet la demande, les différents avis et une proposition de décision du premier ministre et du ministre chargé des mines au conseil d'État pour avis.",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  abd: {
    id: 'abd',
    nom: 'abrogation de la décision',
    description: "Suppression de l'acte administratif pour l'avenir.",
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  des: {
    id: 'des',
    nom: 'désistement du demandeur',
    description:
      "Le demandeur signifie son souhait de ne pas donner suite à sa demande. Le désistement met fin à la démarche. Le demandeur ne peut pas se désister de sa demande après qu'une décision ait été rendue sur celle-ci.",
    fondamentale: false,
    unique: true,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  apu: {
    id: 'apu',
    nom: "publication de l'avis de décision implicite",
    description:
      "Le principe adopté par la réglementation actuelle veut que le silence de l'administration pendant un délai fixé vaut accord implicite. Toutefois il existe des décisions implicites de rejets pour certaines catégories de décisions. En particulier pour les titres miniers, l'octroi, la prolongation et la mutation. Les décisions de rejet ne sont pas publiées, les décisions d'acceptation doivent faire l'objet d'un avis au JORF confirmant que cette dernière est acquise. ",
    fondamentale: false,
    unique: true,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  app: {
    id: 'app',
    nom: 'avis du préfet',
    description:
      "Le préfet émet un avis à l'issue de l'instruction du dossier, prenant en compte les avis délivrés par les services déconcentrés et éventuellement, l'avis de la commission des mines en Guyane.",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  mcp: {
    id: 'mcp',
    nom: 'complétude de la demande',
    description:
      "Phase d'examen qui permet d'établir que toutes les pièces exigées par la réglementation sont bien fournies par le demandeur. Cet examen ne préjuge pas de la décision qui sera apportée par l'administration. ",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  ppu: {
    id: 'ppu',
    nom: 'consultation du public',
    description:
      "La consultation du public est obligatoire pour l'octroi des permis exclusifs de recherches (PER) et les prolongations de concessions minières. La législation ne prévoit pas à quel moment elle intervient. Il a été décidé pour les titres miniers qu'elle aurait lieu dès la déclaration de recevabilité du dossier. ",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  mod: {
    id: 'mod',
    nom: 'modification de la demande',
    description:
      "Le porteur modifie les caractéristiques fondamentales de sa demande. Cette modification ne change pas le délai de décision implicite. Les possibilités sont limitées au gré de l'avancement de la démarche, en particulier après la mise en concurrence et après la consultation du public.\n",
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  rif: {
    id: 'rif',
    nom: "réception d'information",
    description: "Étape par laquelle le service instructeur reçoit l'information demandée.",
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  spo: {
    id: 'spo',
    nom: 'saisine de la commission départementale des mines (CDM)',
    description:
      "Dans les seuls départements d'Outre-mer (Guadeloupe, Guyane, Réunion, Martinique et Mayotte) une commission dédiée à l'examen des demandes minières se réunit comportant les différentes parties prenantes. Cette commission donne un avis consultatif sur le rapport des services déconcentrés. Pour l'instant, cette commission se réunit uniquement en Guyane.",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  men: {
    id: 'men',
    nom: 'enregistrement de la demande',
    description: "L'enregistrement de la demande est une étape de gestion administrative interne du dossier qui se confond au dépôt de la demande lorsqu'elle est numérique et dématérialisée.",
    fondamentale: false,
    unique: true,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  mif: {
    id: 'mif',
    nom: "demande d'informations",
    description: "Étape à l'initiative du service instructeur pour obtenir des informations qui ne sont pas de nature à interrompre le décompte du délai dans lequel une décision doit être prise. \n",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  mec: {
    id: 'mec',
    nom: 'avis de demande concurrente',
    description: "Annonce à un demandeur qu'une demande concurrente à la sienne a été déposée.",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  mnv: {
    id: 'mnv',
    nom: "notification au demandeur (signature de l'avenant à l'autorisation de recherche minière)",
    description: 'après une aco',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  mnb: {
    id: 'mnb',
    nom: 'notification au demandeur (avis favorable de la CARM)',
    description: 'après une aca favorable',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  apd: {
    id: 'apd',
    nom: 'Rapport et avis de la DREAL',
    description:
      "Étape par laquelle le DREAL ou le DEAL rend un avis sur la demande. Cet avis tient compte du rapport rédigé par le service instructeur de la DREAL ou de la DEAL qui s'appuie sur les avis émis par les chefs des services civils et de l'autorité militaire intéressés sur la demande.",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wpo: {
    id: 'wpo',
    nom: 'Arrêté de second donner acte',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wdc: {
    id: 'wdc',
    nom: 'demande de compléments (AOTM ou DOTM)',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  ape: {
    id: 'ape',
    nom: "avis du Conseil d'Etat",
    description:
      "Le conseil d'État donne au premier ministre et au ministre chargé des mines un avis sur la demande et sur son instruction. Cet avis n'est pas conforme mais si la décision s'écarte de l'avis donné, elle doit être validée en conseil des ministres.",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  vfc: {
    id: 'vfc',
    nom: 'validation du paiement des frais de dossier complémentaires',
    description: 'Validation par la direction générale des territoires et des mines (DGTM).',
    fondamentale: false,
    unique: true,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  def: {
    id: 'def',
    nom: "décision de l'Office national des forêts",
    description: "Décision de l'ONF dans le cadre des autorisations d'exploitation (AEX) et des autorisations de recherches minières (ARM) en tant que propriétaire du sol.",
    fondamentale: true,
    unique: true,
    date_fin: toCaminoDate('2018-10-22'),
    public_lecture: true,
    entreprises_lecture: true,
  },
  mcm: {
    id: 'mcm',
    nom: 'demande de compléments (complétude de la demande)',
    description: 'après une mcp',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  exp: {
    id: 'exp',
    nom: 'expertises DREAL ou DGTM',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  pfd: {
    id: 'pfd',
    nom: 'paiement des frais de dossier',
    description: '',
    fondamentale: false,
    unique: true,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  vfd: {
    id: 'vfd',
    nom: 'validation du paiement des frais de dossier',
    description: "Acte attestant que le pétitionnaire d'une autorisation de recherches minières (ARM) a acquitté ses frais de dossier.\n",
    fondamentale: false,
    unique: true,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  mcb: {
    id: 'mcb',
    nom: "demande de compléments (récépissé de déclaration loi sur l'eau)",
    description: 'après une rde',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  mno: {
    id: 'mno',
    nom: 'notification au demandeur',
    description: 'Notification de la décision au demandeur.',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  rno: {
    id: 'rno',
    nom: 'réponse du demandeur',
    description: 'après une notification au demandeur',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  mcr: {
    id: 'mcr',
    nom: 'recevabilité de la demande',
    description:
      "Acte de l'administration établissant que l'examen de complétude a été mené à bien. Le constat de recevabilité donne lieu à un rapport de la part de l'administration qui constate que le dossier de demande est complet. ",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  wao: {
    id: 'wao',
    nom: "arrêté d'ouverture des travaux miniers",
    description: "Arrêté d'ouverture des travaux miniers",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  dae: {
    id: 'dae',
    nom: 'décision de la mission autorité environnementale (examen au cas par cas du projet)',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  pfc: {
    id: 'pfc',
    nom: 'paiement des frais de dossier complémentaires',
    description: 'Supplément de frais de dossier concernant uniquement les ARM mécanisées car elles nécessitent une expertise complémentaire (état des lieux par imagerie).\n',
    fondamentale: false,
    unique: true,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: true,
  },
  mca: {
    id: 'mca',
    nom: 'demande de compléments (recevabilité de la demande)',
    description: 'après une mcr',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  dpu: {
    id: 'dpu',
    nom: 'publication de décision au JORF',
    description:
      "En dehors des refus et décisions implicites de rejet, la décision de l'autorité compétente est publiée soit au recueil administratif de la préfecture soit au JORF pour les décisions du préfet et au JORF pour celle du ministre chargé des mines et du gouvernement. Ces publications pour les actes relatifs aux titres miniers sont publiés par extrait.",
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  mco: {
    id: 'mco',
    nom: 'demande de compléments',
    description:
      "Étape à l'initiative du service instructeur qui demande des compléments au demandeur et lui fixe un délai maximum pour les transmettre. Cette étape interrompt le décompte du délai dans lequel une décision doit être prise. Cette demande doit faire l'objet d'une lettre recommandée avec demande d'accusé de réception.",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  mfr: {
    id: 'mfr',
    nom: 'demande',
    description:
      "La demande caractérise le projet minier porté par le demandeur. Elle inclut les caractéristiques fondamentales du titre ou de l'autorisation dans une lettre appuyée par un dossier et les justifications des capacités techniques et financières. Les informations relevant du secret des affaires incluses à la demande peuvent ne pas être rendues publiques à la demande du porteur de projet.",
    fondamentale: true,
    unique: true,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  wfa: {
    id: 'wfa',
    nom: "Demande d'autorisation d'ouverture de travaux miniers (AOTM)",
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wfo: {
    id: 'wfo',
    nom: "Déclaration d'ouverture de travaux miniers (DOTM)",
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wfd: {
    id: 'wfd',
    nom: "Déclaration d'arrêt définitif de travaux (DADT)",
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wdd: {
    id: 'wdd',
    nom: 'dépot de la demande',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wde: {
    id: 'wde',
    nom: 'demande de compléments (DADT)',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  wrc: {
    id: 'wrc',
    nom: 'reception de compléments',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  war: {
    id: 'war',
    nom: 'avis de réception',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wse: {
    id: 'wse',
    nom: "saisine de l'autorité environnementale",
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wae: {
    id: 'wae',
    nom: "avis de l'autorité environnementale",
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wps: {
    id: 'wps',
    nom: 'Arrêté préfectoral de sursis à statuer',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wtp: {
    id: 'wtp',
    nom: 'Transmission du projet de prescriptions au demandeur',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wau: {
    id: 'wau',
    nom: 'Avis du demandeur sur les prescriptions proposées',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wpp: {
    id: 'wpp',
    nom: 'Arrêté préfectoral de premier donner acte (DADT)',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wpc: {
    id: 'wpc',
    nom: 'Arrêté de prescriptions complémentaires',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  wmt: {
    id: 'wmt',
    nom: 'memoire de fin de travaux',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  rde: {
    id: 'rde',
    nom: "récépissé de déclaration loi sur l'eau",
    description: "Document nécessaire à l'utilisation de l'eau dans le cas d'une autorisation de recherches minières (ARM).",
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  asc: {
    id: 'asc',
    nom: 'Avis des services et commissions consultatives',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  adc: {
    id: 'adc',
    nom: 'Avis des collectivités',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  mci: {
    id: 'mci',
    nom: "Déclaration d'irrecevabilité",
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  acf: {
    id: 'acf',
    nom: 'Attestation de constitution de garanties financières',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  rmc: {
    id: 'rmc',
    nom: 'résultat final de la mise en concurrence',
    description: '',
    fondamentale: true,
    unique: false,
    date_fin: null,
    public_lecture: true,
    entreprises_lecture: true,
  },
  dma: {
    id: 'dma',
    nom: "demande de modification de l'AES",
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  mpb: {
    id: 'mpb',
    nom: 'mesures de publicité',
    description: '',
    fondamentale: false,
    unique: false,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
  ddc: {
    id: 'ddc',
    nom: 'demande de consentement',
    description: '',
    fondamentale: false,
    unique: true,
    date_fin: null,
    public_lecture: false,
    entreprises_lecture: false,
  },
} as const satisfies { [key in EtapeTypeId]: EtapeType<key> }

type IsFondamentale<T> = T extends EtapeTypeId ? ((typeof EtapesTypes)[T] extends { fondamentale: true } ? T : never) : never

type EtapeFondamentaleIdFinder<Etapes> = Etapes extends readonly [infer First, ...infer Rest]
  ? First extends IsFondamentale<First>
    ? [First, ...EtapeFondamentaleIdFinder<Rest>]
    : EtapeFondamentaleIdFinder<Rest>
  : []

type EtapeTypeIdFondamentaleArray = EtapeFondamentaleIdFinder<typeof IDS>
type EtapeTypeIdFondamentale = EtapeFondamentaleIdFinder<typeof IDS>[number]
type EtapeTypeIdNonFondamentale = Exclude<EtapeTypeId, EtapeTypeIdFondamentale>

export const isEtapeTypeId = (etapeTypeId: string): etapeTypeId is EtapeTypeId => {
  return etapeTypeIdValidator.safeParse(etapeTypeId).success
}

export const etapesTypes = Object.values(EtapesTypes)

const ETAPES_DECISIONS_IDS = [
  ETAPES_TYPES.publicationDeDecisionAuJORF,
  ETAPES_TYPES.publicationDeDecisionAuRecueilDesActesAdministratifs,
  ETAPES_TYPES.decisionDeLAutoriteAdministrative,
  ETAPES_TYPES.decisionDeLOfficeNationalDesForets,
  ETAPES_TYPES.signatureDeLautorisationDeRechercheMiniere,
  ETAPES_TYPES.avenantALautorisationDeRechercheMiniere,
] as const satisfies Readonly<EtapeTypeId[]>

const ETAPES_BROUILLONS_IDS = [
  ETAPES_TYPES.demande,
  ETAPES_TYPES.avisDesServicesEtCommissionsConsultatives,
  ETAPES_TYPES.consultationDuPublic,
  ETAPES_TYPES.receptionDinformation,
  ETAPES_TYPES.consultationDesAdministrationsCentrales,
  ETAPES_TYPES.receptionDeComplements,
  ETAPES_TYPES.reponseDuDemandeur,
  ETAPES_TYPES.saisinesEtAvisCGE_AE,
  ETAPES_TYPES.avisDesCollectivites,
] as const satisfies Readonly<EtapeTypeId[]>

export const isEtapeDecision = (etapeTypeId: EtapeTypeId): boolean => ETAPES_DECISIONS_IDS.includes(etapeTypeId)
export const canBeBrouillon = (etapeTypeId: EtapeTypeId): EtapeBrouillon => ETAPES_BROUILLONS_IDS.includes(etapeTypeId) as EtapeBrouillon

export const ETAPES_WITH_AUTOMATIC_STATUTS = [ETAPES_TYPES.consultationDuPublic, ETAPES_TYPES.enquetePublique, ETAPES_TYPES.avisDeMiseEnConcurrenceAuJORF] as const satisfies Readonly<EtapeTypeId[]>

export const isEtapeWithAutomaticStatuts = (etapeTypeId: EtapeTypeId | null): etapeTypeId is (typeof ETAPES_WITH_AUTOMATIC_STATUTS)[number] =>
  etapeTypeId !== null && ETAPES_WITH_AUTOMATIC_STATUTS.includes(etapeTypeId)

export const ETAPE_TYPE_FOR_CONCURRENCY_DATA = ETAPES_TYPES.demande
export const ETAPE_TYPE_FOR_CONSENTEMENT_DATA = ETAPES_TYPES.demande

const etapeVisibiliteValidator = z.enum(['Publique', 'Entreprise et administrations', 'Confidentielle'])
export type EtapeVisibilite = z.infer<typeof etapeVisibiliteValidator>
