import { describe, test, expect } from 'vitest'
import { getAvisTypes } from './avisTypes'
import { firstEtapeDateValidator } from './date'
import { demarcheIdValidator } from './demarche'
import { communeIdValidator } from './static/communes'
import { DEMARCHES_TYPES_IDS } from './static/demarchesTypes'
import { ETAPES_TYPES } from './static/etapesTypes'
import { IS_ARM_MECANISE } from './static/mecanise'
import { TITRES_TYPES_IDS } from './static/titresTypes'
import { MachineInfo } from './machines'

describe('getAvisType', () => {
  const demarcheId = demarcheIdValidator.parse('demarcheId')
  const firstEtapeDate = firstEtapeDateValidator.parse('2022-01-01')

  test("les avis obligatoires deviennent optionnels si on n'a pas de machine", () => {
    const machineInfo = MachineInfo.withDate(TITRES_TYPES_IDS.AUTORISATION_D_EXPLOITATION_METAUX, DEMARCHES_TYPES_IDS.Octroi, demarcheId, firstEtapeDateValidator.parse('1980-11-01'))
    const avis = getAvisTypes(ETAPES_TYPES.demande, machineInfo, [], IS_ARM_MECANISE)
    expect(avis).toMatchInlineSnapshot(`
      {
        "avisDeLaMissionAutoriteEnvironnementale": {
          "id": "avisDeLaMissionAutoriteEnvironnementale",
          "optionnel": true,
        },
        "avisProprietaireDuSol": {
          "id": "avisProprietaireDuSol",
          "optionnel": true,
        },
      }
    `)
  })

  test("avis de l'office national des forêts est optionnel pour la procédure spécifique", () => {
    const machineInfo = MachineInfo.withDate(TITRES_TYPES_IDS.AUTORISATION_DE_RECHERCHE_METAUX, DEMARCHES_TYPES_IDS.Octroi, demarcheId, firstEtapeDateValidator.parse('2024-11-01'))
    expect(machineInfo.machineId).toBe('ProcedureSpecifique')
    const avis = getAvisTypes(ETAPES_TYPES.avisDesServicesEtCommissionsConsultatives, machineInfo, [], IS_ARM_MECANISE)
    expect(avis.avisOfficeNationalDesForets?.optionnel).toBe(true)
  })
  test("avis au cas par cas et avis du propriétaire du sol sont obligatoires pour les demandes d'ARM mécanisée de la procédure spécifique", () => {
    const machineInfo = MachineInfo.withDate(TITRES_TYPES_IDS.AUTORISATION_DE_RECHERCHE_METAUX, DEMARCHES_TYPES_IDS.Octroi, demarcheId, firstEtapeDate)
    expect(getAvisTypes(ETAPES_TYPES.demande, machineInfo, [], IS_ARM_MECANISE)).toMatchInlineSnapshot(`{}`)
    expect(
      getAvisTypes(
        ETAPES_TYPES.demande,
        MachineInfo.withDate(TITRES_TYPES_IDS.AUTORISATION_DE_RECHERCHE_METAUX, DEMARCHES_TYPES_IDS.Octroi, demarcheId, firstEtapeDateValidator.parse('2025-01-01')),
        [],
        IS_ARM_MECANISE
      )
    ).toMatchInlineSnapshot(`
      {
        "avisCasParCas": {
          "id": "avisCasParCas",
          "optionnel": false,
        },
        "avisProprietaireDuSol": {
          "id": "avisProprietaireDuSol",
          "optionnel": false,
        },
      }
    `)
  })
  test('les avis CGE et AE sont obligatoires pour la machine spécifique', () => {
    let machineInfo = MachineInfo.withDate(TITRES_TYPES_IDS.AUTORISATION_DE_RECHERCHE_METAUX, DEMARCHES_TYPES_IDS.Octroi, demarcheId, firstEtapeDate)
    expect(getAvisTypes(ETAPES_TYPES.saisinesEtAvisCGE_AE, machineInfo, [], IS_ARM_MECANISE)).toMatchInlineSnapshot(`
      {
        "autreAvis": {
          "id": "autreAvis",
          "optionnel": true,
        },
        "avisDeLaMissionAutoriteEnvironnementale": {
          "id": "avisDeLaMissionAutoriteEnvironnementale",
          "optionnel": true,
        },
        "avisDuConseilGeneralDeLEconomie": {
          "id": "avisDuConseilGeneralDeLEconomie",
          "optionnel": true,
        },
      }
    `)
    machineInfo = MachineInfo.withDate(TITRES_TYPES_IDS.AUTORISATION_DE_RECHERCHE_METAUX, DEMARCHES_TYPES_IDS.Octroi, demarcheId, firstEtapeDateValidator.parse('2025-01-01'))
    expect(getAvisTypes(ETAPES_TYPES.saisinesEtAvisCGE_AE, machineInfo, [], IS_ARM_MECANISE)).toMatchInlineSnapshot(`
      {
        "autreAvis": {
          "id": "autreAvis",
          "optionnel": true,
        },
        "avisDeLaMissionAutoriteEnvironnementale": {
          "id": "avisDeLaMissionAutoriteEnvironnementale",
          "optionnel": false,
        },
        "avisDuConseilGeneralDeLEconomie": {
          "id": "avisDuConseilGeneralDeLEconomie",
          "optionnel": false,
        },
      }
    `)
  })
  test('getAvisType', () => {
    let machineInfo = MachineInfo.withDate(TITRES_TYPES_IDS.AUTORISATION_DE_RECHERCHE_METAUX, DEMARCHES_TYPES_IDS.Octroi, demarcheId, firstEtapeDate)
    expect(getAvisTypes(ETAPES_TYPES.avisDesCollectivites, machineInfo, [], IS_ARM_MECANISE)).toMatchInlineSnapshot(`
        {
          "avisDUneCollectivite": {
            "id": "avisDUneCollectivite",
            "optionnel": true,
          },
        }
      `)
    expect(getAvisTypes(ETAPES_TYPES.demande, machineInfo, [], IS_ARM_MECANISE)).toMatchInlineSnapshot(`{}`)
    expect(getAvisTypes(ETAPES_TYPES.avisDesServicesEtCommissionsConsultatives, machineInfo, [], IS_ARM_MECANISE)).toMatchInlineSnapshot(`
      {
        "autreAvis": {
          "id": "autreAvis",
          "optionnel": true,
        },
        "avisAgenceRegionaleSante": {
          "id": "avisAgenceRegionaleSante",
          "optionnel": true,
        },
        "avisAutoriteMilitaire": {
          "id": "avisAutoriteMilitaire",
          "optionnel": true,
        },
        "avisCaisseGeneraleSecuriteSociale": {
          "id": "avisCaisseGeneraleSecuriteSociale",
          "optionnel": true,
        },
        "avisConseilDepartementalEnvironnementRisquesSanitairesTechnologiques": {
          "id": "avisConseilDepartementalEnvironnementRisquesSanitairesTechnologiques",
          "optionnel": true,
        },
        "avisDirectionDepartementaleTerritoiresMer": {
          "id": "avisDirectionDepartementaleTerritoiresMer",
          "optionnel": true,
        },
        "avisDirectionRegionaleDesAffairesCulturelles": {
          "id": "avisDirectionRegionaleDesAffairesCulturelles",
          "optionnel": true,
        },
        "avisDirectionRegionaleFinancesPubliques": {
          "id": "avisDirectionRegionaleFinancesPubliques",
          "optionnel": true,
        },
        "avisDirectionsRégionalesEconomieEmploiTravailSolidarités": {
          "id": "avisDirectionsRégionalesEconomieEmploiTravailSolidarités",
          "optionnel": true,
        },
        "avisGendarmerieNationale": {
          "id": "avisGendarmerieNationale",
          "optionnel": true,
        },
        "avisIFREMER": {
          "id": "avisIFREMER",
          "optionnel": true,
        },
        "avisInstitutNationalOrigineQualite": {
          "id": "avisInstitutNationalOrigineQualite",
          "optionnel": true,
        },
        "avisOfficeNationalDesForets": {
          "id": "avisOfficeNationalDesForets",
          "optionnel": false,
        },
        "avisParcNational": {
          "id": "avisParcNational",
          "optionnel": true,
        },
        "avisParcNaturelMarin": {
          "id": "avisParcNaturelMarin",
          "optionnel": true,
        },
        "avisServiceAdministratifLocal": {
          "id": "avisServiceAdministratifLocal",
          "optionnel": true,
        },
        "expertiseOfficeNationalDesForets": {
          "id": "expertiseOfficeNationalDesForets",
          "optionnel": true,
        },
      }
    `)
    expect(getAvisTypes(ETAPES_TYPES.avisDesServicesEtCommissionsConsultatives, machineInfo, [communeIdValidator.parse('97302')], IS_ARM_MECANISE)).toMatchInlineSnapshot(`
      {
        "autreAvis": {
          "id": "autreAvis",
          "optionnel": true,
        },
        "avisAgenceRegionaleSante": {
          "id": "avisAgenceRegionaleSante",
          "optionnel": true,
        },
        "avisAutoriteMilitaire": {
          "id": "avisAutoriteMilitaire",
          "optionnel": true,
        },
        "avisCaisseGeneraleSecuriteSociale": {
          "id": "avisCaisseGeneraleSecuriteSociale",
          "optionnel": true,
        },
        "avisConseilDepartementalEnvironnementRisquesSanitairesTechnologiques": {
          "id": "avisConseilDepartementalEnvironnementRisquesSanitairesTechnologiques",
          "optionnel": true,
        },
        "avisDirectionAlimentationAgricultureForet": {
          "id": "avisDirectionAlimentationAgricultureForet",
          "optionnel": true,
        },
        "avisDirectionDepartementaleTerritoiresMer": {
          "id": "avisDirectionDepartementaleTerritoiresMer",
          "optionnel": true,
        },
        "avisDirectionRegionaleDesAffairesCulturelles": {
          "id": "avisDirectionRegionaleDesAffairesCulturelles",
          "optionnel": true,
        },
        "avisDirectionRegionaleFinancesPubliques": {
          "id": "avisDirectionRegionaleFinancesPubliques",
          "optionnel": true,
        },
        "avisDirectionsRégionalesEconomieEmploiTravailSolidarités": {
          "id": "avisDirectionsRégionalesEconomieEmploiTravailSolidarités",
          "optionnel": true,
        },
        "avisEtatMajorOrpaillagePecheIllicite": {
          "id": "avisEtatMajorOrpaillagePecheIllicite",
          "optionnel": true,
        },
        "avisGendarmerieNationale": {
          "id": "avisGendarmerieNationale",
          "optionnel": true,
        },
        "avisIFREMER": {
          "id": "avisIFREMER",
          "optionnel": true,
        },
        "avisInstitutNationalOrigineQualite": {
          "id": "avisInstitutNationalOrigineQualite",
          "optionnel": true,
        },
        "avisOfficeNationalDesForets": {
          "id": "avisOfficeNationalDesForets",
          "optionnel": false,
        },
        "avisParcNational": {
          "id": "avisParcNational",
          "optionnel": true,
        },
        "avisParcNaturelMarin": {
          "id": "avisParcNaturelMarin",
          "optionnel": true,
        },
        "avisServiceAdministratifLocal": {
          "id": "avisServiceAdministratifLocal",
          "optionnel": true,
        },
        "expertiseOfficeNationalDesForets": {
          "id": "expertiseOfficeNationalDesForets",
          "optionnel": true,
        },
      }
    `)
    machineInfo = MachineInfo.withDate(TITRES_TYPES_IDS.AUTORISATION_D_EXPLOITATION_METAUX, DEMARCHES_TYPES_IDS.Octroi, demarcheId, firstEtapeDate)
    expect(getAvisTypes(ETAPES_TYPES.avisDesServicesEtCommissionsConsultatives, machineInfo, [communeIdValidator.parse('97302')], IS_ARM_MECANISE)).toMatchInlineSnapshot(`
      {
        "autreAvis": {
          "id": "autreAvis",
          "optionnel": true,
        },
        "avisAgenceRegionaleSante": {
          "id": "avisAgenceRegionaleSante",
          "optionnel": true,
        },
        "avisAutoriteMilitaire": {
          "id": "avisAutoriteMilitaire",
          "optionnel": true,
        },
        "avisCaisseGeneraleSecuriteSociale": {
          "id": "avisCaisseGeneraleSecuriteSociale",
          "optionnel": true,
        },
        "avisConseilDepartementalEnvironnementRisquesSanitairesTechnologiques": {
          "id": "avisConseilDepartementalEnvironnementRisquesSanitairesTechnologiques",
          "optionnel": true,
        },
        "avisDirectionAlimentationAgricultureForet": {
          "id": "avisDirectionAlimentationAgricultureForet",
          "optionnel": true,
        },
        "avisDirectionDepartementaleTerritoiresMer": {
          "id": "avisDirectionDepartementaleTerritoiresMer",
          "optionnel": true,
        },
        "avisDirectionRegionaleDesAffairesCulturelles": {
          "id": "avisDirectionRegionaleDesAffairesCulturelles",
          "optionnel": true,
        },
        "avisDirectionRegionaleFinancesPubliques": {
          "id": "avisDirectionRegionaleFinancesPubliques",
          "optionnel": true,
        },
        "avisDirectionsRégionalesEconomieEmploiTravailSolidarités": {
          "id": "avisDirectionsRégionalesEconomieEmploiTravailSolidarités",
          "optionnel": true,
        },
        "avisEtatMajorOrpaillagePecheIllicite": {
          "id": "avisEtatMajorOrpaillagePecheIllicite",
          "optionnel": true,
        },
        "avisGendarmerieNationale": {
          "id": "avisGendarmerieNationale",
          "optionnel": true,
        },
        "avisIFREMER": {
          "id": "avisIFREMER",
          "optionnel": true,
        },
        "avisInstitutNationalOrigineQualite": {
          "id": "avisInstitutNationalOrigineQualite",
          "optionnel": true,
        },
        "avisOfficeNationalDesForets": {
          "id": "avisOfficeNationalDesForets",
          "optionnel": true,
        },
        "avisParcNational": {
          "id": "avisParcNational",
          "optionnel": true,
        },
        "avisParcNaturelMarin": {
          "id": "avisParcNaturelMarin",
          "optionnel": true,
        },
        "avisServiceAdministratifLocal": {
          "id": "avisServiceAdministratifLocal",
          "optionnel": true,
        },
        "confirmationAccordProprietaireDuSol": {
          "id": "confirmationAccordProprietaireDuSol",
          "optionnel": true,
        },
        "expertiseOfficeNationalDesForets": {
          "id": "expertiseOfficeNationalDesForets",
          "optionnel": true,
        },
      }
    `)
    machineInfo = MachineInfo.withDate(TITRES_TYPES_IDS.PERMIS_EXCLUSIF_DE_RECHERCHES_GRANULATS_MARINS, DEMARCHES_TYPES_IDS.Octroi, demarcheId, firstEtapeDate)

    expect(getAvisTypes(ETAPES_TYPES.consultationDesAdministrationsCentrales, machineInfo, [], IS_ARM_MECANISE)).toMatchInlineSnapshot(`
      {
        "autreAvis": {
          "id": "autreAvis",
          "optionnel": true,
        },
      }
    `)
  })
})
