import { ITitreDemarche, IPropId, ITitreEtape, ICommune } from '../../types'

import { titreContenuTitreEtapeFind, titrePropTitreEtapeFind } from './titre-prop-etape-find'
import { newDemarcheId, newEtapeId, newTitreId } from '../../database/models/_format/id-create'
import { firstEtapeDateValidator, toCaminoDate } from 'camino-common/src/date'
import { describe, expect, test } from 'vitest'
import { TitresStatutIds } from 'camino-common/src/static/titresStatuts'
import { newEntrepriseId } from 'camino-common/src/entreprise'
import { ETAPES_TYPES } from 'camino-common/src/static/etapesTypes'
import { ETAPES_STATUTS } from 'camino-common/src/static/etapesStatuts'
import { ETAPE_IS_NOT_BROUILLON, ETAPE_IS_BROUILLON } from 'camino-common/src/etape'
import { km2Validator } from 'camino-common/src/number'
import { DEMARCHES_TYPES_IDS } from 'camino-common/src/static/demarchesTypes'
import { MachineInfo } from 'camino-common/src/machines'
import { TITRES_TYPES_IDS } from 'camino-common/src/static/titresTypes'
const currentDate = toCaminoDate('2023-04-06')

describe("id de l'étape d'une propriété valide (dé-normalise)", () => {
  test("trouve l'id de la dernière étape acceptée de la démarche d'octroi acceptée ayant la propriété 'geojson4326Perimetre'", () => {
    expect(
      titrePropTitreEtapeFind<
        Pick<
          ITitreEtape,
          | 'id'
          | 'ordre'
          | 'date'
          | 'typeId'
          | 'isBrouillon'
          | 'statutId'
          | 'titulaireIds'
          | 'amodiataireIds'
          | 'surface'
          | 'substances'
          | 'administrationsLocales'
          | 'communes'
          | 'forets'
          | 'titreDemarcheId'
          | 'geojson4326Perimetre'
        >
      >(
        currentDate,
        'points',
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1989-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'acc',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1989-oct01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1989-oct01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                ordre: 2,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1989-01-02'),
                geojson4326Perimetre: { type: 'Feature', properties: {}, geometry: { type: 'MultiPolygon', coordinates: [[[[1, 2]]]] } },
              },
              {
                id: newEtapeId('h-cx-courdemanges-1989-oct01-dex01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1989-oct01'),
                typeId: ETAPES_TYPES.decisionDeLAutoriteAdministrative,
                statutId: 'acc',
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1989-01-01'),
                ordre: 1,
              },
            ],
          },
          {
            id: newDemarcheId('h-cx-courdemanges-1989-mut01'),
            typeId: DEMARCHES_TYPES_IDS.Mutation,
            statutId: 'acc',
            ordre: 2,
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1989-mut01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1989-mut01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1989-02-03'),
                ordre: 2,
                geojson4326Perimetre: undefined,
              },
              {
                id: newEtapeId('h-cx-courdemanges-1989-mut01-dex01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1989-mut01'),
                typeId: ETAPES_TYPES.decisionDeLAutoriteAdministrative,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1989-02-02'),
                statutId: 'acc',
                ordre: 1,
              },
            ],
          },
        ],
        TitresStatutIds.Valide
      )?.id
    ).toEqual('h-cx-courdemanges-1989-oct01-dpu01')
  })

  test("ne trouve pas d'id si la dernière étape acceptée de la dernière démarche acceptée possède une propriété 'points' vide", () => {
    expect(
      titrePropTitreEtapeFind(
        currentDate,
        'points',
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1988-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'acc',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1988-oct01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1988-oct01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                ordre: 2,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1989-01-02'),
                geojson4326Perimetre: null,
              },
              {
                id: newEtapeId('h-cx-courdemanges-1988-oct01-dex01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1988-oct01'),
                typeId: ETAPES_TYPES.decisionDeLAutoriteAdministrative,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1989-01-01'),
                statutId: 'acc',
                ordre: 1,
              },
            ],
          },
        ],
        TitresStatutIds.Valide
      )
    ).toBeNull()
  })

  test("trouve l'id de la dernière étape acceptée de la démarche de mutation acceptée ayant la propriété 'points'", () => {
    expect(
      titrePropTitreEtapeFind<
        Pick<
          ITitreEtape,
          | 'id'
          | 'ordre'
          | 'date'
          | 'typeId'
          | 'isBrouillon'
          | 'statutId'
          | 'titulaireIds'
          | 'amodiataireIds'
          | 'surface'
          | 'substances'
          | 'administrationsLocales'
          | 'communes'
          | 'forets'
          | 'titreDemarcheId'
          | 'geojson4326Perimetre'
        >
      >(
        currentDate,
        'points',
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1986-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'acc',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1986-oct01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1986-oct01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1986-01-02'),
                statutId: 'acc',
                ordre: 2,
              },
              {
                id: newEtapeId('h-cx-courdemanges-1986-oct01-dex01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1986-oct01'),
                typeId: ETAPES_TYPES.decisionDeLAutoriteAdministrative,
                statutId: 'acc',
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1986-01-01'),
                ordre: 1,
              },
            ],
          },
          {
            id: newDemarcheId('h-cx-courdemanges-1986-mut01'),
            typeId: DEMARCHES_TYPES_IDS.Mutation,
            statutId: 'acc',
            ordre: 2,
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1986-mut01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1986-mut01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                ordre: 2,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1986-02-02'),

                geojson4326Perimetre: { type: 'Feature', properties: {}, geometry: { type: 'MultiPolygon', coordinates: [[[[1, 2]]]] } },
              },
              {
                id: newEtapeId('h-cx-courdemanges-1986-mut01-dex01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1986-mut01'),
                typeId: ETAPES_TYPES.decisionDeLAutoriteAdministrative,
                statutId: 'acc',
                ordre: 1,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1986-02-01'),
              },
            ],
          },
        ],
        TitresStatutIds.Valide
      )?.id
    ).toEqual('h-cx-courdemanges-1986-mut01-dpu01')
  })

  test("ne trouve pas d'id si aucune étape acceptée ne contient la propriété 'communes'", () => {
    expect(
      titrePropTitreEtapeFind(
        currentDate,
        'communes',
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1986-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'acc',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1986-oct01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1986-oct01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                ordre: 2,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1986-01-02'),
              },
              {
                id: newEtapeId('h-cx-courdemanges-1986-oct01-dex01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1986-oct01'),
                typeId: ETAPES_TYPES.decisionDeLAutoriteAdministrative,
                statutId: 'acc',
                ordre: 1,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1986-01-01'),
              },
            ],
          },
          {
            id: newDemarcheId('h-cx-courdemanges-1986-mut01'),
            typeId: DEMARCHES_TYPES_IDS.Mutation,
            statutId: 'acc',
            ordre: 2,
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1986-mut01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1986-mut01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                ordre: 2,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1986-02-02'),
              },
              {
                id: newEtapeId('h-cx-courdemanges-1986-mut01-dex01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1986-mut01'),
                typeId: ETAPES_TYPES.decisionDeLAutoriteAdministrative,
                statutId: 'acc',
                ordre: 1,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1986-02-01'),
              },
            ],
          },
        ],
        TitresStatutIds.Valide
      )
    ).toBeNull()
  })

  test("trouve l'id de la dernière étape acceptée de la dernière démarche d'octroi en instruction ayant la propriété 'points'", () => {
    expect(
      titrePropTitreEtapeFind(
        currentDate,
        'points',
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1985-mut01'),
            typeId: DEMARCHES_TYPES_IDS.Mutation,
            statutId: 'ins',
            ordre: 2,
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1985-mut01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1985-mut01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1985-01-01'),
                ordre: 2,
                geojson4326Perimetre: { type: 'Feature', properties: {}, geometry: { type: 'MultiPolygon', coordinates: [[[[1, 2]]]] } },
              },
              {
                id: newEtapeId('h-cx-courdemanges-1985-mut01-dex01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1985-mut01'),
                typeId: ETAPES_TYPES.decisionDeLAutoriteAdministrative,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1985-01-01'),
                statutId: 'acc',
                ordre: 1,
              },
            ],
          },
          {
            id: newDemarcheId('h-cx-courdemanges-1985-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'ins',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1985-oct01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1985-oct01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                ordre: 2,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1985-02-02'),
                geojson4326Perimetre: { type: 'Feature', properties: {}, geometry: { type: 'MultiPolygon', coordinates: [[[[1, 2]]]] } },
              },
              {
                id: newEtapeId('h-cx-courdemanges-1985-oct01-dex01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1985-oct01'),
                typeId: ETAPES_TYPES.decisionDeLAutoriteAdministrative,
                statutId: 'acc',
                ordre: 1,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1985-02-01'),
              },
            ],
          },
        ],
        TitresStatutIds.Valide
      )?.id
    ).toEqual('h-cx-courdemanges-1985-oct01-dpu01')
  })

  test("ne trouve pas d'id si l'étape est rejetée", () => {
    expect(
      titrePropTitreEtapeFind(
        currentDate,
        'points',
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1984-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'acc',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1984-oct01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1984-oct01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'rej',
                ordre: 2,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1984-01-01'),
                geojson4326Perimetre: { type: 'Feature', properties: {}, geometry: { type: 'MultiPolygon', coordinates: [[[[1, 2]]]] } },
              },
              {
                id: newEtapeId('h-cx-courdemanges-1984-oct01-dex01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1984-oct01'),
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1984-01-01'),
                typeId: ETAPES_TYPES.decisionDeLAutoriteAdministrative,
                statutId: 'rej',
                ordre: 1,
              },
            ],
          },
        ],
        TitresStatutIds.Valide
      )
    ).toBeNull()
  })

  test("trouve l'id de la dernière étape de formalisation de la demande de la dernière démarche d'octroi acceptée ayant la propriété 'points'", () => {
    expect(
      titrePropTitreEtapeFind(
        currentDate,
        'points',
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1983-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'acc',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1983-oct01-mfr01'),
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1983-01-01'),
                titreDemarcheId: newDemarcheId(newDemarcheId('h-cx-courdemanges-1983-oct01')),
                typeId: ETAPES_TYPES.demande,
                statutId: 'acc',
                ordre: 1,
                geojson4326Perimetre: { properties: {}, geometry: { type: 'MultiPolygon', coordinates: [[[[1, 2]]]] }, type: 'Feature' },
              } as ITitreEtape,
            ],
          },
        ],
        TitresStatutIds.Valide
      )?.id
    ).toEqual('h-cx-courdemanges-1983-oct01-mfr01')
  })

  test("ne trouve pas la dernière étape de dpu d'une démarche de prolongation ou de demande de titre en instruction car l'étape contient un périmètre et le titre a le statut 'modification en instance' et aucune phase n'est valide", () => {
    expect(
      titrePropTitreEtapeFind(
        currentDate,
        'points',
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1981-pro01'),
            typeId: DEMARCHES_TYPES_IDS.Prolongation,
            statutId: 'ins',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1981-pro01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1981-pro01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1981-01-01'),
                ordre: 1,
                geojson4326Perimetre: { type: 'Feature', properties: {}, geometry: { type: 'MultiPolygon', coordinates: [[[[1, 2]]]] } },
              },
            ],
          },
          {
            id: newDemarcheId('h-cx-courdemanges-1981-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'acc',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1981-oct01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1981-oct01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1981-01-01'),
                statutId: 'acc',
                ordre: 1,
              },
            ],
          },
        ],
        TitresStatutIds.ModificationEnInstance
      )
    ).toEqual(null)
  })

  test("ne trouve pas l'id de la dernière étape de dpu d'une démarche de prolongation ou de demande de titre en instruction car l'étape contient un périmètre et le titre a le statut 'modification en instance' mais la phase est encore valide", () => {
    expect(
      titrePropTitreEtapeFind(
        currentDate,
        'points',
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1981-pro01'),
            typeId: DEMARCHES_TYPES_IDS.Prolongation,
            statutId: 'ins',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1981-pro01-dpu01'),
                titreDemarcheId: newDemarcheId(newDemarcheId('h-cx-courdemanges-1981-pro01')),
                date: '1981-01-01',
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: ETAPES_STATUTS.ACCEPTE,
                ordre: 1,
              },
            ] as ITitreEtape[],
          },
          {
            id: newDemarcheId('h-cx-courdemanges-1981-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'acc',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1981-oct01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1981-oct01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1981-01-01'),
                statutId: 'acc',
                ordre: 1,
                concurrence: 'non-applicable',
                hasTitreFrom: 'non-applicable',
                demarcheIdsConsentement: [],
              },
            ],
          },
        ],
        TitresStatutIds.ModificationEnInstance
      )
    ).toBeNull()
  })

  test("ne trouve pas l'id de la dernière étape de dpu car aucune démarche de prolongation ou de demande de titre en instruction ne contient de périmètre et le titre a le statut 'modification en instance'", () => {
    expect(
      titrePropTitreEtapeFind(
        currentDate,
        'points',
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1981-mut01'),
            typeId: DEMARCHES_TYPES_IDS.Mutation,
            statutId: 'ins',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1981-mut01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1981-mut01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1981-01-01'),
                ordre: 1,
                geojson4326Perimetre: { type: 'Feature', properties: {}, geometry: { type: 'MultiPolygon', coordinates: [[[[1, 2]]]] } },
              },
            ],
          },
          {
            id: newDemarcheId('h-cx-courdemanges-1981-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'acc',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1981-oct01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1981-oct01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1981-01-01'),
                statutId: 'acc',
                ordre: 1,
              },
            ],
          },
        ],
        TitresStatutIds.ModificationEnInstance
      )
    ).toBe(null)
  })

  test.each(['points', 'surface', 'communes'] as IPropId[])(
    "trouve l'id de la dernière étape de n’importe quel type d'une démarche de prolongation ou de demande de titre en instruction car l'étape contient la propriété %s et le titre a le statut 'modification en instance' et aucune phase n'est valide",
    propId => {
      expect(
        titrePropTitreEtapeFind(
          currentDate,
          propId,
          [
            {
              id: newDemarcheId('h-cx-courdemanges-1981-pro01'),
              typeId: DEMARCHES_TYPES_IDS.Prolongation,
              statutId: 'ins',
              ordre: 2,
              etapes: [
                {
                  id: newEtapeId('h-cx-courdemanges-1981-pro01-dpu01'),
                  isBrouillon: ETAPE_IS_NOT_BROUILLON,
                  date: toCaminoDate('1981-01-01'),
                  titreDemarcheId: newDemarcheId('h-cx-courdemanges-1981-pro01'),
                  typeId: ETAPES_TYPES.avisDesServicesEtCommissionsConsultatives,
                  statutId: 'fai',
                  ordre: 1,
                  geojson4326Perimetre: { type: 'Feature', properties: {}, geometry: { type: 'MultiPolygon', coordinates: [[[[1, 2]]]] } },
                  surface: km2Validator.parse(3.2),
                  substances: ['auru'],
                  communes: ['paris' as unknown as ICommune],
                },
              ],
            },
            {
              id: newDemarcheId('h-cx-courdemanges-1981-oct01'),
              typeId: DEMARCHES_TYPES_IDS.Octroi,
              statutId: 'acc',
              ordre: 1,
              etapes: [
                {
                  id: newEtapeId('h-cx-courdemanges-1981-oct01-dpu01'),
                  titreDemarcheId: newDemarcheId('h-cx-courdemanges-1981-oct01'),
                  typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                  statutId: 'acc',
                  ordre: 1,
                  isBrouillon: ETAPE_IS_NOT_BROUILLON,
                  date: toCaminoDate('1981-01-01'),
                  geojson4326Perimetre: {
                    type: 'Feature',
                    properties: {},
                    geometry: {
                      type: 'MultiPolygon',
                      coordinates: [
                        [
                          [
                            [1, 2],
                            [2, 3],
                          ],
                        ],
                      ],
                    },
                  },
                  surface: km2Validator.parse(3),
                  substances: ['arge'],
                  communes: ['tours'] as unknown as ICommune[],
                },
              ],
            },
          ],
          TitresStatutIds.ModificationEnInstance
        )?.id
      ).toEqual('h-cx-courdemanges-1981-oct01-dpu01')
    }
  )

  test.each<IPropId>(['titulaires', 'substances'])("ne trouve pas l'id de la mod car la propriété %s n’est pas modifiée par cette étape", propId => {
    expect(
      titrePropTitreEtapeFind(
        currentDate,
        propId,
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1981-pro01'),
            typeId: DEMARCHES_TYPES_IDS.Prolongation,
            statutId: 'ins',
            ordre: 2,
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1981-pro01-dpu01'),
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1981-01-01'),
                titreDemarcheId: newDemarcheId(newDemarcheId('h-cx-courdemanges-1981-pro01')),
                typeId: ETAPES_TYPES.avisDesServicesEtCommissionsConsultatives,
                statutId: 'fai',
                ordre: 1,
                geojson4326Perimetre: { type: 'Feature', properties: {}, geometry: { type: 'MultiPolygon', coordinates: [[[[1, 2]]]] } },
                surface: km2Validator.parse(3.2),
                substances: ['auru'],
                titulaireIds: [newEntrepriseId('titulaire2')],
                amodiataireIds: [newEntrepriseId('amodiataire2')],
              },
            ],
          },
          {
            id: newDemarcheId('h-cx-courdemanges-1981-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'acc',
            ordre: 1,
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1981-oct01-dpu01'),
                titreDemarcheId: newDemarcheId(newDemarcheId('h-cx-courdemanges-1981-oct01')),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1981-01-01'),
                ordre: 1,
                geojson4326Perimetre: {
                  type: 'Feature',
                  properties: {},
                  geometry: {
                    type: 'MultiPolygon',
                    coordinates: [
                      [
                        [
                          [1, 2],
                          [2, 3],
                        ],
                      ],
                    ],
                  },
                },
                surface: km2Validator.parse(3),
                substances: ['arge'],
                titulaireIds: [newEntrepriseId('titulaire1')],
                amodiataireIds: [newEntrepriseId('amodiataire1')],
              },
            ],
          },
        ],
        TitresStatutIds.ModificationEnInstance
      )?.id
    ).toEqual('h-cx-courdemanges-1981-oct01-dpu01')
  })

  test("trouve l'id de l’unique étape de la démarche d’octroi contenant la propriété 'titulaires'", () => {
    expect(
      titrePropTitreEtapeFind(
        currentDate,
        'titulaires',
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1982-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'acc',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1982-oct01-mfr01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1982-oct01'),
                typeId: ETAPES_TYPES.demande,
                statutId: 'fai',
                ordre: 1,
                isBrouillon: ETAPE_IS_BROUILLON,
                date: toCaminoDate('1982-01-01'),
                dateFin: toCaminoDate('2018-12-31'),
                titulaireIds: [newEntrepriseId('fr-123456789')],
              },
            ],
          },
        ],
        TitresStatutIds.DemandeInitiale
      )?.id
    ).toEqual('h-cx-courdemanges-1982-oct01-mfr01')
  })

  // amodiataires

  test("trouve pas d'id si la démarche de l'étape contenant la propriété 'amodiataires' a une phase valide", () => {
    const newDate = toCaminoDate('2015-01-02')
    expect(
      titrePropTitreEtapeFind(
        newDate,
        'amodiataires',
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1982-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'acc',
            demarcheDateDebut: toCaminoDate('1982-01-01'),
            demarcheDateFin: toCaminoDate('2018-12-31'),
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1982-oct01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1982-oct01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                ordre: 1,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1982-01-01'),
                dateFin: toCaminoDate('2018-12-31'),
                amodiataireIds: [newEntrepriseId('fr-123456789')],
              },
            ],
          },
        ],
        TitresStatutIds.Valide
      )?.id
    ).toBe('h-cx-courdemanges-1982-oct01-dpu01')
  })

  test("trouve l'id de dernière étape contenant la propriété 'amodiataires' dont la démarche précédente a une phase valide", () => {
    const newDate = toCaminoDate('2021-10-01')
    expect(
      titrePropTitreEtapeFind(
        newDate,
        'amodiataires',
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1982-amo01'),
            typeId: DEMARCHES_TYPES_IDS.Amodiation,
            statutId: 'acc',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1982-amo01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1982-amo01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                ordre: 1,
                dateFin: toCaminoDate('4018-12-31'),
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1982-01-01'),
                amodiataireIds: [newEntrepriseId('fr-123456789')],
              },
            ],
          },
          {
            id: newDemarcheId('h-cx-courdemanges-1982-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'acc',
            demarcheDateDebut: toCaminoDate('2021-01-01'),
            demarcheDateFin: toCaminoDate('2022-01-01'),
            etapes: [],
          },
        ],
        TitresStatutIds.ModificationEnInstance
      )?.id
    ).toEqual('h-cx-courdemanges-1982-amo01-dpu01')
  })

  test("ne trouve pas l'id de la dernière étape contenant la propriété 'amodiataires'", () => {
    expect(
      titrePropTitreEtapeFind(
        currentDate,
        'amodiataires',
        [
          {
            id: newDemarcheId('h-cx-courdemanges-1981-amo01'),
            typeId: DEMARCHES_TYPES_IDS.Amodiation,
            statutId: 'acc',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1981-amo01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1981-amo01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1981-01-01'),
                ordre: 1,
                amodiataireIds: [newEntrepriseId('fr-123456789')],
              },
            ],
          },
          {
            id: newDemarcheId('h-cx-courdemanges-1981-pro01'),
            typeId: DEMARCHES_TYPES_IDS.Prolongation,
            statutId: 'acc',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1981-pro01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1981-pro01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1981-01-01'),
                statutId: 'acc',
                ordre: 1,
              },
            ],
          },
          {
            id: newDemarcheId('h-cx-courdemanges-1981-oct01'),
            typeId: DEMARCHES_TYPES_IDS.Octroi,
            statutId: 'acc',
            etapes: [
              {
                id: newEtapeId('h-cx-courdemanges-1981-oct01-dpu01'),
                titreDemarcheId: newDemarcheId('h-cx-courdemanges-1981-oct01'),
                isBrouillon: ETAPE_IS_NOT_BROUILLON,
                date: toCaminoDate('1981-01-01'),
                typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
                statutId: 'acc',
                ordre: 1,
              },
            ],
          },
        ],
        TitresStatutIds.ModificationEnInstance
      )
    ).toBeNull()
  })
})

describe("id de l'étape qui a un contenu", () => {
  test("retourne null si aucune étape n'est trouvé", () => {
    const etape1 = titreContenuTitreEtapeFind({ sectionId: 'arm', elementId: 'mecanisee' }, [{ id: newDemarcheId('demarche-id'), etapes: [{ id: newEtapeId('etape-id') }] }] as ITitreDemarche[], 'val')

    const etape2 = titreContenuTitreEtapeFind(
      { sectionId: 'arm', elementId: 'mecanisee' },
      [
        {
          id: newDemarcheId('demarche-id'),
          statutId: 'acc',
          etapes: [{ id: newEtapeId('etape-id'), statutId: 'fai' }],
        },
      ] as ITitreDemarche[],
      'val'
    )

    const demarcheId = newDemarcheId('demarche-id')
    const etape3 = titreContenuTitreEtapeFind(
      { sectionId: 'arm', elementId: 'mecanisee' },
      [
        {
          id: demarcheId,
          titreId: newTitreId('titre-id'),
          typeId: DEMARCHES_TYPES_IDS.Prolongation,
          machineId:
            MachineInfo.withDate(TITRES_TYPES_IDS.AUTORISATION_DE_RECHERCHE_METAUX, DEMARCHES_TYPES_IDS.Prolongation, demarcheId, firstEtapeDateValidator.parse('2020-01-01')).machineId ?? null,
          demarcheDateDebut: toCaminoDate('2020-01-01'),
          demarcheDateFin: toCaminoDate('2020-01-02'),
          etapes: [
            {
              id: newEtapeId('etape-id'),
              titreDemarcheId: demarcheId,
              typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
              isBrouillon: ETAPE_IS_NOT_BROUILLON,
              date: toCaminoDate('2020-01-01'),
              statutId: 'acc',
              contenu: { arm: { mecanisee: true } },
              concurrence: 'non-applicable',
              hasTitreFrom: 'non-applicable',
              demarcheIdsConsentement: [],
            },
          ],
        },
      ],
      'mod'
    )

    expect(etape1).toBeNull()
    expect(etape2).toBeNull()
    expect(etape3).toBeNull()
  })

  test("retourne l'id de l'étape si elle existe", () => {
    const demarcheId = newDemarcheId('demarche-id')
    const demarcheId2 = newDemarcheId('demarche-id-2')
    const etape1 = titreContenuTitreEtapeFind(
      { sectionId: 'arm', elementId: 'mecanisee' },
      [
        {
          id: demarcheId,
          titreId: newTitreId('titre-id'),
          typeId: DEMARCHES_TYPES_IDS.Octroi,
          machineId: MachineInfo.withDate(TITRES_TYPES_IDS.AUTORISATION_DE_RECHERCHE_METAUX, DEMARCHES_TYPES_IDS.Octroi, demarcheId, firstEtapeDateValidator.parse('2020-01-03')).machineId ?? null,
          etapes: [
            {
              id: newEtapeId('etape-id'),
              titreDemarcheId: demarcheId,
              typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
              isBrouillon: ETAPE_IS_NOT_BROUILLON,
              date: toCaminoDate('2020-01-03'),
              statutId: 'acc',
              contenu: { arm: { mecanisee: true } },
              concurrence: 'non-applicable',
              hasTitreFrom: 'non-applicable',
              demarcheIdsConsentement: [],
            },
          ],
        },
        {
          id: demarcheId2,
          titreId: newTitreId('titre-id'),
          typeId: DEMARCHES_TYPES_IDS.Prolongation,
          machineId:
            MachineInfo.withDate(TITRES_TYPES_IDS.AUTORISATION_DE_RECHERCHE_METAUX, DEMARCHES_TYPES_IDS.Prolongation, demarcheId2, firstEtapeDateValidator.parse('2020-01-01')).machineId ?? null,
          etapes: [
            {
              id: newEtapeId('etape-id-2'),
              titreDemarcheId: demarcheId2,
              typeId: ETAPES_TYPES.decisionDeLAutoriteAdministrative,
              isBrouillon: ETAPE_IS_NOT_BROUILLON,
              date: toCaminoDate('2020-01-01'),
              statutId: 'fai',
              concurrence: 'non-applicable',
              hasTitreFrom: 'non-applicable',
              demarcheIdsConsentement: [],
            },
          ],
        },
      ],
      'val'
    )
    expect(etape1?.id).toEqual('etape-id')

    const etape2 = titreContenuTitreEtapeFind(
      { sectionId: 'arm', elementId: 'mecanisee' },
      [
        {
          id: demarcheId,
          titreId: newTitreId('titre-id'),
          typeId: DEMARCHES_TYPES_IDS.Prolongation,
          machineId:
            MachineInfo.withDate(TITRES_TYPES_IDS.AUTORISATION_DE_RECHERCHE_METAUX, DEMARCHES_TYPES_IDS.Prolongation, demarcheId, firstEtapeDateValidator.parse('2020-01-01')).machineId ?? null,
          statutId: 'acc',
          etapes: [
            {
              id: newEtapeId('etape-id'),
              titreDemarcheId: demarcheId,
              typeId: ETAPES_TYPES.publicationDeDecisionAuJORF,
              isBrouillon: ETAPE_IS_NOT_BROUILLON,
              date: toCaminoDate('2020-01-01'),
              statutId: 'acc',
              contenu: { arm: { mecanisee: true } },
              concurrence: 'non-applicable',
              hasTitreFrom: 'non-applicable',
              demarcheIdsConsentement: [],
            },
          ],
        },
      ],
      'mod'
    )

    expect(etape2?.id).toEqual('etape-id')
  })

  test("ne retourne pas l'id de la demande si le titre n’est pas en dmi", () => {
    const demarcheId = newDemarcheId('demarche-id')
    const etape = titreContenuTitreEtapeFind(
      { sectionId: 'arm', elementId: 'mecanisee' },
      [
        {
          id: demarcheId,
          titreId: newTitreId('titre-id'),
          typeId: DEMARCHES_TYPES_IDS.Octroi,
          machineId: MachineInfo.withDate(TITRES_TYPES_IDS.AUTORISATION_DE_RECHERCHE_METAUX, DEMARCHES_TYPES_IDS.Octroi, demarcheId, firstEtapeDateValidator.parse('2020-01-03')).machineId ?? null,
          etapes: [
            {
              id: newEtapeId('etape-id'),
              titreDemarcheId: demarcheId,
              typeId: ETAPES_TYPES.demande,
              isBrouillon: ETAPE_IS_BROUILLON,
              date: toCaminoDate('2020-01-03'),
              statutId: 'fai',
              contenu: { arm: { mecanisee: true } },
              concurrence: { amIFirst: true },
              hasTitreFrom: true,
              demarcheIdsConsentement: [],
            },
          ],
        },
      ],
      'val'
    )
    expect(etape).toBeNull()
  })

  test("retourne l'id de la demande si le titre est en dmi", () => {
    const demarcheId = newDemarcheId('demarche-id')
    const etape = titreContenuTitreEtapeFind(
      { sectionId: 'arm', elementId: 'mecanisee' },
      [
        {
          id: demarcheId,
          titreId: newTitreId('titre-id'),
          typeId: DEMARCHES_TYPES_IDS.Octroi,
          machineId: MachineInfo.withDate(TITRES_TYPES_IDS.AUTORISATION_DE_RECHERCHE_METAUX, DEMARCHES_TYPES_IDS.Octroi, demarcheId, firstEtapeDateValidator.parse('2020-01-03')).machineId ?? null,
          etapes: [
            {
              id: newEtapeId('etape-id'),
              titreDemarcheId: newDemarcheId('demarche-id'),
              typeId: ETAPES_TYPES.demande,
              isBrouillon: ETAPE_IS_BROUILLON,
              date: toCaminoDate('2020-01-03'),
              statutId: 'fai',
              contenu: { arm: { mecanisee: true } },
              concurrence: { amIFirst: true },
              hasTitreFrom: true,
              demarcheIdsConsentement: [],
            },
          ],
        },
      ],
      'dmi'
    )
    expect(etape!.id).toEqual('etape-id')
  })
})
