import { CaminoAnnee } from 'camino-common/src/date'
import { SubstanceFiscaleId, SUBSTANCES_FISCALES_IDS } from 'camino-common/src/static/substancesFiscales'
import { isNullOrUndefined, RecordPartial } from 'camino-common/src/typescript-tools'
import Decimal from 'decimal.js'
import { z } from 'zod'

const anneeData = ['2017', '2018', '2019', '2020', '2021', '2022', '2023', '2024'] as const
type AnneeData = (typeof anneeData)[number]
const redevanceCommunale = {
  '2017': {
    auru: new Decimal(141.2),
    anti: new Decimal(11.2),
    arge: new Decimal(229.4),
    arse: new Decimal(621.3),
    aloh: new Decimal(540.3),
    bism: new Decimal(54.4),
    hydb: new Decimal(46.4),
    hyda: new Decimal(1388.3),
    hydd: new Decimal(8.0),
    cfxa: new Decimal(215.7),
    cuiv: new Decimal(18.1),
    hyde: new Decimal(7.3),
    etai: new Decimal(112.4),
    ferb: new Decimal(319.0),
    fera: new Decimal(463.0),
    fluo: new Decimal(702.4),
    coox: new Decimal(302.5),
    hydf: new Decimal(298.4),
    cfxb: new Decimal(826.0),
    cfxc: new Decimal(200.7),
    lith: new Decimal(46.3),
    mang: new Decimal(345.0),
    moly: new Decimal(229.4),
    hydc: new Decimal(1067.0),
    plom: new Decimal(582.8),
    kclx: new Decimal(243.3),
    naca: new Decimal(668.0),
    nacc: new Decimal(135.8),
    nacb: new Decimal(406.6),
    souf: new Decimal(2.6),
    wolf: new Decimal(126.2),
    uran: new Decimal(274.8),
    zinc: new Decimal(463.0),
    reference: 'https://beta.legifrance.gouv.fr/codes/id/LEGISCTA000006191913/2018-01-01',
  },
  '2018': {
    auru: new Decimal(145.3),
    anti: new Decimal(11.5),
    arge: new Decimal(236.1),
    arse: new Decimal(639.3),
    aloh: new Decimal(556.0),
    bism: new Decimal(56.0),
    hydb: new Decimal(47.6),
    hyda: new Decimal(1428.6),
    hydd: new Decimal(8.2),
    cfxa: new Decimal(222.0),
    cuiv: new Decimal(18.6),
    hyde: new Decimal(7.5),
    etai: new Decimal(115.7),
    ferb: new Decimal(328.3),
    fera: new Decimal(476.4),
    fluo: new Decimal(722.8),
    coox: new Decimal(311.3),
    hydf: new Decimal(307.1),
    cfxb: new Decimal(850.0),
    cfxc: new Decimal(206.5),
    lith: new Decimal(47.6),
    mang: new Decimal(355.0),
    moly: new Decimal(236.1),
    hydc: new Decimal(1067.0),
    plom: new Decimal(599.7),
    kclx: new Decimal(250.4),
    naca: new Decimal(687.4),
    nacc: new Decimal(139.7),
    nacb: new Decimal(418.4),
    souf: new Decimal(2.7),
    wolf: new Decimal(129.9),
    uran: new Decimal(282.8),
    zinc: new Decimal(476.4),
    reference: 'https://beta.legifrance.gouv.fr/codes/id/LEGISCTA000006191913/2019-01-01',
  },
  '2019': {
    auru: new Decimal(149.7),
    anti: new Decimal(11.8),
    arge: new Decimal(243.2),
    arse: new Decimal(658.5),
    aloh: new Decimal(572.7),
    bism: new Decimal(57.7),
    hydb: new Decimal(49.0),
    hyda: new Decimal(1471.5),
    hydd: new Decimal(8.4),
    cfxa: new Decimal(228.7),
    cuiv: new Decimal(19.2),
    hyde: new Decimal(7.7),
    etai: new Decimal(119.2),
    ferb: new Decimal(338.1),
    fera: new Decimal(490.7),
    fluo: new Decimal(744.5),
    coox: new Decimal(320.6),
    hydf: new Decimal(316.3),
    cfxb: new Decimal(875.5),
    cfxc: new Decimal(212.7),
    lith: new Decimal(49.0),
    mang: new Decimal(365.7),
    moly: new Decimal(243.2),
    hydc: new Decimal(1099.0),
    plom: new Decimal(617.7),
    kclx: new Decimal(257.9),
    naca: new Decimal(708.0),
    nacc: new Decimal(143.9),
    nacb: new Decimal(431.0),
    souf: new Decimal(2.8),
    wolf: new Decimal(133.8),
    uran: new Decimal(291.3),
    zinc: new Decimal(490.7),
    reference: 'https://www.legifrance.gouv.fr/codes/id/LEGISCTA000006191913/2020-01-01',
  },
  '2020': {
    auru: new Decimal(153.6),
    anti: new Decimal(12.1),
    arge: new Decimal(249.5),
    arse: new Decimal(675.6),
    aloh: new Decimal(587.6),
    bism: new Decimal(59.2),
    hydb: new Decimal(50.3),
    hyda: new Decimal(1509.8),
    hydd: new Decimal(8.6),
    cfxa: new Decimal(234.6),
    cuiv: new Decimal(19.7),
    hyde: new Decimal(7.9),
    etai: new Decimal(122.3),
    ferb: new Decimal(346.9),
    fera: new Decimal(503.5),
    fluo: new Decimal(763.9),
    coox: new Decimal(328.9),
    hydf: new Decimal(324.5),
    cfxb: new Decimal(898.3),
    cfxc: new Decimal(218.2),
    lith: new Decimal(50.3),
    mang: new Decimal(375.2),
    moly: new Decimal(249.5),
    hydc: new Decimal(1127.6),
    plom: new Decimal(633.8),
    kclx: new Decimal(264.6),
    naca: new Decimal(726.4),
    nacc: new Decimal(147.6),
    nacb: new Decimal(442.2),
    souf: new Decimal(2.9),
    wolf: new Decimal(137.3),
    uran: new Decimal(298.9),
    zinc: new Decimal(503.5),
    reference: 'https://www.legifrance.gouv.fr/codes/id/LEGISCTA000006191913/2020-07-25',
  },
  '2021': {
    auru: new Decimal(166.3),
    anti: new Decimal(13.1),
    arge: new Decimal(270.2),
    arse: new Decimal(731.7),
    aloh: new Decimal(636.4),
    bism: new Decimal(64.1),
    hydb: new Decimal(54.5),
    hyda: new Decimal(1635.1),
    hydd: new Decimal(9.3),
    cfxa: new Decimal(254.1),
    cuiv: new Decimal(21.3),
    hyde: new Decimal(8.6),
    etai: new Decimal(132.5),
    ferb: new Decimal(375.7),
    fera: new Decimal(545.3),
    fluo: new Decimal(827.3),
    coox: new Decimal(356.2),
    hydf: new Decimal(351.4),
    cfxb: new Decimal(972.9),
    cfxc: new Decimal(236.3),
    lith: new Decimal(54.5),
    mang: new Decimal(406.3),
    moly: new Decimal(270.2),
    hydc: new Decimal(1221.2),
    plom: new Decimal(686.4),
    kclx: new Decimal(286.6),
    naca: new Decimal(786.7),
    nacc: new Decimal(159.9),
    nacb: new Decimal(478.9),
    souf: new Decimal(3.1),
    wolf: new Decimal(148.7),
    uran: new Decimal(323.7),
    zinc: new Decimal(545.3),
    reference: 'https://www.legifrance.gouv.fr/codes/id/LEGIARTI000043663105/2021-06-12/',
  },
  '2022': {
    auru: new Decimal(175.4),
    anti: new Decimal(13.8),
    arge: new Decimal(285.1),
    arse: new Decimal(771.9),
    aloh: new Decimal(671.4),
    bism: new Decimal(67.6),
    hydb: new Decimal(57.5),
    hyda: new Decimal(1725.0),
    hydd: new Decimal(9.8),
    cfxa: new Decimal(268.1),
    cuiv: new Decimal(22.5),
    hyde: new Decimal(9.1),
    etai: new Decimal(139.8),
    ferb: new Decimal(396.4),
    fera: new Decimal(575.3),
    fluo: new Decimal(872.8),
    coox: new Decimal(375.8),
    hydf: new Decimal(370.7),
    cfxb: new Decimal(1026.4),
    cfxc: new Decimal(249.3),
    lith: new Decimal(57.5),
    mang: new Decimal(428.6),
    moly: new Decimal(285.1),
    hydc: new Decimal(1288.4),
    plom: new Decimal(724.2),
    kclx: new Decimal(302.4),
    naca: new Decimal(830.0),
    nacc: new Decimal(168.7),
    nacb: new Decimal(505.2),
    souf: new Decimal(3.3),
    wolf: new Decimal(156.9),
    uran: new Decimal(341.5),
    zinc: new Decimal(575.3),
    reference: 'https://www.legifrance.gouv.fr/codes/id/LEGIARTI000045765025/2022-05-07/',
  },
  '2023': {
    auru: new Decimal(183.5),
    anti: new Decimal(14.4),
    arge: new Decimal(298.2),
    arse: new Decimal(807.4),
    aloh: new Decimal(702.3),
    bism: new Decimal(70.7),
    hydb: new Decimal(60.1),
    hyda: new Decimal(1804.3),
    hydd: new Decimal(10.2),
    cfxa: new Decimal(280.4),
    cuiv: new Decimal(23.5),
    hyde: new Decimal(9.5),
    etai: new Decimal(146.2),
    ferb: new Decimal(414.6),
    fera: new Decimal(601.8),
    fluo: new Decimal(912.9),
    coox: new Decimal(393.1),
    hydf: new Decimal(387.7),
    cfxb: new Decimal(1073.6),
    cfxc: new Decimal(260.8),
    lith: new Decimal(60.1),
    mang: new Decimal(448.3),
    moly: new Decimal(298.2),
    hydc: new Decimal(1347.7),
    plom: new Decimal(757.5),
    kclx: new Decimal(316.3),
    naca: new Decimal(868.2),
    nacc: new Decimal(176.5),
    nacb: new Decimal(528.4),
    souf: new Decimal(3.4),
    wolf: new Decimal(164.1),
    uran: new Decimal(357.2),
    zinc: new Decimal(601.8),
    reference: 'https://www.legifrance.gouv.fr/codes/id/LEGIARTI000045765025/2023-06-03/',
  },
  '2024': {
    [SUBSTANCES_FISCALES_IDS.or]: new Decimal(190.8),
    [SUBSTANCES_FISCALES_IDS.uranium]: new Decimal(371.5),
    [SUBSTANCES_FISCALES_IDS['oxyde de tungstène (WO3)']]: new Decimal(170.7),
    [SUBSTANCES_FISCALES_IDS.argent]: new Decimal(310.1),
    [SUBSTANCES_FISCALES_IDS.bauxite]: new Decimal(730.4),
    [SUBSTANCES_FISCALES_IDS.fluorine]: new Decimal(949.4),
    [SUBSTANCES_FISCALES_IDS.sel_ChlorureDeSodium_extraitParAbattage]: new Decimal(902.9),
    [SUBSTANCES_FISCALES_IDS.sel_ChlorureDeSodium_extraitEnDissolutionParSondage]: new Decimal(549.5),
    [SUBSTANCES_FISCALES_IDS.sel_ChlorureDeSodiumContenu_]: new Decimal(183.6),
    [SUBSTANCES_FISCALES_IDS.charbon]: new Decimal(291.6),
    [SUBSTANCES_FISCALES_IDS['pétrole brut']]: new Decimal(1_401.6),
    [SUBSTANCES_FISCALES_IDS['propane et le butane']]: new Decimal(10.6),
    [SUBSTANCES_FISCALES_IDS['essence de dégazolinage']]: new Decimal(9.9),
    [SUBSTANCES_FISCALES_IDS.soufre]: new Decimal(3.5),
    [SUBSTANCES_FISCALES_IDS['lignites pouvoir calorifique égal ou supérieur à 13 MJ/kg']]: new Decimal(1_116.5),
    [SUBSTANCES_FISCALES_IDS['lignites pouvoir calorifique inférieur à 13 MJ/kg']]: new Decimal(271.2),
    [SUBSTANCES_FISCALES_IDS['gaz carbonique']]: new Decimal(408.8),
    [SUBSTANCES_FISCALES_IDS['calcaires et grès bitumineux ou asphaltiques']]: new Decimal(1_876.5),
    [SUBSTANCES_FISCALES_IDS['schistes carbobitumineux et schistes bitumineux']]: new Decimal(62.5),
    [SUBSTANCES_FISCALES_IDS['pyrite de fer']]: new Decimal(625.9),
    [SUBSTANCES_FISCALES_IDS['minerais de fer']]: new Decimal(431.2),
    [SUBSTANCES_FISCALES_IDS.antimoine]: new Decimal(15),
    [SUBSTANCES_FISCALES_IDS.plomb]: new Decimal(787.8),
    [SUBSTANCES_FISCALES_IDS.zinc]: new Decimal(625.9),
    [SUBSTANCES_FISCALES_IDS.étain]: new Decimal(152),
    [SUBSTANCES_FISCALES_IDS.cuivre]: new Decimal(24.4),
    [SUBSTANCES_FISCALES_IDS.arsenic]: new Decimal(839.7),
    [SUBSTANCES_FISCALES_IDS.bismuth]: new Decimal(73.5),
    [SUBSTANCES_FISCALES_IDS.manganèse]: new Decimal(466.2),
    [SUBSTANCES_FISCALES_IDS.molybdène]: new Decimal(310.1),
    [SUBSTANCES_FISCALES_IDS['oxyde de lithium']]: new Decimal(62.5),
    [SUBSTANCES_FISCALES_IDS['oxyde de potassium']]: new Decimal(329),
    [SUBSTANCES_FISCALES_IDS['gaz naturel']]: new Decimal(403.2),
    reference: 'https://www.legifrance.gouv.fr/codes/id/LEGIARTI000045765025/2024-06-03/',
  },
} as const satisfies Record<AnneeData, Record<SubstanceFiscaleId, Decimal> & { reference: string }>

const redevanceDepartementale = {
  '2017': {
    auru: new Decimal(28.2),
    hyde: new Decimal(5.4),
    cuiv: new Decimal(3.7),
    cfxa: new Decimal(104.8),
    hyda: new Decimal(276.4),
    hydb: new Decimal(9.5),
    bism: new Decimal(11.1),
    aloh: new Decimal(108),
    arse: new Decimal(126.2),
    arge: new Decimal(45.7),
    anti: new Decimal(2.5),
    etai: new Decimal(22.3),
    ferb: new Decimal(66.4),
    fera: new Decimal(94.7),
    fluo: new Decimal(142.8),
    coox: new Decimal(61.8),
    hydf: new Decimal(435.7),
    cfxb: new Decimal(163.6),
    cfxc: new Decimal(44.5),
    lith: new Decimal(9.4),
    mang: new Decimal(69.8),
    moly: new Decimal(46.3),
    hydc: new Decimal(1371),
    plom: new Decimal(112.4),
    kclx: new Decimal(48.5),
    hydd: new Decimal(6.1),
    naca: new Decimal(135.8),
    nacc: new Decimal(26.3),
    nacb: new Decimal(80.2),
    souf: new Decimal(1.6),
    uran: new Decimal(54.6),
    wolf: new Decimal(24.7),
    zinc: new Decimal(94.7),
    reference: 'https://beta.legifrance.gouv.fr/codes/section_lc/LEGITEXT000006069577/LEGISCTA000006162672/2018-01-01/',
  },
  '2018': {
    auru: new Decimal(29),
    hyde: new Decimal(5.6),
    cuiv: new Decimal(3.8),
    cfxa: new Decimal(107.8),
    hyda: new Decimal(284.4),
    hydb: new Decimal(9.8),
    bism: new Decimal(11.4),
    aloh: new Decimal(111.1),
    arse: new Decimal(129.9),
    arge: new Decimal(47),
    anti: new Decimal(2.6),
    etai: new Decimal(22.9),
    ferb: new Decimal(68.3),
    fera: new Decimal(97.4),
    fluo: new Decimal(146.9),
    coox: new Decimal(63.6),
    hydf: new Decimal(448.3),
    cfxb: new Decimal(168.3),
    cfxc: new Decimal(45.8),
    lith: new Decimal(9.7),
    mang: new Decimal(71.8),
    moly: new Decimal(47.6),
    hydc: new Decimal(1371),
    plom: new Decimal(115.7),
    kclx: new Decimal(49.9),
    hydd: new Decimal(6.3),
    naca: new Decimal(139.7),
    nacc: new Decimal(27.1),
    nacb: new Decimal(82.5),
    souf: new Decimal(1.6),
    uran: new Decimal(56.2),
    wolf: new Decimal(25.4),
    zinc: new Decimal(97.4),
    reference: 'https://beta.legifrance.gouv.fr/codes/section_lc/LEGITEXT000006069577/LEGISCTA000006162672/2019-01-01/',
  },
  '2019': {
    auru: new Decimal(29.9),
    hyde: new Decimal(5.8),
    cuiv: new Decimal(3.9),
    cfxa: new Decimal(111.0),
    hyda: new Decimal(292.9),
    hydb: new Decimal(10.1),
    bism: new Decimal(11.7),
    aloh: new Decimal(114.4),
    arse: new Decimal(133.8),
    arge: new Decimal(48.4),
    anti: new Decimal(2.7),
    etai: new Decimal(23.6),
    ferb: new Decimal(70.3),
    fera: new Decimal(100.3),
    fluo: new Decimal(151.3),
    coox: new Decimal(65.5),
    hydf: new Decimal(461.7),
    cfxb: new Decimal(173.3),
    cfxc: new Decimal(47.2),
    lith: new Decimal(10.0),
    mang: new Decimal(74.0),
    moly: new Decimal(49.0),
    hydc: new Decimal(1412.1),
    plom: new Decimal(119.2),
    kclx: new Decimal(51.4),
    hydd: new Decimal(6.5),
    naca: new Decimal(143.9),
    nacc: new Decimal(27.9),
    nacb: new Decimal(85.0),
    souf: new Decimal(1.6),
    uran: new Decimal(57.9),
    wolf: new Decimal(26.2),
    zinc: new Decimal(100.3),
    reference: 'https://www.legifrance.gouv.fr/codes/id/LEGIARTI000038686694/2019-06-08/',
  },
  '2020': {
    auru: new Decimal(30.7),
    hyde: new Decimal(6.0),
    cuiv: new Decimal(4.0),
    cfxa: new Decimal(113.9),
    hyda: new Decimal(300.5),
    hydb: new Decimal(10.4),
    bism: new Decimal(12.0),
    aloh: new Decimal(117.4),
    arse: new Decimal(137.3),
    arge: new Decimal(49.7),
    anti: new Decimal(2.8),
    etai: new Decimal(24.2),
    ferb: new Decimal(72.1),
    fera: new Decimal(102.9),
    fluo: new Decimal(155.2),
    coox: new Decimal(67.2),
    hydf: new Decimal(473.7),
    cfxb: new Decimal(177.8),
    cfxc: new Decimal(48.4),
    lith: new Decimal(10.3),
    mang: new Decimal(75.9),
    moly: new Decimal(50.3),
    hydc: new Decimal(1448.8),
    plom: new Decimal(112.3),
    kclx: new Decimal(52.7),
    hydd: new Decimal(6.7),
    naca: new Decimal(147.6),
    nacc: new Decimal(28.6),
    nacb: new Decimal(87.2),
    souf: new Decimal(1.6),
    uran: new Decimal(59.4),
    wolf: new Decimal(26.9),
    zinc: new Decimal(102.9),
    reference: 'https://www.legifrance.gouv.fr/codes/id/LEGIARTI000042159975/2020-07-25/',
  },
  '2021': {
    auru: new Decimal(33.2),
    hyde: new Decimal(6.5),
    cuiv: new Decimal(4.3),
    cfxa: new Decimal(123.4),
    hyda: new Decimal(325.4),
    hydb: new Decimal(11.3),
    bism: new Decimal(13.0),
    aloh: new Decimal(127.1),
    arse: new Decimal(148.7),
    arge: new Decimal(53.8),
    anti: new Decimal(3.0),
    etai: new Decimal(26.2),
    ferb: new Decimal(78.1),
    fera: new Decimal(111.4),
    fluo: new Decimal(168.1),
    coox: new Decimal(72.8),
    hydf: new Decimal(513.0),
    cfxb: new Decimal(192.6),
    cfxc: new Decimal(52.4),
    lith: new Decimal(11.2),
    mang: new Decimal(82.2),
    moly: new Decimal(54.5),
    hydc: new Decimal(1569.1),
    plom: new Decimal(132.5),
    kclx: new Decimal(57.1),
    hydd: new Decimal(7.3),
    naca: new Decimal(159.9),
    nacc: new Decimal(31.0),
    nacb: new Decimal(94.4),
    souf: new Decimal(1.7),
    uran: new Decimal(64.3),
    wolf: new Decimal(29.1),
    zinc: new Decimal(111.4),
    reference: 'https://www.legifrance.gouv.fr/codes/id/LEGIARTI000043663002/2021-06-12/',
  },
  '2022': {
    auru: new Decimal(35.0),
    hyde: new Decimal(6.9),
    cuiv: new Decimal(4.5),
    cfxa: new Decimal(130.2),
    hyda: new Decimal(343.3),
    hydb: new Decimal(11.9),
    bism: new Decimal(13.7),
    aloh: new Decimal(134.1),
    arse: new Decimal(156.9),
    arge: new Decimal(56.8),
    anti: new Decimal(3.2),
    etai: new Decimal(27.6),
    ferb: new Decimal(82.4),
    fera: new Decimal(117.5),
    fluo: new Decimal(177.3),
    coox: new Decimal(76.8),
    hydf: new Decimal(541.2),
    cfxb: new Decimal(203.2),
    cfxc: new Decimal(55.3),
    lith: new Decimal(11.8),
    mang: new Decimal(86.7),
    moly: new Decimal(57.5),
    hydc: new Decimal(1655.4),
    plom: new Decimal(139.8),
    kclx: new Decimal(60.2),
    hydd: new Decimal(7.7),
    naca: new Decimal(168.7),
    nacc: new Decimal(32.7),
    nacb: new Decimal(99.6),
    souf: new Decimal(1.8),
    uran: new Decimal(67.8),
    wolf: new Decimal(30.7),
    zinc: new Decimal(117.5),
    reference: 'https://www.legifrance.gouv.fr/codes/id/LEGIARTI000045764991/2022-05-07/',
  },
  '2023': {
    auru: new Decimal(36.6),
    hyde: new Decimal(7.2),
    cuiv: new Decimal(4.7),
    cfxa: new Decimal(136.2),
    hyda: new Decimal(359.1),
    hydb: new Decimal(12.4),
    bism: new Decimal(14.3),
    aloh: new Decimal(140.3),
    arse: new Decimal(164.1),
    arge: new Decimal(59.4),
    anti: new Decimal(3.3),
    etai: new Decimal(28.9),
    ferb: new Decimal(86.2),
    fera: new Decimal(122.9),
    fluo: new Decimal(185.5),
    coox: new Decimal(80.3),
    hydf: new Decimal(566.1),
    cfxb: new Decimal(212.5),
    cfxc: new Decimal(57.8),
    lith: new Decimal(12.3),
    mang: new Decimal(90.7),
    moly: new Decimal(60.1),
    hydc: new Decimal(1731.5),
    plom: new Decimal(146.2),
    kclx: new Decimal(63),
    hydd: new Decimal(8.1),
    naca: new Decimal(176.5),
    nacc: new Decimal(34.2),
    nacb: new Decimal(104.2),
    souf: new Decimal(1.9),
    uran: new Decimal(70.9),
    wolf: new Decimal(32.1),
    zinc: new Decimal(122.9),
    reference: 'https://www.legifrance.gouv.fr/codes/id/LEGIARTI000045764991/2023-06-03/',
  },
  '2024': {
    [SUBSTANCES_FISCALES_IDS.or]: new Decimal(38.1),
    [SUBSTANCES_FISCALES_IDS.uranium]: new Decimal(73.7),
    [SUBSTANCES_FISCALES_IDS['oxyde de tungstène (WO3)']]: new Decimal(33.4),
    [SUBSTANCES_FISCALES_IDS.argent]: new Decimal(61.8),
    [SUBSTANCES_FISCALES_IDS.bauxite]: new Decimal(145.9),
    [SUBSTANCES_FISCALES_IDS.fluorine]: new Decimal(192.9),
    [SUBSTANCES_FISCALES_IDS.sel_ChlorureDeSodium_extraitParAbattage]: new Decimal(183.6),
    [SUBSTANCES_FISCALES_IDS.sel_ChlorureDeSodium_extraitEnDissolutionParSondage]: new Decimal(108.4),
    [SUBSTANCES_FISCALES_IDS.sel_ChlorureDeSodiumContenu_]: new Decimal(35.6),
    [SUBSTANCES_FISCALES_IDS.charbon]: new Decimal(141.7),
    [SUBSTANCES_FISCALES_IDS['pétrole brut']]: new Decimal(1_800.8),
    [SUBSTANCES_FISCALES_IDS['propane et le butane']]: new Decimal(8.4),
    [SUBSTANCES_FISCALES_IDS['essence de dégazolinage']]: new Decimal(7.5),
    [SUBSTANCES_FISCALES_IDS.soufre]: new Decimal(2),
    [SUBSTANCES_FISCALES_IDS['lignites pouvoir calorifique égal ou supérieur à 13 MJ/kg']]: new Decimal(221),
    [SUBSTANCES_FISCALES_IDS['lignites pouvoir calorifique inférieur à 13 MJ/kg']]: new Decimal(60.1),
    [SUBSTANCES_FISCALES_IDS['gaz carbonique']]: new Decimal(83.5),
    [SUBSTANCES_FISCALES_IDS['calcaires et grès bitumineux ou asphaltiques']]: new Decimal(373.5),
    [SUBSTANCES_FISCALES_IDS['schistes carbobitumineux et schistes bitumineux']]: new Decimal(12.9),
    [SUBSTANCES_FISCALES_IDS['pyrite de fer']]: new Decimal(127.8),
    [SUBSTANCES_FISCALES_IDS['minerais de fer']]: new Decimal(89.6),
    [SUBSTANCES_FISCALES_IDS.antimoine]: new Decimal(3.4),
    [SUBSTANCES_FISCALES_IDS.plomb]: new Decimal(152),
    [SUBSTANCES_FISCALES_IDS.zinc]: new Decimal(127.8),
    [SUBSTANCES_FISCALES_IDS.étain]: new Decimal(30.1),
    [SUBSTANCES_FISCALES_IDS.cuivre]: new Decimal(4.9),
    [SUBSTANCES_FISCALES_IDS.arsenic]: new Decimal(170.7),
    [SUBSTANCES_FISCALES_IDS.bismuth]: new Decimal(14.9),
    [SUBSTANCES_FISCALES_IDS.manganèse]: new Decimal(94.3),
    [SUBSTANCES_FISCALES_IDS.molybdène]: new Decimal(62.5),
    [SUBSTANCES_FISCALES_IDS['oxyde de lithium']]: new Decimal(12.8),
    [SUBSTANCES_FISCALES_IDS['oxyde de potassium']]: new Decimal(65.5),
    [SUBSTANCES_FISCALES_IDS['gaz naturel']]: new Decimal(588.7),
    reference: 'https://www.legifrance.gouv.fr/codes/id/LEGIARTI000045764991/2024-06-03/',
  },
} as const satisfies Record<AnneeData, Record<SubstanceFiscaleId, Decimal> & { reference: string }>

export type EntrepriseCategory = 'pme' | 'autre'
const categoriesForTaxeAurifereGuyane = {
  pme: {
    '2017': {
      value: new Decimal(362.95),
      reference: 'https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000036016552/2018-01-01',
    },
    '2018': {
      value: new Decimal(358.3),
      reference: 'https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000037342798/2019-01-01',
    },
    '2019': {
      value: new Decimal(345.23),
      reference: 'https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000039355892/2020-01-01',
    },
    '2020': {
      value: new Decimal(400.35),
      reference: 'https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000042327688/2020-09-13',
    },
    '2021': {
      value: new Decimal(498.06),
      reference: 'https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000044329274/2021-11-17',
    },
    '2022': {
      value: new Decimal(488.97),
      reference: 'https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000044329274/2022-07-27/',
    },
    '2023': {
      value: new Decimal(549.88),
      reference: 'https://www.legifrance.gouv.fr/codes/id/LEGIARTI000048046958/2023-09-07',
    },
    '2024': {
      value: new Decimal(577.19),
      reference: 'https://www.legifrance.gouv.fr/codes/id/LEGIARTI000048046958/2024-12-06',
    },
  },
  autre: {
    '2017': {
      value: new Decimal(725.9),
      reference: 'https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000036016552/2018-01-01',
    },
    '2018': {
      value: new Decimal(716.6),
      reference: 'https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000037342798/2019-01-01',
    },
    '2019': {
      value: new Decimal(690.47),
      reference: 'https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000039355892/2020-01-01',
    },
    '2020': {
      value: new Decimal(800.71),
      reference: 'https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000042327688/2020-09-13',
    },
    '2021': {
      value: new Decimal(996.13),
      reference: 'https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000044329274/2021-11-17',
    },
    '2022': {
      value: new Decimal(977.95),
      reference: 'https://www.legifrance.gouv.fr/codes/article_lc/LEGIARTI000044329274/2022-07-27/',
    },
    '2023': {
      value: new Decimal(1099.77),
      reference: 'https://www.legifrance.gouv.fr/codes/id/LEGIARTI000048046958/2023-09-07',
    },
    '2024': {
      value: new Decimal(1154.38),
      reference: 'https://www.legifrance.gouv.fr/codes/id/LEGIARTI000048046958/2024-12-06',
    },
  },
} as const satisfies Record<EntrepriseCategory, Record<AnneeData, { value: Decimal; reference: string }>>

const SIP_IDS = ['cayenne', 'kourou', 'saintLaurentDuMaroni'] as const
export const sipIdValidator = z.enum(SIP_IDS)
export type Sip = z.infer<typeof sipIdValidator>
export const isSip = (value: string): value is Sip => sipIdValidator.safeParse(value).success
export const getSips = (annee: CaminoAnnee): RecordPartial<Sip, { nom: string; communes: string[] }> => {
  if (annee < '2024') {
    return oldSips
  } else {
    return newSips
  }
}
const oldSips = {
  cayenne: {
    nom: 'SIP de Cayenne',
    communes: [
      '97313', // Montsinery-Tonnegrande
      '97314', // Ouanary
      '97356', // Camopi
      '97302', // Cayenne
      '97301', // Régina
      '97309', // Remire-Montjoly
      '97310', // Roura
      '97308', // Saint-Georges
      '97307', // Matoury
    ],
  },
  kourou: {
    nom: 'SIP de Kourou',
    communes: [
      '97304', // Kourou
      '97305', // Macouria
      '97303', // Iracoubo
      '97312', // Sinnamary
      '97358', // Saint-Élie
    ],
  },
  saintLaurentDuMaroni: {
    nom: 'SIP de Saint-Laurent du Maroni',
    communes: [
      '97362', //  Papaichton
      '97360', //  Apatou
      '97361', //  Awala-Yalimapo
      '97357', //  Grand-Santi
      '97306', //  Mana
      '97352', //  Saül
      '97353', //  Maripasoula
      '97311', //  Saint-Laurent-du-Maroni
    ],
  },
} as const satisfies Record<Sip, { nom: string; communes: string[] }>

const newSips = {
  cayenne: {
    nom: 'SIP de Cayenne',
    communes: [
      '97313', // Montsinery-Tonnegrande
      '97314', // Ouanary
      '97356', // Camopi
      '97302', // Cayenne
      '97301', // Régina
      '97309', // Remire-Montjoly
      '97310', // Roura
      '97308', // Saint-Georges
      '97307', // Matoury
    ],
  },
  saintLaurentDuMaroni: {
    nom: 'SIP de Saint-Laurent du Maroni',
    communes: [
      '97362', //  Papaichton
      '97360', //  Apatou
      '97361', //  Awala-Yalimapo
      '97357', //  Grand-Santi
      '97306', //  Mana
      '97352', //  Saül
      '97353', //  Maripasoula
      '97311', //  Saint-Laurent-du-Maroni
      '97304', // Kourou
      '97305', // Macouria
      '97303', // Iracoubo
      '97312', // Sinnamary
      '97358', // Saint-Élie
    ],
  },
} as const satisfies RecordPartial<Sip, { nom: string; communes: string[] }>

export const getRedevanceCommunale = (annee: CaminoAnnee, substanceId: SubstanceFiscaleId): Decimal => {
  if (annee < anneeData[0]) {
    throw new Error('Impossible de calculer la redevance pour cette année (données absentes)')
  }
  const anneeFound = anneeData.find(value => value === annee)
  if (isNullOrUndefined(anneeFound)) {
    return redevanceCommunale[anneeData[anneeData.length - 1]][substanceId]
  } else {
    return redevanceCommunale[anneeFound][substanceId]
  }
}

export const getRedevanceDepartementale = (annee: CaminoAnnee, substanceId: SubstanceFiscaleId): Decimal => {
  if (annee < anneeData[0]) {
    throw new Error('Impossible de calculer la redevance pour cette année (données absentes)')
  }
  const anneeFound = anneeData.find(value => value === annee)
  if (isNullOrUndefined(anneeFound)) {
    return redevanceDepartementale[anneeData[anneeData.length - 1]][substanceId]
  } else {
    return redevanceDepartementale[anneeFound][substanceId]
  }
}

export const getCategoriesForTaxeAurifereGuyane = (annee: CaminoAnnee, category: EntrepriseCategory): Decimal => {
  if (annee < anneeData[0]) {
    throw new Error('Impossible de calculer la redevance pour cette année (données absentes)')
  }

  const anneeFound = anneeData.find(value => value === annee)
  if (isNullOrUndefined(anneeFound)) {
    return categoriesForTaxeAurifereGuyane[category][anneeData[anneeData.length - 1]].value
  } else {
    return categoriesForTaxeAurifereGuyane[category][anneeFound].value
  }
}

type TarifsBySubstances = Record<SubstanceFiscaleId, { tarifDepartemental: Decimal; tarifCommunal: Decimal }>
export const getAllTarifsBySubstances = (annee: CaminoAnnee): TarifsBySubstances => {
  const anneeFound = anneeData.find(value => value === annee)
  const data: AnneeData = isNullOrUndefined(anneeFound) ? anneeData[anneeData.length - 1] : anneeFound
  if (annee < anneeData[0]) {
    throw new Error('Impossible de calculer la redevance pour cette année (données absentes)')
  }

  return {
    auru: {
      tarifCommunal: redevanceCommunale[data].auru,
      tarifDepartemental: redevanceDepartementale[data].auru,
    },
    aloh: {
      tarifDepartemental: redevanceDepartementale[data].aloh,
      tarifCommunal: redevanceCommunale[data].aloh,
    },
    anti: {
      tarifDepartemental: redevanceDepartementale[data].anti,
      tarifCommunal: redevanceCommunale[data].anti,
    },
    arge: {
      tarifDepartemental: redevanceDepartementale[data].arge,
      tarifCommunal: redevanceCommunale[data].arge,
    },
    arse: {
      tarifDepartemental: redevanceDepartementale[data].arse,
      tarifCommunal: redevanceCommunale[data].arse,
    },
    bism: {
      tarifDepartemental: redevanceDepartementale[data].bism,
      tarifCommunal: redevanceCommunale[data].bism,
    },
    cfxa: {
      tarifDepartemental: redevanceDepartementale[data].cfxa,
      tarifCommunal: redevanceCommunale[data].cfxa,
    },
    cfxb: {
      tarifDepartemental: redevanceDepartementale[data].cfxb,
      tarifCommunal: redevanceCommunale[data].cfxb,
    },
    cfxc: {
      tarifDepartemental: redevanceDepartementale[data].cfxc,
      tarifCommunal: redevanceCommunale[data].cfxc,
    },
    coox: {
      tarifDepartemental: redevanceDepartementale[data].coox,
      tarifCommunal: redevanceCommunale[data].coox,
    },
    cuiv: {
      tarifDepartemental: redevanceDepartementale[data].cuiv,
      tarifCommunal: redevanceCommunale[data].cuiv,
    },
    etai: {
      tarifDepartemental: redevanceDepartementale[data].etai,
      tarifCommunal: redevanceCommunale[data].etai,
    },
    fera: {
      tarifDepartemental: redevanceDepartementale[data].fera,
      tarifCommunal: redevanceCommunale[data].fera,
    },
    ferb: {
      tarifDepartemental: redevanceDepartementale[data].ferb,
      tarifCommunal: redevanceCommunale[data].ferb,
    },
    fluo: {
      tarifDepartemental: redevanceDepartementale[data].fluo,
      tarifCommunal: redevanceCommunale[data].fluo,
    },
    hyda: {
      tarifDepartemental: redevanceDepartementale[data].hyda,
      tarifCommunal: redevanceCommunale[data].hyda,
    },
    hydb: {
      tarifDepartemental: redevanceDepartementale[data].hydb,
      tarifCommunal: redevanceCommunale[data].hydb,
    },
    hydc: {
      tarifDepartemental: redevanceDepartementale[data].hydc,
      tarifCommunal: redevanceCommunale[data].hydc,
    },
    hydd: {
      tarifDepartemental: redevanceDepartementale[data].hydd,
      tarifCommunal: redevanceCommunale[data].hydd,
    },
    hyde: {
      tarifDepartemental: redevanceDepartementale[data].hyde,
      tarifCommunal: redevanceCommunale[data].hyde,
    },
    hydf: {
      tarifDepartemental: redevanceDepartementale[data].hydf,
      tarifCommunal: redevanceCommunale[data].hydf,
    },
    kclx: {
      tarifDepartemental: redevanceDepartementale[data].kclx,
      tarifCommunal: redevanceCommunale[data].kclx,
    },
    lith: {
      tarifDepartemental: redevanceDepartementale[data].lith,
      tarifCommunal: redevanceCommunale[data].lith,
    },
    mang: {
      tarifDepartemental: redevanceDepartementale[data].mang,
      tarifCommunal: redevanceCommunale[data].mang,
    },
    moly: {
      tarifDepartemental: redevanceDepartementale[data].moly,
      tarifCommunal: redevanceCommunale[data].moly,
    },
    naca: {
      tarifDepartemental: redevanceDepartementale[data].naca,
      tarifCommunal: redevanceCommunale[data].naca,
    },
    nacb: {
      tarifDepartemental: redevanceDepartementale[data].nacb,
      tarifCommunal: redevanceCommunale[data].nacb,
    },
    nacc: {
      tarifDepartemental: redevanceDepartementale[data].nacc,
      tarifCommunal: redevanceCommunale[data].nacc,
    },
    plom: {
      tarifDepartemental: redevanceDepartementale[data].plom,
      tarifCommunal: redevanceCommunale[data].plom,
    },
    souf: {
      tarifDepartemental: redevanceDepartementale[data].souf,
      tarifCommunal: redevanceCommunale[data].souf,
    },
    uran: {
      tarifDepartemental: redevanceDepartementale[data].uran,
      tarifCommunal: redevanceCommunale[data].uran,
    },
    wolf: {
      tarifDepartemental: redevanceDepartementale[data].wolf,
      tarifCommunal: redevanceCommunale[data].wolf,
    },
    zinc: {
      tarifDepartemental: redevanceDepartementale[data].zinc,
      tarifCommunal: redevanceCommunale[data].zinc,
    },
  }
}

// 2009-01-01: https://www.legifrance.gouv.fr/codes/id/LEGIARTI000020058692/2009-01-01
// 2016-01-01: https://www.legifrance.gouv.fr/codes/id/LEGIARTI000031817025/2016-01-01
export const taxeAurifereGuyaneDeductionMontantMax = new Decimal(5000)

// 2009-01-01: https://www.legifrance.gouv.fr/codes/id/LEGIARTI000020058692/2009-01-01
// 2016-01-01: https://www.legifrance.gouv.fr/codes/id/LEGIARTI000031817025/2016-01-01
export const taxeAurifereBrutDeductionTaux = new Decimal(0.45)
