import { setDateAndOrderAndInterpretMachine } from '../machine-test-helper'
import { EtapesTypesEtapesStatuts as ETES } from 'camino-common/src/static/etapesTypesEtapesStatuts'
import { describe, expect, test } from 'vitest'
import { ProcedureSimplifieeMachine } from './procedure-simplifiee.machine'
import { TITRES_TYPES_IDS } from 'camino-common/src/static/titresTypes'
import { DEMARCHES_TYPES_IDS } from 'camino-common/src/static/demarchesTypes'

const psMachine = new ProcedureSimplifieeMachine(TITRES_TYPES_IDS.PERMIS_D_EXPLOITATION_GEOTHERMIE, DEMARCHES_TYPES_IDS.Octroi)
const psProMachine = new ProcedureSimplifieeMachine(TITRES_TYPES_IDS.PERMIS_D_EXPLOITATION_GEOTHERMIE, DEMARCHES_TYPES_IDS.Prolongation)
const psRetMachine = new ProcedureSimplifieeMachine(TITRES_TYPES_IDS.PERMIS_D_EXPLOITATION_GEOTHERMIE, DEMARCHES_TYPES_IDS.RenonciationTotale)

describe('vérifie l’arbre des procédures historiques et simplifiées', () => {
  test('peut faire une renonciation totale', () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psRetMachine, '2024-10-10', [ETES.decisionDeLAutoriteAdministrative.ACCEPTE, ETES.publicationDeDecisionAuJORF.FAIT])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                                    (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE]",
        "RENDRE_DECISION_ADMINISTRATION_ACCEPTEE (publique      , accepté                ) -> [PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS]",
        "2024-10-11",
        "PUBLIER_DECISION_ACCEPTEE_AU_JORF       (publique      , accepté et publié      ) -> [FAIRE_ABROGATION]",
        "2024-10-11",
      ]
    `)
  })
  test('statut de la démarche simplifiee sans étape', () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2024-10-10', [])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE]",
      ]
    `)
  })
  test('statut de la démarche historique sans étape', () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-14', [])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
      ]
    `)
  })
  test('statut de la démarche incomplète sans étape', () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '1999-04-14', [])
    expect(tree).toMatchInlineSnapshot(`
       [
         "RIEN (confidentielle, en construction        ) -> [FAIRE_DEMANDE,PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE,SAISIR_INFORMATION_HISTORIQUE_INCOMPLETE]",
       ]
     `)
  })

  test('peut faire une démarche historique incomplète', () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '1999-04-14', [ETES.informationsHistoriquesIncompletes.FAIT])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                                     (confidentielle, en construction        ) -> [FAIRE_DEMANDE,PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE,SAISIR_INFORMATION_HISTORIQUE_INCOMPLETE]",
        "SAISIR_INFORMATION_HISTORIQUE_INCOMPLETE (confidentielle, accepté                ) -> []",
        "[1999-04-15 -> 2000-04-15]",
      ]
    `)
  })

  test('peut publier une démarche historique incomplète', () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '1999-04-14', [ETES.publicationDeDecisionAuJORF.FAIT])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                              (confidentielle, en construction        ) -> [FAIRE_DEMANDE,PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE,SAISIR_INFORMATION_HISTORIQUE_INCOMPLETE]",
        "PUBLIER_DECISION_ACCEPTEE_AU_JORF (publique      , accepté et publié      ) -> [FAIRE_ABROGATION]",
        "[1999-04-15 -> 2000-04-15]",
      ]
    `)
  })

  test('peut déposer une demande', () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-14', [ETES.demande.FAIT, ETES.enregistrementDeLaDemande.FAIT])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE       (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "ENREGISTRER_DEMANDE (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
      ]
    `)
  })

  test('ne peut pas faire deux dépôts de la demande', () => {
    const etapes = [ETES.demande.FAIT, ETES.enregistrementDeLaDemande.FAIT, ETES.enregistrementDeLaDemande.FAIT]
    expect(() => setDateAndOrderAndInterpretMachine(psMachine, '2022-04-12', etapes)).toThrowErrorMatchingInlineSnapshot(
      `[Error: Error: cannot execute step: '{"duree":12,"etapeTypeId":"men","etapeStatutId":"fai","date":"2022-04-15"}' after '["mfr_fai","men_fai"]'. The event {"type":"ENREGISTRER_DEMANDE","date":"2022-04-15","status":"fai","duree":12} should be one of 'CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE']`
    )
  })

  test('peut faire une ouverture de la consultation du public', () => {
    const etapes = [ETES.demande.FAIT, ETES.enregistrementDeLaDemande.FAIT, ETES.consultationDuPublic.EN_COURS]
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-16', etapes)
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                          (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE                 (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "ENREGISTRER_DEMANDE           (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "OUVRIR_CONSULTATION_DU_PUBLIC (publique      , en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS]",
      ]
    `)
  })

  test("tant que l'ouverture du public n'est pas en cours, la demarche est confidentielle", () => {
    const etapes = [ETES.demande.FAIT, ETES.enregistrementDeLaDemande.FAIT, ETES.consultationDuPublic.PROGRAMME]
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-16', etapes)
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                          (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE                 (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "ENREGISTRER_DEMANDE           (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "OUVRIR_CONSULTATION_DU_PUBLIC (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS]",
      ]
    `)
  })

  test("peut rendre une décision d'administration acceptée", () => {
    const etapes = [ETES.decisionDeLAutoriteAdministrative.ACCEPTE]
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-15', etapes)
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                                    (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RENDRE_DECISION_ADMINISTRATION_ACCEPTEE (publique      , accepté et publié      ) -> [PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS]",
        "[2022-04-16 -> 2023-04-16]",
      ]
    `)
  })

  test('peut rendre une décision acceptée au recueil des actes administratifs', () => {
    const etapes = [ETES.demande.FAIT, ETES.decisionDeLAutoriteAdministrative.ACCEPTE, ETES.publicationDeDecisionAuRecueilDesActesAdministratifs.FAIT]
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-14', etapes)
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                                                 (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE                                        (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RENDRE_DECISION_ADMINISTRATION_ACCEPTEE              (publique      , accepté et publié      ) -> [PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS]",
        "[2022-04-16 -> 2023-04-16]",
        "PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS (publique      , accepté et publié      ) -> [FAIRE_ABROGATION]",
        "[2022-04-16 -> 2023-04-16]",
      ]
    `)
  })

  test('peut classer la demande sans suite', () => {
    const etapes = [ETES.demande.FAIT, ETES.classementSansSuite.FAIT]
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-14', etapes)
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN               (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE      (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "CLASSER_SANS_SUITE (confidentielle, classé sans suite      ) -> []",
      ]
    `)
  })

  test('peut classer la demande sans suite après une ouverture de la consultation du public', () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-15', [ETES.demande.FAIT, ETES.consultationDuPublic.EN_COURS, ETES.classementSansSuite.FAIT])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                          (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE                 (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "OUVRIR_CONSULTATION_DU_PUBLIC (publique      , en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS]",
        "CLASSER_SANS_SUITE            (publique      , classé sans suite      ) -> []",
      ]
    `)
  })

  test('peut classer la demande sans suite après une ouverture de la consultation du public sur une prolongation', () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psProMachine, '2022-04-15', [ETES.demande.FAIT, ETES.consultationDuPublic.EN_COURS, ETES.classementSansSuite.FAIT])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                          (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE                 (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "[2022-04-16 -> null]",
        "OUVRIR_CONSULTATION_DU_PUBLIC (publique      , en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS]",
        "[2022-04-16 -> null]",
        "CLASSER_SANS_SUITE            (publique      , classé sans suite      ) -> []",
        "[2022-04-16 -> 2022-04-18]",
      ]
    `)
  })

  test('peut faire un désistement par le demandeur', () => {
    const etapes = [ETES.demande.FAIT, ETES.desistementDuDemandeur.FAIT]
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-14', etapes)
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                      (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE             (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "DESISTER_PAR_LE_DEMANDEUR (confidentielle, désisté                ) -> []",
      ]
    `)
  })

  test("peut faire une décision de l'autorité administrative avant le 1er juillet 2024", () => {
    const etapes = [ETES.decisionDeLAutoriteAdministrative.ACCEPTE, ETES.publicationDeDecisionAuJORF.FAIT]
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-14', etapes)
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                                    (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RENDRE_DECISION_ADMINISTRATION_ACCEPTEE (publique      , accepté et publié      ) -> [PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS]",
        "[2022-04-15 -> 2023-04-15]",
        "PUBLIER_DECISION_ACCEPTEE_AU_JORF       (publique      , accepté et publié      ) -> [FAIRE_ABROGATION]",
        "[2022-04-15 -> 2023-04-15]",
      ]
    `)
  })

  test("peut faire une décision de l'autorité administrative après le 1er juillet 2024", () => {
    const etapes = [ETES.decisionDeLAutoriteAdministrative.ACCEPTE]
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2024-07-14', etapes)
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                                    (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE]",
        "RENDRE_DECISION_ADMINISTRATION_ACCEPTEE (publique      , accepté                ) -> [PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS]",
        "[2024-07-15 -> 2025-07-15]",
      ]
    `)
  })

  test('peut faire un avis des collectivités', () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-15', [
      ETES.demande.FAIT,
      ETES.avisDesCollectivites.FAIT,
      ETES.avisDuPrefet.FAVORABLE,
      ETES.decisionDeLAutoriteAdministrative.ACCEPTE,
    ])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                                    (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE                           (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RENDRE_AVIS_COLLECTIVITES               (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_PREFET,RENDRE_AVIS_SERVICES_COMMISSIONS]",
        "RENDRE_AVIS_PREFET                      (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RENDRE_DECISION_ADMINISTRATION_ACCEPTEE (publique      , accepté et publié      ) -> [PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS]",
        "[2022-04-19 -> 2023-04-19]",
      ]
    `)
  })

  test('peut faire un avis des collectivités', () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-15', [
      ETES.demande.FAIT,
      ETES.avisDesServicesEtCommissionsConsultatives.FAIT,
      ETES.avisDuPrefet.FAVORABLE,
      ETES.decisionDeLAutoriteAdministrative.ACCEPTE,
    ])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                                    (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE                           (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RENDRE_AVIS_SERVICES_COMMISSIONS        (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_PREFET]",
        "RENDRE_AVIS_PREFET                      (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RENDRE_DECISION_ADMINISTRATION_ACCEPTEE (publique      , accepté et publié      ) -> [PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS]",
        "[2022-04-19 -> 2023-04-19]",
      ]
    `)
  })

  test('peut faire un desistement par le demandeur après une ouverture de la consultation du public', () => {
    const etapes = [ETES.demande.FAIT, ETES.consultationDuPublic.TERMINE, ETES.desistementDuDemandeur.FAIT]
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-14', etapes)
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                          (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE                 (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "OUVRIR_CONSULTATION_DU_PUBLIC (publique      , en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "DESISTER_PAR_LE_DEMANDEUR     (publique      , désisté                ) -> []",
      ]
    `)
  })

  test("peut rejeter une décision de l'administration puis un publication au JORF", () => {
    const etapes = [ETES.decisionDeLAutoriteAdministrative.REJETE, ETES.publicationDeDecisionAuJORF.FAIT]
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-14', etapes)
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                                   (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RENDRE_DECISION_ADMINISTRATION_REJETEE (confidentielle, rejeté                 ) -> [PUBLIER_DECISION_ACCEPTEE_AU_JORF]",
        "PUBLIER_DECISION_ACCEPTEE_AU_JORF      (confidentielle, rejeté                 ) -> []",
      ]
    `)
  })

  test("peut faire une demande d'information", () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-16', [ETES.demande.FAIT, ETES.demandeDinformations.FAIT])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                 (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE        (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "DEMANDER_INFORMATION (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
      ]
    `)
  })
  test("peut faire deux demandes d'information consécutives", () => {
    const etapes = [
      ETES.demande.FAIT,
      ETES.demandeDinformations.FAIT,
      ETES.receptionDinformation.FAIT,
      ETES.consultationDuPublic.TERMINE,
      ETES.demandeDinformations.FAIT,
      ETES.receptionDinformation.FAIT,
    ]
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-01', etapes)
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                          (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE                 (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "DEMANDER_INFORMATION          (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RECEVOIR_INFORMATION          (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "OUVRIR_CONSULTATION_DU_PUBLIC (publique      , en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "DEMANDER_INFORMATION          (publique      , en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RECEVOIR_INFORMATION          (publique      , en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
      ]
    `)
  })

  test('peut faire la démarche la plus complète possible', () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-14', [
      ETES.demande.FAIT,
      ETES.enregistrementDeLaDemande.FAIT,
      ETES.consultationDuPublic.TERMINE,
      ETES.decisionDeLAutoriteAdministrative.ACCEPTE,
      ETES.publicationDeDecisionAuJORF.FAIT,
      ETES.abrogationDeLaDecision.FAIT,
      ETES.publicationDeDecisionAuRecueilDesActesAdministratifs.FAIT,
    ])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                                                 (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE                                        (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "ENREGISTRER_DEMANDE                                  (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "OUVRIR_CONSULTATION_DU_PUBLIC                        (publique      , en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RENDRE_DECISION_ADMINISTRATION_ACCEPTEE              (publique      , accepté et publié      ) -> [PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS]",
        "[2022-04-18 -> 2023-04-18]",
        "PUBLIER_DECISION_ACCEPTEE_AU_JORF                    (publique      , accepté et publié      ) -> [FAIRE_ABROGATION]",
        "[2022-04-18 -> 2023-04-18]",
        "FAIRE_ABROGATION                                     (publique      , accepté et publié      ) -> [PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS]",
        "[2022-04-18 -> 2022-04-20]",
        "PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS (publique      , rejeté après abrogation) -> []",
        "[2022-04-18 -> 2022-04-20]",
      ]
    `)
  })

  test('peut abroger et publier', () => {
    const etapes = [ETES.decisionDeLAutoriteAdministrative.ACCEPTE, ETES.publicationDeDecisionAuJORF.FAIT, ETES.abrogationDeLaDecision.FAIT, ETES.publicationDeDecisionAuJORF.FAIT]
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-18', etapes)
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                                    (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RENDRE_DECISION_ADMINISTRATION_ACCEPTEE (publique      , accepté et publié      ) -> [PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS]",
        "[2022-04-19 -> 2023-04-19]",
        "PUBLIER_DECISION_ACCEPTEE_AU_JORF       (publique      , accepté et publié      ) -> [FAIRE_ABROGATION]",
        "[2022-04-19 -> 2023-04-19]",
        "FAIRE_ABROGATION                        (publique      , accepté et publié      ) -> [PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS]",
        "[2022-04-19 -> 2022-04-21]",
        "PUBLIER_DECISION_ACCEPTEE_AU_JORF       (publique      , rejeté après abrogation) -> []",
        "[2022-04-19 -> 2022-04-21]",
      ]
    `)
  })

  test("peut rejeter un décision de l'administration", () => {
    const etapes = [ETES.demande.FAIT, ETES.enregistrementDeLaDemande.FAIT, ETES.consultationDuPublic.TERMINE, ETES.decisionDeLAutoriteAdministrative.REJETE]
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-14', etapes)
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                                   (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "FAIRE_DEMANDE                          (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,ENREGISTRER_DEMANDE,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "ENREGISTRER_DEMANDE                    (confidentielle, en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,OUVRIR_CONSULTATION_DU_PUBLIC,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "OUVRIR_CONSULTATION_DU_PUBLIC          (publique      , en instruction         ) -> [CLASSER_SANS_SUITE,DEMANDER_INFORMATION,DESISTER_PAR_LE_DEMANDEUR,RECEVOIR_INFORMATION,RENDRE_AVIS_COLLECTIVITES,RENDRE_AVIS_SERVICES_COMMISSIONS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RENDRE_DECISION_ADMINISTRATION_REJETEE (confidentielle, rejeté                 ) -> []",
      ]
    `)
  })

  test("peut rejeter immédiatement une décision de l'administration", () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-08', [ETES.decisionDeLAutoriteAdministrative.REJETE])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                                   (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RENDRE_DECISION_ADMINISTRATION_REJETEE (confidentielle, rejeté                 ) -> [PUBLIER_DECISION_ACCEPTEE_AU_JORF]",
      ]
    `)
  })

  test("peut rejeter par décision implicite une décision de l'administration", () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '2022-04-08', [ETES.decisionDeLAutoriteAdministrative.REJETE_DECISION_IMPLICITE])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                                                      (confidentielle, en construction        ) -> [FAIRE_DEMANDE,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE]",
        "RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE (publique      , rejeté                 ) -> []",
      ]
    `)
  })

  test('peut publier au JORF sur les démarches très historiques', () => {
    const { tree } = setDateAndOrderAndInterpretMachine(psMachine, '1999-01-01', [ETES.publicationDeDecisionAuJORF.FAIT])
    expect(tree).toMatchInlineSnapshot(`
      [
        "RIEN                              (confidentielle, en construction        ) -> [FAIRE_DEMANDE,PUBLIER_DECISION_ACCEPTEE_AU_JORF,PUBLIER_DECISION_AU_RECUEIL_DES_ACTES_ADMINISTRATIFS,RENDRE_DECISION_ADMINISTRATION_ACCEPTEE,RENDRE_DECISION_ADMINISTRATION_REJETEE,RENDRE_DECISION_ADMINISTRATION_REJETEE_DECISION_IMPLICITE,SAISIR_INFORMATION_HISTORIQUE_INCOMPLETE]",
        "PUBLIER_DECISION_ACCEPTEE_AU_JORF (publique      , accepté et publié      ) -> [FAIRE_ABROGATION]",
        "[1999-01-02 -> 2000-01-02]",
      ]
    `)
  })
})
