import yaml
from git import Repo


def get_git_commit_info(repo_path):
    """
    Gets the commit info from the git repository.

    :param repo_path: The path to the git repository.
    :type repo_path: str
    :return: A dictionary containing:
        - 'branch': The name of the current branch.
        - 'SHA1': The SHA1 hash of the last commit.
        - 'message': The commit message.
        - 'author': The author of the commit.
        - 'date': The date of the commit.
    :rtype: dict
    """
    try:
        repo = Repo(repo_path)
        head_commit = repo.head.commit
        branch_name = repo.active_branch.name
        sha1_hash = head_commit.hexsha
    except Exception as e:
        print(f"Error getting commit info: {e}")
        return None

    return {
        'branch': branch_name,
        'SHA1': sha1_hash,
        'message': head_commit.message,
        'author': head_commit.author.name,
        'date': head_commit.committed_datetime.strftime("%Y-%m-%d %H:%M:%S")
    }


def write_commit_info_to_config(commit_info, config_file):
    """
    Write the commit info to the mkdocs.yml file

    :param commit_info: Information about the last commit
    :type commit_info: dict
    :param config_file: Configuration file for mkdocs
    :type config_file: str
    """
    try:
        with open(config_file, 'r+') as file:
            config = yaml.safe_load(file)
            config['extra']['commit_info'] = commit_info
            file.seek(0)  # Move the cursor to the start of the file
            yaml.dump(config, file)  # Convert the config to YAML and write it to the file
    except Exception as e:
        print(f"Error writing to config: {e}")


if __name__ == "__main__":
    repo_path = '.'
    config_file = 'mkdocs.yml'
    commit_info = get_git_commit_info(repo_path)
    if commit_info is not None:
        write_commit_info_to_config(commit_info, config_file)
