package org.mte.numecoeval.referentiel.domain.ports.input.impl;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVRecord;
import org.mte.numecoeval.referentiel.domain.data.ResultatImport;
import org.mte.numecoeval.referentiel.domain.exception.ReferentielException;
import org.mte.numecoeval.referentiel.domain.ports.input.ImportCSVReferentielPort;
import org.mte.numecoeval.referentiel.infrastructure.restapi.dto.HypotheseDTO;

import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

@Slf4j
public class ImportHypothesePortImpl implements ImportCSVReferentielPort<HypotheseDTO> {
    private static final String HEADER_VALEUR = "valeur";
    private static final String[] MANDATORY_HEADERS = new String[]{"cle", HEADER_VALEUR, "source"};

    private static final String[] HEADERS = new String[]{"cle", HEADER_VALEUR, "source", "description"};

    public static String[] getHeaders() {
        return HEADERS;
    }

    public void checkCSVRecord(CSVRecord csvRecord) throws ReferentielException {
        checkAllHeadersAreMapped(csvRecord, MANDATORY_HEADERS);
        checkFieldIsMappedAndNotBlankInCSVRecord(csvRecord, "cle");
        checkFieldIsMappedAndNotBlankInCSVRecord(csvRecord, HEADER_VALEUR);
    }

    @Override
    public ResultatImport<HypotheseDTO> importCSV(InputStream csvInputStream) {
        ResultatImport<HypotheseDTO> resultatImport = new ResultatImport<>();
        List<HypotheseDTO> dtos = new ArrayList<>();

        try (Reader reader = new InputStreamReader(csvInputStream)) {
            Iterable<CSVRecord> records = CSVFormat.DEFAULT.builder()
                    .setHeader()
                    .setDelimiter(CSV_SEPARATOR)
                    .setTrim(true)
                    .setAllowMissingColumnNames(true)
                    .setSkipHeaderRecord(true)
                    .build().parse(reader);
            records.forEach(csvRecord -> {
                try {
                    checkCSVRecord(csvRecord);
                    dtos.add(HypotheseDTO.builder()
                            .valeur(csvRecord.get(HEADER_VALEUR).trim())
                            .code(csvRecord.get("cle").trim())
                            .source(csvRecord.get("source").trim())
                            .description(getStringValueFromRecord(csvRecord, "description"))
                            .build());
                } catch (Exception e) {
                    log.error("Erreur prévue lors de la lecture de la ligne {} : {}", csvRecord.getRecordNumber() + 1, e.getMessage());
                    resultatImport.getErreurs().add(e.getMessage());
                }
            });

        } catch (Exception e) {
            log.error("Erreur de traitement du fichier", e);

            resultatImport.setErreurs(Collections.singletonList("Le fichier CSV n'a pas pu être lu."));
            resultatImport.setNbrLignesImportees(0);
            resultatImport.setObjects(null);
            return resultatImport;
        }

        resultatImport.setObjects(dtos);
        resultatImport.setNbrLignesImportees(dtos.size());

        return resultatImport;
    }
}
