import {
  Container,
  Heading,
  SkeletonText,
  Table,
  TableContainer,
  Tbody,
  Td,
  Th,
  Thead,
  Tr,
} from "@chakra-ui/react"
import { useQuery, useQueryClient } from "@tanstack/react-query"
import { createFileRoute, useNavigate } from "@tanstack/react-router"
import { useEffect } from "react"
import { z } from "zod"

import { CollectionsService } from "../../client"
import ActionsMenu from "../../components/Common/ActionsMenu"
import Navbar from "../../components/Common/Navbar"
import AddCollection from "../../components/Collections/AddCollection"
import { PaginationFooter } from "../../components/Common/PaginationFooter.tsx"

const collectionsSearchSchema = z.object({
  page: z.number().catch(1),
})

export const Route = createFileRoute("/_layout/collections")({
  component: Collections,
  validateSearch: (search) => collectionsSearchSchema.parse(search),
})

const PER_PAGE = 5

function getCollectionsQueryOptions({ page }: { page: number }) {
  return {
    queryFn: () =>
      CollectionsService.readCollections({ page: page, size: PER_PAGE }),
    queryKey: ["collections", { page }],
  }
}

function CollectionsTable() {
  const queryClient = useQueryClient()
  const { page } = Route.useSearch()
  const navigate = useNavigate({ from: Route.fullPath })
  const setPage = (page: number) =>
    navigate({ search: (prev: {[key: string]: string}) => ({ ...prev, page }) })

  const {
    data: collections,
    isPending,
    isPlaceholderData,
  } = useQuery({
    ...getCollectionsQueryOptions({ page }),
    placeholderData: (prevData) => prevData,
  })

  const hasNextPage = !isPlaceholderData && collections?.items.length === PER_PAGE // collections?.pages > page
  const hasPreviousPage = page > 1

  useEffect(() => {
    if (hasNextPage) {
      queryClient.prefetchQuery(getCollectionsQueryOptions({ page: page }))
    }
  }, [page, queryClient, hasNextPage])

  return (
    <>
      <TableContainer>
        <Table size={{ base: "sm", md: "md" }}>
          <Thead>
            <Tr>
              <Th>ID</Th>
              <Th>Title</Th>
              <Th>Description</Th>
              <Th>Type</Th>
              <Th>Owner</Th>
              <Th>Actions</Th>
            </Tr>
          </Thead>
          {isPending ? (
            <Tbody>
              <Tr>
                {new Array(6).fill(null).map((_, index) => (
                  <Td key={index}>
                    <SkeletonText noOfLines={1} paddingBlock="16px" />
                  </Td>
                ))}
              </Tr>
            </Tbody>
          ) : (
            <Tbody>
              {collections?.items.map((collection) => (
                <Tr key={collection.id} opacity={isPlaceholderData ? 0.5 : 1}>
                  <Td>{collection.id}</Td>
                  <Td isTruncated maxWidth="150px">
                    {collection.title}
                  </Td>
                  <Td
                    color={!collection.description ? "ui.dim" : "inherit"}
                    isTruncated
                    maxWidth="150px"
                  >
                    {collection.description || "N/A"}
                  </Td>
                  <Td>{collection.type}</Td>
                  <Td>{collection.owner}</Td>
                  <Td>
                    <ActionsMenu type={"Collection"} value={collection} disabled={!collection.can_edit}/>
                  </Td>
                </Tr>
              ))}
            </Tbody>
          )}
        </Table>
      </TableContainer>
      <PaginationFooter
        page={page}
        onChangePage={setPage}
        hasNextPage={hasNextPage}
        hasPreviousPage={hasPreviousPage}
      />
    </>
  )
}

function Collections() {
  return (
    <Container maxW="full">
      <Heading size="lg" textAlign={{ base: "center", md: "left" }} pt={12}>
        Collections Management
      </Heading>

      <Navbar type={"Collection"} addModalAs={AddCollection} />
      <CollectionsTable />
    </Container>
  )
}
