import contextlib
import logging
from collections.abc import Iterator
from dataclasses import dataclass
from datetime import datetime
from pathlib import Path
from typing import Any

import emails  # type: ignore
from jinja2 import Template

from app.core.config import settings
from app.core.s3 import client as s3_client
from app.models import Collection, Document, User

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


@dataclass
class EmailData:
    html_content: str
    subject: str


def render_email_template(*, template_name: str, context: dict[str, Any]) -> str:
    template_str = (Path(__file__).parent / "email-templates" / "build" / template_name).read_text()
    html_content = Template(template_str).render(context)
    return html_content


def send_email(
    *,
    email_to: str,
    subject: str = "",
    html_content: str = "",
) -> None:
    assert settings.emails_enabled, "no provided configuration for email variables"
    message = emails.Message(
        subject=subject,
        html=html_content,
        mail_from=(settings.EMAILS_FROM_NAME, settings.EMAILS_FROM_EMAIL),
    )
    smtp_options = {"host": settings.SMTP_HOST, "port": settings.SMTP_PORT}
    if settings.SMTP_TLS:
        smtp_options["tls"] = True
    elif settings.SMTP_SSL:
        smtp_options["ssl"] = True
    if settings.SMTP_USER:
        smtp_options["user"] = settings.SMTP_USER
    if settings.SMTP_PASSWORD:
        smtp_options["password"] = settings.SMTP_PASSWORD
    response = message.send(to=email_to, smtp=smtp_options)
    logger.info(f"send email result: {response}")


def generate_test_email(email_to: str) -> EmailData:
    project_name = settings.PROJECT_NAME
    subject = f"{project_name} - Test email"
    html_content = render_email_template(
        template_name="test_email.html",
        context={"project_name": settings.PROJECT_NAME, "email": email_to},
    )
    return EmailData(html_content=html_content, subject=subject)


def generate_new_account_email(email_to: str, username: str, password: str) -> EmailData:
    project_name = settings.PROJECT_NAME
    subject = f"{project_name} - New account for user {username}"
    html_content = render_email_template(
        template_name="new_account.html",
        context={
            "project_name": settings.PROJECT_NAME,
            "username": username,
            "password": password,
            "email": email_to,
            "link": settings.FRONTEND_HOST,
        },
    )
    return EmailData(html_content=html_content, subject=subject)


@contextlib.contextmanager
def user_permissions(user: User) -> Iterator[None]:
    """
    Attach _current_user to Collection and Document in order to determine permissions (can_read, can_edit)
    To be used as:

    with user_permissions(user):
        do_something

    WARNING: the values of the properties that rely on current_user can change if they are used outside the context.
    Thefore it is sometimes necessary to dump the model to keep the correct values (see crud.py)
    """
    Collection._current_user = user
    Document._current_user = user
    yield
    if hasattr(Collection, "_current_user"):  # for mypy
        del Collection._current_user
    if hasattr(Document, "_current_user"):  # for mypy
        del Document._current_user


def create_document_metadata(document: Document) -> None:
    """
    Create metadata.yaml in same bucket as document file. Needed by processing task
    """
    metadata = f"""
Nom du pdf: {document.filename}
'Nb pages ': {document.n_pages}
title: {document.title}
Auteur: Uknown
Date de publication: Uknown
publishing_date: "{datetime.now().year}"
doc_source: {document.collection.title}
url: {document.url}
date_avis_final: {datetime.now().year}
departement: Uknown
type_avis: Unkown
"""
    s3_client.CloudPath(document.path.parent / "metadata.yaml").write_text(metadata)
    return None
