import logging
import uuid

from sqlmodel import Session, select

from app import crud
from app.core.config import settings
from app.core.db import engine, init_db
from app.models import Collection, CollectionCreate, Group, User, UserCreate

GROUP_TABLE = [
    {"id": 1, "name": settings.DEFAULT_GROUP},
    {"id": 2, "name": "ADEME", "domain": "ademe.fr"},
    {"id": 3, "name": "CEREMA", "domain": "cerema.fr"},
    {"id": 4, "name": "MTE", "domain": "developpement-durable.gouv.fr"},
]

USER_TABLE = [
    {
        "id": uuid.uuid4(),
        "group_id": 1,
        "role": "superadmin",
        "email": settings.FIRST_SUPERUSER,
        "password": settings.FIRST_SUPERUSER_PASSWORD,
    },
    {"id": uuid.uuid4(), "group_id": 2, "role": "groupadmin", "email": "admin@ademe.fr", "password": "testthis"},
    {"id": uuid.uuid4(), "group_id": 2, "role": "contributor", "email": "contributor@ademe.fr", "password": "testthis"},
    {"id": uuid.uuid4(), "group_id": 2, "role": "user", "email": "user@ademe.fr", "password": "testthis"},
    {"id": uuid.uuid4(), "group_id": 3, "role": "groupadmin", "email": "admin@cerema.fr", "password": "testthis"},
]

COLLECTION_TABLE = [
    {"title": "Publique", "type": "public", "group_id": 1, "user_id": None},
    {"title": "ADEME publique", "type": "public", "group_id": 2, "user_id": None},
    {"title": "ADEME restreinte", "type": "restricted", "group_id": 2, "user_id": None},
    {"title": "Personal 1", "type": "personal", "group_id": 2, "user_id": USER_TABLE[3]["id"]},
    {"title": "Personal 2", "type": "personal", "group_id": 3, "user_id": USER_TABLE[4]["id"]},
]

# DOCUMENT_TABLE = [
#     {"id": uuid.uuid4(), "collection_id": 1},
#     {"id": uuid.uuid4(), "collection_id": 2},
# ]


logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


def create_test_data(session: Session) -> None:
    init_db(session)
    users_replace = {}
    for item in GROUP_TABLE[1:]:
        if not session.get(Group, item["id"]):
            crud.create_generic_item(session=session, model=Group, item=Group(**item))
    for idx, item in enumerate(USER_TABLE):
        if idx == 0:
            continue
        if not session.exec(select(User).where(User.email == item["email"])).first():
            user = crud.create_user(session=session, user_create=UserCreate(**item))
            users_replace[item["id"]] = user.id

    for item in COLLECTION_TABLE:
        # if not session.exec(select(Collection).where(Collection.type == item["t"]), item["id"]).first():
        crud.create_generic_item(
            session=session,
            model=Collection,
            item=CollectionCreate(**item),
            user_id=users_replace.get(item["user_id"]),
            group_id=item["group_id"],
        )


def main() -> None:
    logger.info("Creating test data")
    with Session(engine) as session:
        create_test_data(session)
    logger.info("Test data created")


if __name__ == "__main__":
    main()
