import uuid
from collections.abc import Generator
from typing import cast

import pytest
from fastapi.testclient import TestClient
from sqlmodel import Session

from app.core.config import settings
from app.core.db import engine, init_db
from app.main import app
from app.models import Group
from app.tests.test_data import GROUP_TABLE, USER_TABLE
from app.tests.utils.user import authentication_token_from_email
from app.tests.utils.utils import get_superuser_token_headers, random_lower_string


@pytest.fixture(scope="session", autouse=True)
def db() -> Generator[Session, None, None]:
    with Session(engine) as session:
        from sqlmodel import SQLModel

        SQLModel.metadata.create_all(engine)

        init_db(session)  # TODO: create groups and users
        yield session
        session.rollback()


@pytest.fixture(scope="module")
def client() -> Generator[TestClient, None, None]:
    with TestClient(app) as c:
        yield c


@pytest.fixture(scope="module")
def superuser_token_headers(client: TestClient) -> dict[str, str]:
    return get_superuser_token_headers(client)


@pytest.fixture(scope="module")
def normal_user_token_headers(client: TestClient, db: Session) -> dict[str, str]:
    return authentication_token_from_email(client=client, email=settings.EMAIL_TEST_USER, db=db)


@pytest.fixture(scope="function")
def group_session(session: Session) -> Generator[Session, None, None]:
    for entry in GROUP_TABLE:
        session.add(Group(name=random_lower_string(), **entry))
    session.commit()
    yield session
    session.rollback()


@pytest.fixture(scope="function")
def get_user_id(user_idx: int) -> uuid.UUID:
    return cast(uuid.UUID, USER_TABLE[user_idx]["id"])
