import uuid

from fastapi.testclient import TestClient
from sqlmodel import Session

from app.core.config import settings
from app.tests.utils.collection import create_random_collection


def test_create_collection(client: TestClient, superuser_token_headers: dict[str, str]) -> None:
    data = {"title": "Foo", "description": "Fighters"}
    response = client.post(
        f"{settings.API_V1_STR}/collections/",
        headers=superuser_token_headers,
        json=data,
    )
    assert response.status_code == 200
    content = response.json()
    assert content["title"] == data["title"]
    assert content["description"] == data["description"]
    assert "id" in content
    assert "user_id" in content


def test_create_duplicate_collection(client: TestClient, superuser_token_headers: dict[str, str], db: Session) -> None:
    collection = create_random_collection(db)
    data = {"title": collection.title}
    response = client.post(
        f"{settings.API_V1_STR}/collections/",
        headers=superuser_token_headers,
        json=data,
    )
    assert response.status_code == 400
    assert response.json() == {"detail": "A collection with the given name already exists"}


def test_read_collection(client: TestClient, superuser_token_headers: dict[str, str], db: Session) -> None:
    collection = create_random_collection(db)
    response = client.get(
        f"{settings.API_V1_STR}/collections/{collection.id}",
        headers=superuser_token_headers,
    )
    assert response.status_code == 200
    content = response.json()
    assert content["title"] == collection.title
    assert content["description"] == collection.description
    assert content["id"] == str(collection.id)
    assert content["user_id"] == str(collection.user_id)


def test_read_collection_not_found(client: TestClient, superuser_token_headers: dict[str, str]) -> None:
    response = client.get(
        f"{settings.API_V1_STR}/collections/{uuid.uuid4()}",
        headers=superuser_token_headers,
    )
    assert response.status_code == 404
    content = response.json()
    assert content["detail"] == "Collection not found"


def test_read_collection_not_enough_permissions(
    client: TestClient, normal_user_token_headers: dict[str, str], db: Session
) -> None:
    collection = create_random_collection(db)
    response = client.get(
        f"{settings.API_V1_STR}/collections/{collection.id}",
        headers=normal_user_token_headers,
    )
    assert response.status_code // 100 == 4
    content = response.json()
    assert content["detail"] == "The user doesn't have enough privileges"


def test_read_collections(client: TestClient, superuser_token_headers: dict[str, str], db: Session) -> None:
    create_random_collection(db)
    create_random_collection(db)
    response = client.get(
        f"{settings.API_V1_STR}/collections/",
        headers=superuser_token_headers,
    )
    assert response.status_code == 200
    content = response.json()
    assert len(content["items"]) >= 2


def test_update_collection(client: TestClient, superuser_token_headers: dict[str, str], db: Session) -> None:
    collection = create_random_collection(db)
    data = {"title": "Updated title", "description": "Updated description"}
    response = client.put(
        f"{settings.API_V1_STR}/collections/{collection.id}",
        headers=superuser_token_headers,
        json=data,
    )
    assert response.status_code == 200
    content = response.json()
    assert content["title"] == data["title"]
    assert content["description"] == data["description"]
    assert content["id"] == str(collection.id)
    assert content["user_id"] == str(collection.user_id)


def test_update_collection_not_found(client: TestClient, superuser_token_headers: dict[str, str]) -> None:
    data = {"title": "Updated title", "description": "Updated description"}
    response = client.put(
        f"{settings.API_V1_STR}/collections/{uuid.uuid4()}",
        headers=superuser_token_headers,
        json=data,
    )
    assert response.status_code == 404
    content = response.json()
    assert content["detail"] == "Collection not found"


def test_update_collection_not_enough_permissions(
    client: TestClient, normal_user_token_headers: dict[str, str], db: Session
) -> None:
    collection = create_random_collection(db)
    data = {"title": "Updated title", "description": "Updated description"}
    response = client.put(
        f"{settings.API_V1_STR}/collections/{collection.id}",
        headers=normal_user_token_headers,
        json=data,
    )
    assert response.status_code // 100 == 4
    content = response.json()
    assert content["detail"] == "The user doesn't have enough privileges"


def test_delete_collection(client: TestClient, superuser_token_headers: dict[str, str], db: Session) -> None:
    collection = create_random_collection(db)
    response = client.delete(
        f"{settings.API_V1_STR}/collections/{collection.id}",
        headers=superuser_token_headers,
    )
    assert response.status_code == 200
    content = response.json()
    assert content["message"] == "Collection deleted successfully"


def test_delete_collection_not_found(client: TestClient, superuser_token_headers: dict[str, str]) -> None:
    response = client.delete(
        f"{settings.API_V1_STR}/collections/{uuid.uuid4()}",
        headers=superuser_token_headers,
    )
    assert response.status_code == 404
    content = response.json()
    assert content["detail"] == "Collection not found"


def test_delete_collection_not_enough_permissions(
    client: TestClient, normal_user_token_headers: dict[str, str], db: Session
) -> None:
    collection = create_random_collection(db)
    response = client.delete(
        f"{settings.API_V1_STR}/collections/{collection.id}",
        headers=normal_user_token_headers,
    )
    assert response.status_code // 100 == 4
    content = response.json()
    assert content["detail"] == "The user doesn't have enough privileges"
