import uuid
from datetime import datetime
from enum import Enum
from typing import Any

from pydantic import BaseModel, EmailStr, computed_field
from sqlalchemy.dialects.postgresql import JSONB
from sqlmodel import Field, Relationship, SQLModel

from app.core.config import CollectionType as CollectionType  # "as" to explicitely define
from app.core.config import DomainName
from app.core.s3 import COLLECTIONS_BUCKET_NAME, client


# Enums
class UserRole(str, Enum):
    superadmin = "superadmin"
    groupadmin = "groupadmin"
    contributor = "contributor"
    user = "user"


class GroupBase(SQLModel):
    name: str = Field(unique=True, index=True, max_length=255)
    domain: DomainName | None = Field(default=None, unique=True, min_length=5, max_length=255)


class GroupUpdate(SQLModel):
    name: str | None


class GroupPublic(GroupBase):
    id: int


class Group(GroupBase, table=True):
    id: int | None = Field(default=None, primary_key=True)

    users: list["User"] = Relationship(back_populates="group")
    collections: list["Collection"] = Relationship(back_populates="group")


# Shared properties
class UserBase(SQLModel):
    email: EmailStr = Field(unique=True, index=True, max_length=255)
    is_active: bool = True
    role: UserRole = UserRole.user
    full_name: str | None = Field(default=None, max_length=255)
    sso: bool = False


# Properties to receive via API on creation
class UserCreate(UserBase):
    group_id: int
    password: str = Field(min_length=8, max_length=40)


class UserCreateSSO(UserBase):
    id: uuid.UUID
    sso: bool = True
    group_id: int
    hashed_password: str | None = Field(default=None)


class UserRegister(SQLModel):
    email: EmailStr = Field(max_length=255)
    password: str = Field(min_length=8, max_length=40)
    full_name: str | None = Field(default=None, max_length=255)
    group_id: int


# Properties to receive via API on update, all are optional
class UserUpdate(UserBase):
    email: EmailStr | None = Field(default=None, max_length=255)  # type: ignore
    password: str | None = Field(default=None, min_length=8, max_length=40)
    group_id: int | None = None


class UserUpdateMe(SQLModel):
    full_name: str | None = Field(default=None, max_length=255)
    email: EmailStr | None = Field(default=None, max_length=255)


class UpdatePassword(SQLModel):
    current_password: str = Field(min_length=8, max_length=40)
    new_password: str = Field(min_length=8, max_length=40)


# Database model, database table inferred from class name
class User(UserBase, table=True):
    id: uuid.UUID = Field(default_factory=uuid.uuid4, primary_key=True)
    hashed_password: str | None
    group_id: int = Field(foreign_key="group.id", nullable=False, ondelete="CASCADE")
    group: Group = Relationship(back_populates="users")

    collections: list["Collection"] = Relationship(back_populates="user", cascade_delete=True)


# Properties to return via API, id is always required
class UserPublic(UserBase):
    id: uuid.UUID
    group_id: int


# Shared properties
class CollectionBase(SQLModel):
    title: str = Field(unique=True, min_length=1, max_length=255)
    description: str | None = Field(default=None, max_length=255)
    type: CollectionType = Field(default=CollectionType.personal)


# Properties to receive on collection creation
class CollectionCreate(CollectionBase):
    pass


# Properties to receive on collection update
class CollectionUpdate(CollectionBase):
    title: str | None = Field(default=None, min_length=1, max_length=255)  # type: ignore


# Properties to return via API, id is always required
class CollectionPublic(CollectionBase):
    id: uuid.UUID
    created_at: datetime
    user_id: uuid.UUID | None
    group_id: int
    owner: str
    can_edit: bool | None = None


# Database model, database table inferred from class name
class Collection(CollectionBase, table=True):
    id: uuid.UUID = Field(default_factory=uuid.uuid4, primary_key=True)
    created_at: datetime = Field(default_factory=datetime.utcnow, nullable=False)
    user_id: uuid.UUID | None = Field(foreign_key="user.id", ondelete="CASCADE")
    group_id: int = Field(foreign_key="group.id", nullable=False, ondelete="CASCADE")

    user: User | None = Relationship(back_populates="collections")
    group: Group | None = Relationship(back_populates="collections")
    documents: list["Document"] = Relationship(back_populates="collection")

    @computed_field  # type: ignore[prop-decorator]
    @property
    def owner(self) -> str:
        if self.user_id is None and self.group is not None:
            return self.group.name
        if self.user is not None:
            return self.user.full_name or self.user.email
        return "?"

    @computed_field  # type: ignore[prop-decorator]
    @property
    def can_read(self) -> bool | None:
        """
        True (False) if collection can(not) be read by current user (defined dynamically)
        or None if user info not available
        """
        if not hasattr(self, "_current_user") or self._current_user is None:
            return None
        return bool(
            self.type == CollectionType.public
            or self._current_user.role == UserRole.superadmin
            or self.user_id == self._current_user.id  # personal
            or (not self.type == CollectionType.personal and self.group_id == self._current_user.group_id)
        )

    @computed_field  # type: ignore[prop-decorator]
    @property
    def can_edit(self) -> bool | None:
        """
        True (False) if collection can(not) be edited or deleted by current user (defined dynamically)
        or None if user info not available
        """
        if not hasattr(self, "_current_user") or self._current_user is None:
            return None
        return bool(
            self._current_user.role == UserRole.superadmin
            or (self._current_user.role == UserRole.groupadmin and self.group_id == self._current_user.group_id)
            or self.user_id == self._current_user.id
        )


class DocumentCreate(SQLModel):
    collection_id: uuid.UUID
    title: str | None = Field(default=None, description="Document title (initially from pdf metadata if present)")
    meta: dict[str, Any] | None = Field(default=None)
    # Does not work with HttpUrl or AnyUrl: https://github.com/fastapi/sqlmodel/discussions/730
    # url: AnyUrl | None = Field(default=None, sa_type=AutoString)
    # url: AnyUrl | None = Field(default=None)
    url: str | None = Field(default=None)
    filename: str = Field(description="Initial filename when uploaded or downloaded")
    n_pages: int | None = Field(default=None, description="Number of pages")
    content_length: int = Field(description="File size in bytes")
    # docengine_id: str | None = None


class Document(DocumentCreate, table=True):
    id: uuid.UUID = Field(default_factory=uuid.uuid4, primary_key=True)
    meta: dict[str, Any] | None = Field(default=None, sa_type=JSONB)
    comments: dict[str, Any] | None = Field(
        default=None,
        sa_type=JSONB,
        description="Instant JSON comments given by PSPDFKit (see https://www.nutrient.io/guides/web/json/schema/comments/#comments)",
    )
    processing_task_id: uuid.UUID | None = Field(default=None)

    collection_id: uuid.UUID = Field(foreign_key="collection.id", nullable=False, ondelete="CASCADE")
    collection: Collection = Relationship(back_populates="documents")
    created_at: datetime = Field(default_factory=datetime.utcnow, nullable=False)
    updated_at: datetime = Field(default_factory=datetime.utcnow, nullable=False)

    @computed_field  # type: ignore[prop-decorator]
    @property
    def can_read(self) -> bool | None:
        """
        True (False) if document can(not) be edited or deleted by current user (defined dynamically)
        or None if user info not available
        """
        return self.collection.can_read

    @computed_field  # type: ignore[prop-decorator]
    @property
    def can_edit(self) -> bool | None:
        """
        True (False) if collection can(not) be edited or deleted by current user (defined dynamically)
        or None if user info not available
        """
        if not hasattr(self, "_current_user") or self._current_user is None:
            return None
        return bool(
            self._current_user.role == UserRole.superadmin
            or (
                self._current_user.role in (UserRole.groupadmin, UserRole.contributor)
                and self.collection.group_id == self._current_user.group_id
            )
            or self.collection.user_id == self._current_user.id
        )

    @property
    def path(self) -> Any:
        return client.CloudPath(f"s3://{COLLECTIONS_BUCKET_NAME}/{self.collection.title}/{self.id}/raw_document.pdf")


class DocumentUpdate(SQLModel):
    title: str | None
    meta: dict[str, Any] | None
    comments: dict[str, Any] | None


class DocumentPublic(DocumentCreate):
    id: uuid.UUID
    created_at: datetime
    updated_at: datetime
    processing_task_id: uuid.UUID | None
    can_edit: bool | None


# Generic message
class Message(SQLModel):
    message: str


# JSON payload containing access token
class Token(SQLModel):
    access_token: str
    token_type: str = "bearer"


# Contents of JWT token
class TokenPayload(SQLModel):
    sub: str | None = None


class NewPassword(SQLModel):
    token: str
    new_password: str = Field(min_length=8, max_length=40)


class ProcessingStatus(SQLModel):
    status: str | None
    details: dict[str, Any] | str | None = None


# class SearchRequest(BaseModel):
#     query: str
#     date: list[int]
#     source: list[str]
#     type_avis_cgdd: list[str] | None = []
#     region_cgdd: list[str] | None = []


# class SearchResponse(BaseModel):
#     cards: list[dict]
#     filter_visible: bool


class Metadata(BaseModel):
    title: str | None = None


class HistoryItem(BaseModel):
    role: str
    content: str
    metadata: Metadata | None = None


class ChatMessage(BaseModel):
    query: str
    collection_ids: list[uuid.UUID]


class SearchRequest(BaseModel):
    qdrant_collection_name: str
    query: str
    sources: list[str] = Field(min_length=1)
    dates: list[int] = []
    type_avis: list[str] = []
    region: list[str] = []


class MessageRequest(SearchRequest):
    history: list[dict[str, str]] = []
    audience: str = "experts"
