import uuid
from typing import Any, cast

from fastapi import HTTPException
from pydantic import EmailStr
from sqlmodel import Session, SQLModel, select

from app.core.config import settings
from app.core.security import get_password_hash, verify_password
from app.models import (
    Group,
    Message,
    User,
    UserCreate,
    UserUpdate,
)


def create_user(*, session: Session, user_create: UserCreate) -> User:
    user = create_generic_item(
        session=session, model=User, item=user_create, hashed_password=get_password_hash(user_create.password)
    )
    return cast(User, user)


def update_user(*, session: Session, db_user: User, user_in: UserUpdate) -> Any:
    user_data = user_in.model_dump(exclude_unset=True)
    extra_data = {}
    if "password" in user_data:
        password = user_data["password"]
        hashed_password = get_password_hash(password)
        extra_data["hashed_password"] = hashed_password
    db_user.sqlmodel_update(user_data, update=extra_data)
    session.add(db_user)
    session.commit()
    session.refresh(db_user)
    return db_user


def get_user_by_email(*, session: Session, email: str) -> User | None:
    statement = select(User).where(User.email == email)
    session_user = session.exec(statement).first()
    return session_user


def authenticate(*, session: Session, email: str, password: str) -> User | None:
    db_user = get_user_by_email(session=session, email=email)
    if not db_user:
        return None
    if not db_user.hashed_password:  # sso login
        return None
    if not verify_password(password, db_user.hashed_password):
        return None
    return db_user


# Generic CRUD methods
def create_generic_item(session: Session, model: type[SQLModel], item: SQLModel, **kwargs: Any) -> SQLModel:
    db_item = model.model_validate(item, update=kwargs)
    session.add(db_item)
    session.commit()
    session.refresh(db_item)
    return db_item


def read_generic_item(session: Session, model: type[SQLModel], item_id: int | uuid.UUID) -> SQLModel:
    item = session.get(model, item_id)
    if not item:
        raise HTTPException(status_code=404, detail=f"{model.__name__} not found")
    if hasattr(item, "can_read") and not item.can_read:
        raise HTTPException(status_code=403, detail="The user doesn't have enough privileges")
    return item


def update_generic_item(
    session: Session, model: type[SQLModel], item_id: int | uuid.UUID, item_update: SQLModel
) -> SQLModel:
    db_item = session.get(model, item_id)
    if not db_item:
        raise HTTPException(status_code=404, detail=f"{model.__name__} not found")
    if hasattr(db_item, "can_edit") and not db_item.can_edit:
        raise HTTPException(status_code=403, detail="The user doesn't have enough privileges")

    # Convert update model to dict, removing None values
    update_data = item_update.model_dump(exclude_unset=True)

    # Update item attributes
    db_item.sqlmodel_update(update_data)

    session.add(db_item)
    session.commit()
    session.refresh(db_item)
    return db_item


def delete_generic_item(session: Session, model: type[SQLModel], item_id: int | uuid.UUID) -> Message:
    item = session.get(model, item_id)
    if not item:
        raise HTTPException(status_code=404, detail=f"{model.__name__} not found")
    if hasattr(item, "can_edit") and not item.can_edit:
        raise HTTPException(status_code=403, detail="The user doesn't have enough privileges")
    session.delete(item)
    session.commit()
    return Message(message=f"{model.__name__} deleted successfully")


def get_group_from_email(session: Session, email: EmailStr) -> int | None:
    """
    Return the id of the group where to add the user with the given email (or None)
    """
    domain = email.split("@")[1]
    statement = select(Group)
    groups = session.exec(statement).all()
    for group in groups:
        if group.name == settings.DEFAULT_GROUP:
            continue
        if group.domain and group.id and domain in group.domain:
            return group.id
    return None
