from datetime import datetime, timedelta, timezone
from typing import Any

import jwt
import requests
from fastapi_sso.sso.base import DiscoveryDocument, OpenID, SSOBase
from fastapi_sso.sso.generic import create_provider
from httpx import AsyncClient
from passlib.context import CryptContext
from pydantic import HttpUrl, TypeAdapter
from starlette.datastructures import URL

from app.core.config import settings

pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")


ALGORITHM = "HS256"


type_adapter = TypeAdapter(DiscoveryDocument)


def create_access_token(subject: str | Any, expires_delta: timedelta) -> str:
    expire = datetime.now(timezone.utc) + expires_delta
    to_encode = {"exp": expire, "sub": str(subject)}
    encoded_jwt = jwt.encode(to_encode, settings.SECRET_KEY, algorithm=ALGORITHM)
    return encoded_jwt


def verify_password(plain_password: str, hashed_password: str) -> bool:
    return pwd_context.verify(plain_password, hashed_password)


def get_password_hash(password: str) -> str:
    return pwd_context.hash(password)


def convert_user(response: dict[str, Any], _client: AsyncClient | None) -> OpenID:
    """Convert user information returned by OIDC"""
    return OpenID(
        id=response["sub"],
        email=response["email"],
        display_name=response["name"],
        first_name=response["given_name"],
        last_name=response["family_name"],
    )


def get_discovery_document(url: str) -> DiscoveryDocument:
    response = requests.get(url)
    assert response.status_code == 200, response.text
    return type_adapter.validate_python(response.json())


def get_sso(redirect_url: HttpUrl, base_url: URL) -> SSOBase:
    assert settings.OPENID_CONFIG_URL, "OpenID config url not provided"
    assert settings.KEYCLOAK_CLIENT_ID, "Keycloak client ID not provided"
    discovery_document = get_discovery_document(url=settings.OPENID_CONFIG_URL)
    GenericSSO = create_provider(name="oidc", discovery_document=discovery_document, response_convertor=convert_user)
    return GenericSSO(
        client_id=settings.KEYCLOAK_CLIENT_ID,
        client_secret="",
        redirect_uri=f"{base_url}api/v1/login/sso/callback?redirect_url={redirect_url}",  # TODO: better way ?
        allow_insecure_http=True,
    )
