import urllib
from datetime import datetime, timedelta, timezone
from typing import Annotated, Any, cast

from fastapi import APIRouter, Depends, HTTPException
from fastapi.security import OAuth2PasswordRequestForm
from pydantic import HttpUrl
from starlette.requests import Request
from starlette.responses import RedirectResponse

from app import crud
from app.api.deps import CurrentUser, SessionDep
from app.core import security
from app.core.config import settings
from app.core.security import get_sso
from app.models import Token, User, UserCreateSSO, UserPublic

router = APIRouter()


@router.post("/login/access-token")
def login_access_token(session: SessionDep, form_data: Annotated[OAuth2PasswordRequestForm, Depends()]) -> Token:
    """
    OAuth2 compatible token login, get an access token for future requests
    """
    user = crud.authenticate(session=session, email=form_data.username, password=form_data.password)
    if not user:
        raise HTTPException(status_code=400, detail="Incorrect email or password")
    elif not user.is_active:
        raise HTTPException(status_code=400, detail="Inactive user")
    access_token_expires = timedelta(minutes=settings.ACCESS_TOKEN_EXPIRE_MINUTES)
    return Token(access_token=security.create_access_token(user.id, expires_delta=access_token_expires))


@router.post("/login/test-token", response_model=UserPublic)
def test_token(current_user: CurrentUser) -> Any:
    """
    Test access token
    """
    return current_user


@router.get("/login/sso")
async def sso_login(request: Request, redirect_url: HttpUrl) -> RedirectResponse:
    """Generate login url and redirect"""
    async with get_sso(redirect_url, request.base_url) as sso:
        return await sso.get_login_redirect()


@router.get("/login/sso/callback")
async def sso_callback(request: Request, session: SessionDep) -> RedirectResponse:
    """Process login response from OIDC and redirect to requested url"""
    try:
        redirect_url = request.query_params.get("redirect_url")
        if not redirect_url:
            raise HTTPException(status_code=400, detail="Missing redirect_url")

        # TODO: validate redirect_url ?

        async with get_sso(HttpUrl(url=redirect_url), request.base_url) as sso:
            user = await sso.verify_and_process(request)
        if user is None or user.email is None:
            raise HTTPException(401, "Failed to fetch user information")

        # Fetch user in db; if not present, add it to the corresponding group
        db_user = crud.get_user_by_email(session=session, email=user.email)
        if not db_user:
            group_id = crud.get_group_from_email(session=session, email=user.email)
            if group_id is None:
                raise HTTPException(status_code=403, detail=f"User or domain not recognised: {user.email}")
            user_sso = UserCreateSSO(
                id=user.id, email=user.email, full_name=user.display_name, group_id=group_id, hashed_password=None
            )
            db_user = cast(User, crud.create_generic_item(session=session, model=User, item=user_sso))

        access_token_expires = timedelta(minutes=settings.ACCESS_TOKEN_EXPIRE_MINUTES)
        token = security.create_access_token(db_user.id, expires_delta=access_token_expires)
        # Construct redirect URL with token because if front and back are on differents domains cookie won't be accepted be client
        url_with_token = f"{redirect_url}?token={token}"
        response = RedirectResponse(url=url_with_token)
        # The best way to store data in a request response would be with cookies or with url parameters
        # As suggested here : https://tomasvotava.github.io/fastapi-sso/how-to-guides/use-with-fastapi-security/
        # We use cookie
        response.set_cookie(key="token", value=token, expires=datetime.now(timezone.utc) + access_token_expires)
        return response
    except Exception as e:
        # Handle authentication errors
        error_redirect = f"{redirect_url}?error={urllib.parse.quote(str(e))}"
        return RedirectResponse(url=error_redirect)
