import uuid
from typing import Any

from fastapi import APIRouter, HTTPException
from fastapi_pagination import Page
from fastapi_pagination.ext.sqlalchemy import paginate
from sqlmodel import select

from app import crud
from app.api.deps import SessionDep
from app.models import Group, GroupBase, GroupPublic, GroupUpdate, Message

router = APIRouter()


@router.get("/", response_model=Page[GroupPublic])
def read_groups(session: SessionDep) -> Any:
    """
    Retrieve groups.
    """
    return paginate(session, select(Group))


@router.post("/", response_model=GroupPublic)
def create_group(*, session: SessionDep, group_in: GroupBase) -> Any:
    """
    Create new group.
    """
    statement = select(Group).where(Group.name == group_in.name)
    group = session.exec(statement).first()
    if group:
        raise HTTPException(
            status_code=400,
            detail="The group with this name already exists in the system.",
        )
    return crud.create_generic_item(session=session, model=Group, item=group_in)


@router.get("/{group_id}", response_model=GroupPublic)
def read_group_by_id(group_id: uuid.UUID, session: SessionDep) -> Any:
    """
    Get a specific group by id.
    """
    return session.get(Group, group_id)


@router.patch(
    "/{group_id}",
    response_model=GroupPublic,
)
def update_group(
    *,
    session: SessionDep,
    group_id: uuid.UUID,
    group_in: GroupUpdate,
) -> Any:
    """
    Update a group.
    """
    return crud.update_generic_item(session=session, model=Group, item_id=group_id, item_update=group_in)


@router.delete("/{group_id}")
def delete_group(session: SessionDep, group_id: uuid.UUID) -> Message:
    """
    Delete a group.
    """
    group = session.get(Group, group_id)
    if not group:
        raise HTTPException(status_code=404, detail="Group not found")
    session.delete(group)
    session.commit()
    return Message(message="Group deleted successfully")
