import uuid
from typing import Any, cast

from celery.result import AsyncResult
from fastapi import APIRouter, HTTPException, status
from fastapi.responses import FileResponse, Response
from fastapi_pagination import Page
from fastapi_pagination.ext.sqlalchemy import paginate
from sqlmodel import select

from app import crud
from app.api.deps import CurrentSuperAdmin, CurrentUser, CurrentUserWithQuery, SessionDep
from app.core.celery import celery_app
from app.core.config import settings
from app.models import (
    Collection,
    Document,
    DocumentCreate,
    DocumentPublic,
    DocumentUpdate,
    Message,
    ProcessingStatus,
    UserRole,
)
from app.utils import create_document_metadata, user_permissions

router = APIRouter()


@router.get("/", response_model=Page[DocumentPublic])
def read_documents(session: SessionDep, current_user: CurrentSuperAdmin) -> Any:
    """
    Retrieve documents.
    """
    with user_permissions(current_user):
        return paginate(session, select(Document))


@router.post("/", response_model=DocumentPublic)
def create_document(*, session: SessionDep, current_user: CurrentUser, document_in: DocumentCreate) -> Any:
    """
    Create new document.
    """
    with user_permissions(current_user):
        collection = cast(
            Collection, crud.read_generic_item(session=session, model=Collection, item_id=document_in.collection_id)
        )
        if not (collection.can_read and (collection.can_edit or current_user.role == UserRole.contributor)):
            raise HTTPException(status_code=403, detail="The user doesn't have enough privileges")
        return crud.create_generic_item(session=session, model=Document, item=document_in).model_dump()


@router.get("/{id}", response_model=DocumentPublic)
def read_document(session: SessionDep, current_user: CurrentUser, id: uuid.UUID) -> Any:
    """
    Get document by ID.
    """
    with user_permissions(current_user):
        return crud.read_generic_item(session=session, model=Document, item_id=id).model_dump()


# We need to craft our HEAD response in order to be compatible with PSPDFKit expectations
# And since the GET function is dynamic FastAPI could not guess the content length
# https://github.com/fastapi/fastapi/issues/1773 helped a bit
@router.head("/{id}/file")
def get_head(session: SessionDep, current_user: CurrentUserWithQuery, id: uuid.UUID) -> Any:
    with user_permissions(current_user):
        doc = cast(Document, crud.read_generic_item(session=session, model=Document, item_id=id))
        if doc:
            res = Response(media_type="application/pdf")
            res.headers["Accept-ranges"] = "bytes"
            res.headers["Content-Length"] = str(doc.content_length)
            res.headers["Content-Range"] = "bytes 0-" + str(doc.content_length - 1) + "/" + str(doc.content_length)
            res.status_code = status.HTTP_206_PARTIAL_CONTENT
            return res
        else:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="Aucun fichier avec correspond à l'identifiant",
            )


@router.get("/{id}/file")
async def get_file(session: SessionDep, current_user: CurrentUserWithQuery, id: uuid.UUID) -> Any:
    with user_permissions(current_user):
        doc = cast(Document, crud.read_generic_item(session=session, model=Document, item_id=id))
        if doc:
            return FileResponse(doc.path)
        else:
            raise HTTPException(
                status_code=status.HTTP_404_NOT_FOUND,
                detail="Aucun fichier avec correspond à l'identifiant",
            )


@router.put("/{id}", response_model=DocumentPublic)
def update_document(
    *,
    session: SessionDep,
    current_user: CurrentUser,
    id: uuid.UUID,
    document_in: DocumentUpdate,
) -> Any:
    """
    Update an document.
    """
    with user_permissions(current_user):
        return crud.update_generic_item(
            session=session, model=Document, item_update=document_in, item_id=id
        ).model_dump()


@router.delete("/{id}")
def delete_document(session: SessionDep, current_user: CurrentUser, id: uuid.UUID) -> Message:
    """
    Delete an document.
    """
    with user_permissions(current_user):
        return crud.delete_generic_item(session=session, model=Document, item_id=id)

        # TODO: delete s3 path with file and metadata.yml if present; send signal to delete vectors


@router.post("/{id}/process")
def process_document(session: SessionDep, current_user: CurrentUser, id: uuid.UUID, rerun: bool = False) -> Message:
    """
    Process a document: send a celery task and register task id in db
    """
    with user_permissions(current_user):
        document: Document = cast(Document, crud.read_generic_item(session=session, model=Document, item_id=id))
        if not document.can_edit:
            raise HTTPException(status_code=403, detail="The user doesn't have enough privileges")
        if document.processing_task_id and not rerun:
            raise HTTPException(
                status_code=status.HTTP_400_BAD_REQUEST, detail="Document already processed and rerun not requested"
            )
        if celery_app is None:
            raise HTTPException(
                status_code=status.HTTP_503_SERVICE_UNAVAILABLE, detail="Document processing unavailable"
            )

        # Create metadata.yaml
        create_document_metadata(document)

        # Send task
        task_id = uuid.uuid4()
        qdrant_collection_name = settings.QDRANT_COLLECTION_NAMES[document.collection.type]
        kwargs = {
            "document_id": str(document.id),
            "collection_name": document.collection.title,
            "qdrant_collection_name": qdrant_collection_name,
        }
        celery_app.send_task("sofia.process_document", kwargs=kwargs, task_id=str(task_id))

        # Register task id in db
        document.processing_task_id = task_id
        session.add(document)
        session.commit()

        return Message(message="Task created")


@router.get("/{id}/status")
def get_processing_status(session: SessionDep, current_user: CurrentUser, id: uuid.UUID) -> ProcessingStatus:
    """
    Get status of document processing
    """
    with user_permissions(current_user):
        document: Document = cast(Document, crud.read_generic_item(session=session, model=Document, item_id=id))
        if celery_app is None or document.processing_task_id is None:
            return ProcessingStatus(status=None)
        result: AsyncResult[None] = AsyncResult(str(document.processing_task_id), backend=celery_app.backend)
        return ProcessingStatus(status=result.status, details=result.info)
