import uuid
from io import BytesIO
from typing import Any, cast

import pikepdf
from fastapi import APIRouter, HTTPException, UploadFile, status
from fastapi_pagination import Page
from fastapi_pagination.ext.sqlalchemy import paginate
from sqlalchemy import and_, or_
from sqlmodel import select

from app import crud
from app.api.deps import CurrentUser, SessionDep
from app.models import (
    Collection,
    CollectionCreate,
    CollectionPublic,
    CollectionType,
    CollectionUpdate,
    Document,
    DocumentCreate,
    DocumentPublic,
    Message,
    UserRole,
)
from app.utils import user_permissions

router = APIRouter()


@router.get("/", response_model=Page[CollectionPublic])
def read_collections(session: SessionDep, current_user: CurrentUser) -> Any:
    """
    Retrieve collections.
    """
    with user_permissions(current_user):
        if current_user.role == UserRole.superadmin:
            statement = select(Collection)
        elif current_user.role == UserRole.groupadmin:
            statement = select(Collection).where(
                or_(Collection.type == CollectionType.public, Collection.group_id == current_user.group_id)  # type: ignore[arg-type]
            )
        else:
            statement = select(Collection).where(
                or_(
                    Collection.type == CollectionType.public,  # type: ignore[arg-type]
                    Collection.user_id == current_user.id,  # type: ignore[arg-type]
                    and_(Collection.type == CollectionType.restricted, Collection.group_id == current_user.group_id),  # type: ignore[arg-type]
                )
            )
        return paginate(session, statement)


@router.get("/{id}", response_model=CollectionPublic)
def read_collection(session: SessionDep, current_user: CurrentUser, id: uuid.UUID) -> Any:
    """
    Get collection by ID.
    """
    with user_permissions(current_user):
        return crud.read_generic_item(session=session, model=Collection, item_id=id)  # .model_dump()


@router.post("/", response_model=CollectionPublic)
def create_collection(*, session: SessionDep, current_user: CurrentUser, collection_in: CollectionCreate) -> Any:
    """
    Create new collection.
    """
    with user_permissions(current_user):
        if (
            current_user.role not in (UserRole.superadmin, UserRole.groupadmin)
            and collection_in.type != CollectionType.personal
        ):
            raise HTTPException(status_code=403, detail="The user doesn't have enough privileges")

        # Check if collection with the same title does not exist already
        statement = select(Collection).where(Collection.title == collection_in.title)
        if session.exec(statement).first() is not None:
            raise HTTPException(status_code=400, detail="A collection with the given name already exists")

        return crud.create_generic_item(
            session=session,
            model=Collection,
            item=collection_in,
            user_id=current_user.id if collection_in.type == CollectionType.personal else None,
            group_id=current_user.group_id,
        ).model_dump()


@router.put("/{id}", response_model=CollectionPublic)
def update_collection(
    *,
    session: SessionDep,
    current_user: CurrentUser,
    id: uuid.UUID,
    collection_in: CollectionUpdate,
) -> Any:
    """
    Update an collection.
    """
    with user_permissions(current_user):
        return crud.update_generic_item(
            session=session, model=Collection, item_update=collection_in, item_id=id
        ).model_dump()


@router.delete("/{id}")
def delete_collection(session: SessionDep, current_user: CurrentUser, id: uuid.UUID) -> Message:
    """
    Delete an collection.
    """
    with user_permissions(current_user):
        return crud.delete_generic_item(session=session, model=Collection, item_id=id)


@router.get("/{id}/documents", response_model=Page[DocumentPublic])
def read_documents(session: SessionDep, current_user: CurrentUser, id: uuid.UUID) -> Any:
    # Fetch collection to make sure the user can read it
    read_collection(session=session, current_user=current_user, id=id)
    with user_permissions(current_user):
        return paginate(session, select(Document).where(Document.collection_id == id))


def handle_pdf_upload(session: Any, collection_id: uuid.UUID, file: UploadFile) -> Document:
    """
    Save pdf data to the document table and pdf content to a S3 bucket.
    Linearize the document before if it's not already linearized.
    """
    # TODO: get n_pages
    try:
        doc = pikepdf.open(filename_or_stream=BytesIO(file.file.read()))
        should_linearize = not doc.is_linearized
        if doc.is_linearized and not doc.check_linearization():
            should_linearize = True
            print("Document content incorrect linearization data")
        # I was unable to work with metadata on PDF < 2.0
        # meta = doc.open_metadata()
        # doc_title = meta["dc:title"]
        doc_title = ""
        if "/Title" in doc.docinfo:
            doc_title = str(doc.docinfo["/Title"])
        # Linearize document on the fly
        if should_linearize:
            print(f"Document '{file.filename}' not linearized, convert it to a streamable format")
            stream = BytesIO()
            doc.save(filename_or_stream=stream, linearize=True)
            file.file = stream
            stream.seek(0)
            # @todo find a way to avoid the need to re read the stream in order to get the new size
            file.size = len(stream.read())
        # Important because minio needs it after
        file.file.seek(0)
    except Exception as exc:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Erreur à l'ouverture du document pdf",
        ) from exc
    try:
        new_doc = DocumentCreate(
            collection_id=collection_id,
            filename=file.filename,
            content_length=file.size,
            n_pages=len(doc.pages),
            title=doc_title,
        )
        created_doc = cast(Document, crud.create_generic_item(session=session, model=Document, item=new_doc))
        created_doc.path.write_bytes(file.file.read())
        return created_doc
    except Exception as exc:
        raise HTTPException(
            status_code=status.HTTP_400_BAD_REQUEST,
            detail="Erreur lors de la sauvegarde du document pdf",
        ) from exc


@router.post("/{id}/document", response_model=list[DocumentPublic])
async def create_document(
    *, session: SessionDep, current_user: CurrentUser, id: uuid.UUID, files: list[UploadFile]
) -> Any:
    with user_permissions(current_user):
        collection = cast(Collection, crud.read_generic_item(session=session, model=Collection, item_id=id))
        if not (collection.can_read and (collection.can_edit or current_user.role == UserRole.contributor)):
            raise HTTPException(status_code=403, detail="The user doesn't have enough privileges")
        uploaded_documents = []
        for file in files:
            doc = handle_pdf_upload(session, id, file)
            uploaded_documents.append(doc.model_dump())
        return uploaded_documents
