import json
import time
from collections.abc import Generator
from typing import Any

import httpx
from fastapi import APIRouter, HTTPException, status
from fastapi.responses import StreamingResponse
from sqlmodel import col, select
from starlette.background import BackgroundTask

from app.api.deps import CurrentUser, SessionDep
from app.core.config import settings
from app.models import ChatMessage, Collection, MessageRequest
from app.utils import user_permissions

# This file is inspired from : https://gitlab.adullact.net/dgfip/projets-ia/caradoc/-/blob/dc346202070924f0fa64ebfd495429682f150722/api/app/routers/chat.py
router = APIRouter()


client = httpx.AsyncClient(timeout=settings.SOFIA_CHAT.request_timeout if settings.SOFIA_CHAT else 120)


def generate_user_prompt_response(
    generator: Any, sources: list[dict[str, str]], message: str
) -> Generator[str, None, None]:
    """Generate a stream of data containing the sources and the response message

    Args:
        generator: The response generator
        sources: The sources that helped generate the response

    Returns:
        None

    """

    # We start by returning the sources and metadata we already have at our disposal
    yield f"""{json.dumps({
        "event": "sources",
        "data": {
            "sources": sources,
        },
    })}$$$\n"""
    if len(sources) == 0:
        yield f"""{json.dumps({
            "event": "content",
            "data": {
                "content": "Aucune information n'a été trouvée pour répondre à votre requête, veuillez charger de nouveaux documents et réessayer."
            },
        })}$$$\n"""
    else:
        # We return the chunks that constitute the content of the response message
        for chunk in generator(message):
            yield f"""{json.dumps({
                "event": "content",
                "data": {
                    "content": chunk
                },
            })}$$$\n"""


def fake_message_gen(message: str) -> Generator[str, None, None]:
    time.sleep(1)
    yield "Bonjour, je ne fais que répéter votre message : "
    time.sleep(1)
    yield message


# def response_generator(query: str) -> Generator[str, None, None]:
#     pass


@router.post(
    "/message",
    response_description="Answer user prompt request",
)
async def process_message(session: SessionDep, current_user: CurrentUser, chat_message: ChatMessage) -> Any:
    """Process the user prompt request and generate an accurate answer

    Args:
        message_request (MessageRequest): The data received from the client
        - collection_id (str, optional): The collection id
        - message (str): The message

    Returns:
        StreamingResponse: A stream of data containing the sources and the response message

    Raises:
        HTTPException

    """
    if not settings.SOFIA_CHAT:
        raise HTTPException(status_code=status.HTTP_503_SERVICE_UNAVAILABLE, detail="Sofia chat not configured")

    with user_permissions(current_user):
        statement = select(Collection).where(col(Collection.id).in_(chat_message.collection_ids))
        collections = session.exec(statement).all()

        missing = set(chat_message.collection_ids).difference({x.id for x in collections})
        if any(missing):
            raise HTTPException(status_code=404, detail=f"Invalid collections: {missing}")

        cannot_read = [x.id for x in collections if not x.can_read]
        if any(cannot_read):
            raise HTTPException(status_code=400, detail=f"Not enough privileges to query {cannot_read}")

        if len({col.type for col in collections}) > 1:
            raise HTTPException(status_code=400, detail="Cannot query collections of different type")

        message_request = MessageRequest(
            qdrant_collection_name=settings.QDRANT_COLLECTION_NAMES[collections[0].type],
            query=chat_message.query,
            sources=[x.title for x in collections],
        )
        # TODO: verify arg type
        req = client.build_request(
            settings.SOFIA_CHAT.request_type,
            settings.SOFIA_CHAT.chat,
            data=message_request.model_dump_json(),  # type: ignore[arg-type]
        )
        # print(req)
        r = await client.send(req, stream=True)
        return StreamingResponse(r.aiter_lines(), background=BackgroundTask(r.aclose))

        # try:  # TODO: needed ?
        #     message = message_request.query

        #     # @temp don't want to use llm-guard as a dependencie because it rellies on torch
        #     # which downloads all CUDA libs
        #     # We execute our RAG pipeline
        #     # Check message for LLM security purpose
        #     # if not sanitize_input(message=message):
        #     #     raise HTTPException(
        #     #         status_code=status.HTTP_400_BAD_REQUEST,
        #     #         detail=f"Error while sanitizing input for LLM",
        #     #     )

        #     # We extract the sources that helped generate the response
        #     sources = [
        #         {
        #             "id": "7334a90a-cf69-4d27-be69-e8a3dfb799cc",
        #             "content": "qualité   chimique,   et   la   vérification   du   respect   des   seuils   réglementaires   définis   pour   chaque   élément polluant.",
        #             "filename": "11206_avis_pgpod_blavet_35_56_29_2024apb10.pdf",
        #             "filetype": "pdf",
        #             "index": "66a8b5f8f1f4e43c7ee70a0b",
        #             "file_id": "66a8b60df1f4e43c7ee70a12",
        #             "score": "0.17998885",
        #         },
        #         {
        #             "id": "1364efb4-7ba2-4d95-ad17-fb376227da89",
        #             "content": "in de démontrer l’absence d’incidence du projet sur les eaux souterraines, sur le cours d’eau et sur les zones humides proches.",
        #             "filename": "11219_extension_carriere_guitternel_22_2024apb19.pdf",
        #             "filetype": "pdf",
        #             "index": "66a8b5f8f1f4e43c7ee70a0b",
        #             "file_id": "66a8b60df1f4e43c7ee70a11",
        #             "score": "0.15602958",
        #         },
        #     ]

        #     # We return a stream of data containing the sources and the response message
        #     return StreamingResponse(
        #         generate_user_prompt_response(fake_message_gen, sources, message),
        #         media_type="text/event-stream",
        #         headers={
        #             "Cache-Control": "no-cache",
        #             "X-Accel-Buffering": "no",
        #         },
        #     )
        # except Exception as e:
        #     raise HTTPException(status_code=500, detail="Error while processing incoming message") from e
