from collections.abc import Generator
from typing import Annotated

import jwt
from fastapi import Depends, HTTPException, Query, status
from fastapi.security import OAuth2PasswordBearer
from jwt.exceptions import InvalidTokenError
from pydantic import ValidationError
from sqlmodel import Session

from app.core import security
from app.core.config import settings
from app.core.db import engine
from app.models import TokenPayload, User, UserRole

reusable_oauth2 = OAuth2PasswordBearer(tokenUrl=f"{settings.API_V1_STR}/login/access-token")


def get_db() -> Generator[Session, None, None]:
    with Session(engine) as session:
        yield session


SessionDep = Annotated[Session, Depends(get_db)]
TokenDep = Annotated[str, Depends(reusable_oauth2)]


def get_current_user_generic(session: SessionDep, token: str) -> User:
    try:
        payload = jwt.decode(token, settings.SECRET_KEY, algorithms=[security.ALGORITHM])
        token_data = TokenPayload(**payload)
    except (InvalidTokenError, ValidationError):
        raise HTTPException(
            status_code=status.HTTP_403_FORBIDDEN,
            detail="Could not validate credentials",
        )
    user = session.get(User, token_data.sub)
    if not user:
        raise HTTPException(status_code=404, detail="User not found")
    if not user.is_active:
        raise HTTPException(status_code=400, detail="Inactive user")
    return user


def get_current_user(session: SessionDep, token: TokenDep) -> User:
    return get_current_user_generic(session, token)


CurrentUser = Annotated[User, Depends(get_current_user)]


def get_current_user_with_query(session: SessionDep, token: str = Query(None)) -> User:
    """
    Some routes needs to send token within the querystring instead of the classic Authorization
    header because we're not responsible for the request being made and therefore can't set specific headers.
    i.e : PSPDFKit viewer only accept a document url and craft request on his own
    """
    return get_current_user_generic(session, token)


CurrentUserWithQuery = Annotated[User, Depends(get_current_user_with_query)]


def get_current_active_superuser(current_user: CurrentUser) -> User:
    if not current_user.role == UserRole.superadmin:
        raise HTTPException(status_code=403, detail="The user doesn't have enough privileges")
    return current_user


CurrentSuperAdmin = Annotated[User, Depends(get_current_active_superuser)]


def get_current_admin_user(current_user: CurrentUser) -> User:
    if current_user.role not in (UserRole.superadmin, UserRole.groupadmin):
        raise HTTPException(status_code=403, detail="The user doesn't have enough privileges")
    return current_user


CurrentAdminUser = Annotated[User, Depends(get_current_admin_user)]
