import React, { useState, useEffect, useContext } from "react";
import ReactDOM from "react-dom/client";
import {
  BrowserRouter,
  Routes,
  Route,
  Link,
  useLocation,
  useNavigate,
} from "react-router-dom";
import { GlobalStyles } from "tss-react";
import { startReactDsfr } from "@codegouvfr/react-dsfr/spa";
import { Badge } from "@codegouvfr/react-dsfr/Badge";
import { Header } from "@codegouvfr/react-dsfr/Header";
import { Footer } from "@codegouvfr/react-dsfr/Footer";
import { headerFooterDisplayItem } from "@codegouvfr/react-dsfr/Display";

import { AlertProvider } from "@/shared/contexts/AlertContext";
import styles from "./main.module.css";
import ProtectedRoute from "@/components/ProtectedRoute";
import { AuthContext, AuthProvider } from "./components/AuthProvider";
import { LoginPage } from "./LoginPage";
import { Home } from "./Home";
import { Home as HomePdfJs } from "./HomePdfJs";
import ChatPage from "./chat/page";
import { forgeUrl } from "./shared/fetchUtils";

startReactDsfr({ defaultColorScheme: "system", Link });

declare module "@codegouvfr/react-dsfr/spa" {
  interface RegisterLink {
    Link: typeof Link;
  }
}

ReactDOM.createRoot(document.getElementById("root") as HTMLElement).render(
  <React.StrictMode>
    <BrowserRouter basename={import.meta.env.VITE_BASENAME}>
      <AlertProvider>
        <AuthProvider>
          <Root />
        </AuthProvider>
      </AlertProvider>
    </BrowserRouter>
  </React.StrictMode>
);

if (import.meta.env.VITE_COMMIT_SHORT_SHA) {
  console.log(`Liriae-viewer build : ${import.meta.env.VITE_COMMIT_SHORT_SHA}`);
}

function Root() {
  const location = useLocation();
  const navigate = useNavigate();
  const [isFullScreen, setIsFullScreen] = useState(false);
  const { currentUser, setAccessToken } = useContext(AuthContext);

  const onFullScreen = () => {
    setIsFullScreen((prevFullScreen) => !prevFullScreen);
  };
  const mainCss = isFullScreen ? styles.fullscreenMain : styles.main;
  const quickAccessItems = currentUser
    ? [
        {
          buttonProps: {
            onClick: function noRefCheck() {},
          },
          iconId: "fr-icon-account-circle-line",
          text: currentUser.full_name || currentUser.email,
        },
        {
          buttonProps: {
            onClick: function noRefCheck() {
              setAccessToken(null);
              navigate("/login");
            },
          },
          iconId: "fr-icon-lock-line",
          text: "Se déconnecter",
        },
      ]
    : [
        {
          buttonProps: {
            onClick: function noRefCheck() {
              window.location.href =
                forgeUrl(
                  `/login/sso?redirect_url=${encodeURIComponent(
                    window.location.origin
                  )}`
                ) + "/";
            },
          },
          iconId: "fr-icon-account-circle-line",
          text: "Se connecter",
        },
      ];
  return (
    <>
      <GlobalStyles
        styles={{
          html: {
            scrollBehavior: "smooth",
          },
        }}
      />
      {!isFullScreen && (
        <Header
          brandTop={
            <>
              République
              <br />
              Française
            </>
          }
          serviceTitle={
            <>
              LIRIAe{" "}
              <Badge as="span" noIcon severity="success">
                Beta
              </Badge>
            </>
          }
          homeLinkProps={{
            to: "/",
            title:
              "Accueil - Nom de l’entité (ministère, secrétariat d‘état, gouvernement)",
          }}
          // @ts-ignore
          quickAccessItems={[headerFooterDisplayItem, ...quickAccessItems]}
          navigation={[
            {
              isActive: location.pathname === "/" ? true : false,
              linkProps: {
                to: "/",
                target: "_self",
              },
              text: "Lecture",
            },
            {
              isActive: location.pathname === "/chat" ? true : false,
              linkProps: {
                to: "/chat",
                target: "_self",
              },
              text: "Conversation",
            },
          ]}
        />
      )}
      <div className={mainCss}>
        <Routes>
          <Route
            path="/"
            element={
              <ProtectedRoute>
                <Home onFullScreen={onFullScreen} isFullScreen={isFullScreen} />
              </ProtectedRoute>
            }
          />
          <Route path="/login" element={<LoginPage />} />
          <Route
            path="/pdfjs"
            element={
              <ProtectedRoute>
                <HomePdfJs />
              </ProtectedRoute>
            }
          />
          <Route
            path="/chat"
            element={
              <ProtectedRoute>
                <ChatPage />
              </ProtectedRoute>
            }
          />
          <Route path="*" element={<h1>404</h1>} />
        </Routes>
      </div>
      {!isFullScreen && (
        <Footer
          accessibility="non compliant"
          contentDescription={``}
          bottomItems={[headerFooterDisplayItem]}
        />
      )}
    </>
  );
}
