import {
  ChangeEvent,
  useState,
  useContext,
  useRef,
  useEffect,
  useMemo,
} from "react";
import { useLocation, useNavigate } from "react-router-dom";

import { Button } from "@codegouvfr/react-dsfr/Button";
import AlertContext from "@/shared/contexts/AlertContext";

import { LiriaeDocument, OnSelectionType } from "@/shared/types";
import DocumentList from "./DocumentList";
import CircularProgress from "@mui/material/CircularProgress";
import { forgeUrl } from "@/shared/fetchUtils";
import { AuthContext } from "./AuthProvider";
import { DropZone } from "./DropZone";

export default function ViewerSidePanel({
  onSelection,
  selectedDocument,
}: {
  onSelection: OnSelectionType;
  selectedDocument: LiriaeDocument | null;
}) {
  const location = useLocation();
  const navigate = useNavigate();
  const searchParams = new URLSearchParams(location.search);
  const documentIdInQuery = searchParams.get("documentId");
  const pageNumInQuery = searchParams.get("pageNum");
  const {
    accessToken,
    currentCollection,
    liriaeDocuments,
    setLiriaeDocuments,
    isProcessingServiceAvailable,
    setIsProcessingServiceAvailable,
  } = useContext(AuthContext);
  // const [liriaeDocuments, setLiriaeDocuments] = useState<LiriaeDocument[]>([]);
  const documentsRef = useRef(liriaeDocuments);

  const [isUploading, setIsUploading] = useState(false);
  // We'll need to display an alert message whether the files upload is successful or not
  const { setAlert } = useContext(AlertContext);
  const uploadInputRef = useRef<HTMLInputElement>(null);
  const isProcessingFile = useRef(false);

  // @todo use zod for that...
  const parsePageNum = (pageNumStr: string | null) => {
    return pageNumStr && !isNaN(parseInt(pageNumStr, 10))
      ? parseInt(pageNumStr, 10)
      : 0;
  };

  const navigateToDoc = (document: LiriaeDocument, pageNum?: number) => {
    navigate(
      `/?documentId=${document.id}${pageNum ? "&pageNum=" + pageNum : ""}`
    );
  };

  useEffect(() => {
    if (
      (documentIdInQuery &&
        selectedDocument &&
        selectedDocument.id !== documentIdInQuery) ||
      !selectedDocument
    ) {
      const newSelectedDoc = liriaeDocuments.find(
        (doc) => doc.id === documentIdInQuery
      );
      if (newSelectedDoc) {
        const pageNum = parsePageNum(pageNumInQuery);
        onSelection(newSelectedDoc, pageNum);
      }
    }
  }, [documentIdInQuery, pageNumInQuery, liriaeDocuments]);

  const uploadFilesToCurrentCollection = async (
    fileList: FileList | File[]
  ) => {
    const data = new FormData();

    for (const file of fileList) {
      data.append(`files`, file, file.name);
    }

    setIsUploading(true);

    try {
      const response = await fetch(
        forgeUrl(`/collections/${currentCollection!.id}/document`),
        {
          headers: {
            Authorization: accessToken!,
          },
          method: "POST",
          body: data,
        }
      );

      if (response.ok) {
        setAlert({
          description: "Les fichiers ont été ajoutés avec succès.",
          isOpen: true,
          severity: "success",
          title: "Ajout réussi",
          autoHide: true,
        });
        const doc = await response.json();
        // Automatically select the new document
        if (doc) {
          // And load it
          navigateToDoc(doc[0]);
        }
        // Add the document to the list
        setLiriaeDocuments([...liriaeDocuments, ...doc]);
      } else {
        setAlert({
          description:
            "Erreur lors de l'envoie des fichiers. Veuillez réessayer ultérieurement.",
          isOpen: true,
          severity: "error",
          title: "Erreur",
        });
      }
    } catch (e: any) {
      setAlert({
        description: e.message,
        isOpen: true,
        severity: "error",
        title: "Erreur",
      });
    } finally {
      setIsUploading(false);
    }
  };

  const handleFileChange = async (e: ChangeEvent<HTMLInputElement>) => {
    const fileList = e.target.files;

    if (!fileList?.length) {
      return;
    }

    uploadFilesToCurrentCollection(fileList);
  };

  const handleClick = async () => {
    if (uploadInputRef.current) {
      uploadInputRef.current.click();
    }
  };

  const deleteDocument = async (documentId: string) => {
    try {
      const response = await fetch(forgeUrl(`/documents/${documentId}`), {
        method: "DELETE",
        headers: {
          Authorization: accessToken!,
        },
      });
      if (response.ok) {
        const newList = liriaeDocuments.filter((doc) => doc.id != documentId);
        if (newList.length > 0 && selectedDocument?.id === documentId) {
          navigateToDoc(newList[0]);
        } else if (newList.length === 0) {
          // Clear query string
          navigate("/");
          onSelection(null);
        }
        setLiriaeDocuments(newList);
      } else {
        setAlert({
          description: "Erreur lors de la suppression du fichier.",
          isOpen: true,
          severity: "error",
          title: "Erreur",
        });
      }
    } catch (e: any) {
      setAlert({
        description: e.message,
        isOpen: true,
        severity: "error",
        title: "Erreur",
      });
    }
  };

  useEffect(() => {
    if (liriaeDocuments.length > 0) {
      // Check if there is not already a document in query string (first click on link)
      let selectedDocId = liriaeDocuments[0];
      let pageNum = 1;
      if (documentIdInQuery) {
        const doc = liriaeDocuments.find(
          (d: LiriaeDocument) => d.id === documentIdInQuery
        );
        if (doc) {
          selectedDocId = doc;
          if (pageNumInQuery) {
            pageNum = parsePageNum(pageNumInQuery);
          }
        }
      }
      // And load it
      navigateToDoc(selectedDocId, pageNum);
    }
  }, [liriaeDocuments]);

  useEffect(() => {
    documentsRef.current = liriaeDocuments;
  }, [liriaeDocuments]);

  useEffect(() => {
    (async () => {
      if (isProcessingServiceAvailable && !isProcessingFile.current) {
        isProcessingFile.current = true;
        let documentChanged = false;
        for (const doc of liriaeDocuments) {
          try {
            if (!doc.processing_task_id) {
              const res = await fetch(
                forgeUrl(`/documents/${doc.id}/process`),
                {
                  method: "POST",
                  headers: {
                    Authorization: accessToken!,
                  },
                }
              );
              // @todo use http status
              if (res.status === 503) {
                setIsProcessingServiceAvailable(false);
                // Don't try to process any other document because service is unavailable
                break;
              } else if (res.ok) {
                doc.processing_task_id = await res.json();
                documentChanged = true;
              }
            }
          } catch (e) {
            console.error(e);
            setIsProcessingServiceAvailable(false);
          }
        }
        isProcessingFile.current = false;
        if (documentChanged) {
          setLiriaeDocuments([...liriaeDocuments]);
        }
      }
    })();
  }, [liriaeDocuments]);

  return (
    <>
      <DocumentList
        onDelete={deleteDocument}
        selectedDocument={selectedDocument}
      />
      <DropZone onDrop={uploadFilesToCurrentCollection} />
      <input
        style={{ display: "none" }}
        ref={uploadInputRef}
        type="file"
        accept="application/pdf"
        multiple
        onChange={handleFileChange}
      />
      <Button
        type="button"
        disabled={isUploading}
        // @ts-ignore
        iconId={isUploading || "fr-icon-file-add-line"}
        iconPosition="right"
        onClick={handleClick}
        priority="secondary"
      >
        {isUploading ? "Envoi en cours" : "Ajouter des fichiers"}
        {isUploading && (
          <CircularProgress
            size={24}
            sx={{
              marginLeft: "12px",
            }}
          />
        )}
      </Button>
    </>
  );
}
