import { useContext, useEffect, useRef } from "react";

import "./PdfViewerComponent.css";
import { LiriaeDocument } from "@/shared/types";
import { AuthContext } from "./AuthProvider";

let PSPDFKit: any;
export default function PdfViewerComponent({
  useDocEngine,
  document,
  onFullScreen,
  initialPageNum,
}: {
  useDocEngine: boolean;
  document: LiriaeDocument;
  onFullScreen: () => void;
  initialPageNum: number;
}) {
  const initialPageNumRef = useRef(initialPageNum);
  const documentRef = useRef(document);
  const { accessToken, currentUser } = useContext(AuthContext);
  const containerRef = useRef<HTMLDivElement>(null);
  let instance: any; // Can't find a way to use PSPDFKit types : PSPDFKit.Instance;

  const updateComments = async (instance: any, document: any) => {
    const com = await instance.exportInstantJSON();
    // Only extract comments and annotations
    document.comments = {
      comments: com.comments,
      annotations: com.annotations,
      format: com.format,
    };

    await fetch(
      `${import.meta.env.VITE_API_URL}/api/v1/documents/${document.id}`,
      {
        method: "PUT",
        body: JSON.stringify(document),
        headers: {
          Authorization: accessToken!,
          "Content-Type": "application/json",
        },
      }
    );
  };
  useEffect(() => {
    const container = containerRef.current as HTMLDivElement;
    // Since there is a import that may take time in the async function
    // We need to store a ref of the initialPageNum and document value
    // to be sure this one will be used on viewer initialisation
    // Note : pagenum are zero based
    initialPageNumRef.current = initialPageNum > 0 ? initialPageNum - 1 : 0;
    documentRef.current = document;
    (async function () {
      if (!PSPDFKit) {
        // Load pspdfkit asynchronously to reduce bundle size
        PSPDFKit = await import("pspdfkit");
      }

      const {
        createBlock: createBlock,
        Interfaces: Interfaces,
        Recipes: Recipes,
        Core: Core,
      } = PSPDFKit.UI;

      if (container) {
        PSPDFKit.unload(container); // Ensure that there's only one PSPDFKit instance.
      }

      const customUIConfiguration = {
        [PSPDFKit.UIElement.Sidebar]: {
          [PSPDFKit.SidebarMode.ANNOTATIONS]: ({
            containerNode,
          }: {
            containerNode: any;
          }) => ({
            node: containerNode,
            onRenderItem: ({
              item: annotation,
              itemContainerNode,
            }: {
              item: any;
              itemContainerNode: any;
            }) => {
              // @note we can have instance type specific text
              // see for example :
              // https://github.com/Narashiman-K/SidebarShowAnnotationTextNotType/blob/9f9fd4b15d829866a0906dad7a83a02db1d96195/src/components/PdfViewerComponent.jsx
              if (
                annotation &&
                // this porperty is not documented and it seems it's the only way to distinguish
                // simple highlight and comment hightlights
                instance &&
                !annotation.isCommentThreadRoot
              ) {
                // Display the selected text instead of the type of annotation
                // Excpect for comments where we want to display the text of the first comment (default behaviour)
                instance
                  .getTextFromRects(annotation.pageIndex, annotation.rects)
                  .then((text: string) => {
                    itemContainerNode.querySelector("span").textContent = text;
                  });
              }
            },
          }),
        },
      };

      // Remove all unecessary buttons
      const toolsToRemove = [
        "note",
        "stamp",
        "image",
        "document-editor",
        "document-crop",
        "polygon",
        "polyline",
        "cloudy-polygon",
        "ellipse",
        "rectangle",
        "arrow",
        "pan",
        "multi-annotations-selection",
        //"ink",
        //"highlighter",
        "ink-eraser",
        "signature",
        "text",
        "callout",
        "line",
        "link",
      ];

      // Can't find a way to use PSPDFKit.ToolbarItem
      const toolbarItems: any[] = PSPDFKit.defaultToolbarItems.filter(
        (item: any) => {
          return !toolsToRemove.includes(item.type);
        }
      );

      // Add fullscreen button
      const fullscreenItem = {
        type: "custom",
        id: "fullscreen-mode",
        title: "Mode plein écran",
        // SVG icon now work but it's a bit big
        icon: '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="currentColor"><path d="M8 3V5H4V9H2V3H8ZM2 21V15H4V19H8V21H2ZM22 21H16V19H20V15H22V21ZM22 9H20V5H16V3H22V9Z"></path></svg>',
        onPress: () => {
          onFullScreen();
        },
      };

      // Put comment button right after text-highlighter
      toolbarItems.splice(toolbarItems.length - 4, 0, { type: "comment" });
      toolbarItems.push(fullscreenItem);

      // Scroll to a specific page
      const initialViewState = new PSPDFKit.ViewState({
        currentPageIndex: initialPageNumRef.current,
      });
      const psPdfKitConfig: any = {
        licenseKey: import.meta.env.VITE_PSPDFKIT_LICENSE_KEY,
        // Container where PSPDFKit should be mounted.
        container,
        initialViewState,
        // @temp disable linearized loading to bypass comments loading bugs
        // allowLinearizedLoading: true,
        customUI: customUIConfiguration,
        // Use the public directory URL as a base URL. PSPDFKit will download its library assets from here.
        baseUrl: `${window.location.protocol}//${window.location.host}${
          import.meta.env.BASE_URL || ""
        }`,
        toolbarItems,
        // Only let comment and highlight option on TextSelection
        inlineTextSelectionToolbarItems: ({
          defaultItems,
        }: {
          defaultItems: any;
        }) => {
          return defaultItems.filter(
            (item: any) => item.type === "comment" || item.type === "highlight"
          );
        },
        // @temp should load real user list
        mentionableUsers: [
          {
            name: "Jane Doe",
            displayName: "Jane Doe",
            id: "jane_doe",
            description: "jane@doe.com",
          },
          {
            name: "John Doe",
            displayName: "John Doe",
            id: "john_doe",
            description: "john@doe.com",
          },
        ],
        ui: {
          //This is a way to modify comment thread ui
          [Interfaces.CommentThread]: ({ props }: { props: any }) => ({
            content: createBlock(
              Recipes.CommentThread,
              { ...props },
              ({ ui, comments }: { ui: any; comments: any }) => {
                // const commentThread = ui.getBlockById("comment-thread")
                if (comments.length > 0) {
                  // Remove possibility to reply
                  // https://tinyurl.com/5dmakrwm
                  // ui.removeChildById("editor");
                }
                return ui.createComponent();
              }
            ).createComponent(),
          }),
        },
      };

      const currentDocument = documentRef.current;
      if (useDocEngine) {
        const res = await fetch(
          `${import.meta.env.VITE_API_URL}/docengine-jwt/${
            currentDocument.docengine_id
          }`
        );
        const jwt = await res.json();
        psPdfKitConfig.serverUrl = "http://localhost:5000/";
        psPdfKitConfig.authPayload = { jwt: jwt.token };
        psPdfKitConfig.instant = true;
        psPdfKitConfig.documentId = currentDocument.docengine_id;
      } else {
        // Remove Bearer
        const token = accessToken?.split(" ")[1];
        psPdfKitConfig.document = `${
          import.meta.env.VITE_API_URL
        }/api/v1/documents/${currentDocument.id}/file?token=${token}`;
        if (currentDocument.comments) {
          // @bug : doesn't work with allowLinearizedLoading option
          psPdfKitConfig.instantJSON = document.comments;
        }
      }

      instance = await PSPDFKit.load(psPdfKitConfig);
      instance.setAnnotationCreatorName(currentUser?.full_name);
      instance.setViewState((viewState: any) =>
        viewState.set("sidebarOptions", {
          [PSPDFKit.SidebarMode.ANNOTATIONS]: {
            includeContent: [
              ...PSPDFKit.defaultAnnotationsSidebarContent,
              PSPDFKit.Comment,
            ],
          },
        })
      );

      instance.addEventListener("comments.didSave", () => {
        updateComments(instance, currentDocument);
      });

      if (currentDocument.comments) {
        // @bug doesn't handle comments with response
        // await instance.applyOperations([
        //   {
        //     type: "applyInstantJson",
        //     instantJson: currentDocument.comments,
        //   },
        // ]);
      }
    })();

    // Cleanup event listener and unload PSPDFKit on component unmount
    return () => {
      if (PSPDFKit && instance) {
        PSPDFKit.unload(container);
      }
    };
  }, [document, initialPageNum]);

  return <div ref={containerRef} style={{ width: "100%", height: "100vh" }} />;
}
