import { Link } from "react-router-dom";
import List from "@mui/material/List";
import ListItem from "@mui/material/ListItem";
import ListItemText from "@mui/material/ListItemText";
import Tooltip from "@mui/material/Tooltip";
import CheckCircleOutlineIcon from "@mui/icons-material/CheckCircleOutline";
import AccessTimeIcon from "@mui/icons-material/AccessTime";
import ErrorOutlineIcon from "@mui/icons-material/ErrorOutline";
import HelpOutlineIcon from "@mui/icons-material/HelpOutline";
import SettingsSuggestIcon from "@mui/icons-material/SettingsSuggest";
import StopIcon from "@mui/icons-material/Stop";

import { LiriaeDocument, OnSelectionType, OnDeleteType } from "@/shared/types";
import { ProcessingStatus } from "@/shared/types/ProcessingStatus";
import { fr } from "@codegouvfr/react-dsfr";
import { cx } from "@codegouvfr/react-dsfr/tools/cx";
import CircularProgressWithContent from "./CircularProgressWithContent";
import { useContext, useState } from "react";
import { AuthContext } from "./AuthProvider";
import { forgeUrl } from "@/shared/fetchUtils";
import "./DocumentList.css";
import Button from "@codegouvfr/react-dsfr/Button";

function ProcessingIndicator({
  status,
  onStopProcessing,
}: {
  status: ProcessingStatus;
  onStopProcessing: () => void;
}) {
  if (!status) {
    return <></>;
  }
  const [isHovered, setIsHovered] = useState(false);
  const statusInfo = {
    UNKNOWN: {
      tooltip: "En attente d'information sur l'état du traitement IA",
      color: "info",
      icon: HelpOutlineIcon,
    },
    FAILED: {
      tooltip: "Traitement IA en erreur",
      color: "error",
      icon: ErrorOutlineIcon,
    },
    PENDING: {
      tooltip: "Traitement IA en attente",
      color: "warning",
      icon: AccessTimeIcon,
    },
    RETRY: {
      tooltip: "Traitement IA en attente",
      color: "warning",
      icon: AccessTimeIcon,
    },
    STARTED: {
      tooltip: "Traitement IA en cours",
      color: "info",
      icon: SettingsSuggestIcon,
    },
    SUCCESS: {
      tooltip: "Traitement IA effectué avec succès",
      color: "success",
      icon: CheckCircleOutlineIcon,
    },
  } as Record<ProcessingStatus, any>;
  const info = statusInfo[status];

  if (!info) {
    return <></>;
  }
  return status === "FAILED" || status === "SUCCESS" || status === "UNKNOWN" ? (
    // No progess for definitive status
    <Tooltip title={info.tooltip}>
      <info.icon color={info.color} />
    </Tooltip>
  ) : (
    <CircularProgressWithContent
      content={
        <>
          <Tooltip
            title={isHovered ? "Arrêter le traitement IA" : info.tooltip}
          >
            <div
              onMouseEnter={() => setIsHovered(true)}
              onMouseLeave={() => setIsHovered(false)}
              onClick={() => isHovered && onStopProcessing()}
            >
              {/* Allow use to stop processing */}
              {isHovered ? <StopIcon /> : <info.icon color={info.color} />}
            </div>
          </Tooltip>
        </>
      }
    />
  );
}

export default function DocumentList({
  selectedDocument,
  onDelete,
}: {
  selectedDocument: LiriaeDocument | null;
  onDelete: OnDeleteType;
}) {
  const { liriaeDocuments } = useContext(AuthContext);
  const { accessToken } = useContext(AuthContext);
  const [disabledDeleteDocId, setDisabledDeleteDocId] = useState("");

  const handleStopProcessingClick = (docId: string) => {
    fetch(forgeUrl(`/documents/${docId}/process/stop`), {
      headers: {
        Authorization: accessToken!,
      },
      method: "POST",
    });
  };

  return (
    <nav className={cx(fr.cx("fr-sidemenu"))}>
      <List dense={true} className={cx(fr.cx("fr-sidemenu__list"))}>
        <div className={cx(fr.cx("fr-sidemenu__title"))}>Documents</div>
        {liriaeDocuments.map((doc) => (
          <ListItem
            className={cx(fr.cx("fr-sidemenu__item"))}
            key={doc.id}
            secondaryAction={
              <Button
                iconId="fr-icon-delete-fill"
                onClick={() => {
                  setDisabledDeleteDocId(doc.id);
                  onDelete(doc.id);
                }}
                priority="tertiary no outline"
                title="Supprimer ce document"
              />
            }
          >
            <Link
              className={cx(fr.cx("fr-sidemenu__link"))}
              to={`/?documentId=${doc.id}`}
              aria-current={
                selectedDocument?.id === doc.id ? "page" : undefined
              }
            >
              <ListItemText
                primary={doc.title || doc.filename}
                secondary={doc.title && doc.filename}
              />
            </Link>
            <ProcessingIndicator
              status={doc.processing_status}
              onStopProcessing={() => handleStopProcessingClick(doc.id)}
            />
          </ListItem>
        ))}
      </List>
    </nav>
  );
}
