import React, { useEffect, useState, createContext, useRef } from "react";
import { useLocation, useNavigate } from "react-router-dom";

import { forgeUrl } from "@/shared/fetchUtils";
import { LiriaeCollection, LiriaeDocument, User } from "@/shared/types";

type AuthContextType = {
  accessToken: string | null;
  setAccessToken: (token: string | null) => void;
  currentCollection: LiriaeCollection | null;
  liriaeDocuments: LiriaeDocument[];
  setLiriaeDocuments: (documents: LiriaeDocument[]) => void;
  isLoading: boolean;
  isAuthenticated: boolean;
  currentUser: User | null;
  isProcessingServiceAvailable: boolean;
  setIsProcessingServiceAvailable: (available: boolean) => void;
};

export const AuthContext = createContext<AuthContextType>({
  accessToken: null,
  setAccessToken: (token: string | null) => {},
  currentCollection: null,
  liriaeDocuments: [],
  setLiriaeDocuments: (documents: LiriaeDocument[]) => {},
  isLoading: true,
  isAuthenticated: false,
  currentUser: null,
  isProcessingServiceAvailable: true,
  setIsProcessingServiceAvailable: (available: boolean) => {},
});

export const AuthProvider: React.FC<{ children: React.ReactNode }> = ({
  children,
}) => {
  const [accessToken, setAccessToken] = useState<string | null>(
    localStorage.getItem("access_token")
  );
  const [currentCollection, setCurrentCollection] =
    useState<LiriaeCollection | null>(null);
  const documentsRef = useRef<LiriaeDocument[]>([]);
  const [liriaeDocuments, setLiriaeDocuments] = useState<LiriaeDocument[]>([]);
  const [isLoading, setIsLoading] = useState(true);
  const [isAuthenticated, setIsAuthenticated] = useState(false);
  const [currentUser, setCurrentUser] = useState<User | null>(null);
  // We don't know yet the processing service status
  const [isProcessingServiceAvailable, setIsProcessingServiceAvailable] =
    useState(true);
  const navigate = useNavigate();
  const location = useLocation();
  const searchParams = new URLSearchParams(location.search);
  const queryToken = searchParams.get("token");

  const isTokenValid = async (tokenHeader: string) => {
    try {
      const res = await fetch(forgeUrl("/login/test-token"), {
        headers: { Authorization: tokenHeader },
        method: "POST",
      });
      if (res.ok) {
        const data = await res.json();
        setCurrentUser(data);
      } else {
        setCurrentUser(null);
      }

      return res.ok;
    } catch {
      setCurrentUser(null);
      return false;
    }
  };

  useEffect(() => {
    // Token is return by the SSO through query parameters
    // We need to validate it before saving it to localStorage
    const verifyToken = async () => {
      if (!accessToken && queryToken) {
        setIsLoading(true);
        const tokenHeader = `Bearer ${queryToken}`;
        const isValid = await isTokenValid(tokenHeader);

        // In all cases we need to remove token from url
        searchParams.delete("token");
        navigate({
          pathname: location.pathname,
          search: searchParams.toString() ? `?${searchParams.toString()}` : "",
        });

        if (isValid) {
          setIsAuthenticated(true);
          setAccessToken(tokenHeader);
        } else {
          console.error(`Token : ${queryToken} not valid.`);
        }
      }
      setIsLoading(false);
    };

    verifyToken();
  }, [queryToken]);

  useEffect(() => {
    // We might have a token in the localStorage but it can be not valid
    // ie : too old or server restarted
    if (accessToken && !isAuthenticated) {
      isTokenValid(accessToken).then((isValid) => {
        if (isValid) {
          setIsAuthenticated(true);
        } else {
          // Reset token
          setAccessToken(null);
        }
      });
    }
  }, [accessToken]);

  useEffect(() => {
    if (accessToken) {
      localStorage.setItem("access_token", accessToken);
    } else {
      localStorage.removeItem("access_token");
      setCurrentUser(null);
      setIsAuthenticated(false);
      setLiriaeDocuments([]);
      setCurrentCollection(null);
    }
  }, [accessToken]);

  useEffect(() => {
    const getFirstEditableCollection = async () => {
      if (accessToken) {
        setIsLoading(true);
        const response = await fetch(forgeUrl(`/collections/`), {
          headers: {
            Authorization: accessToken,
          },
        });

        if (response.ok) {
          const collections = await response.json();
          const personalCollections = collections.items.filter(
            (col: any) => col.type === "personal"
          );
          if (personalCollections.length === 0) {
            const response = await fetch(forgeUrl(`/collections/`), {
              method: "POST",
              // Just use timestamp to avoid title collision
              body: JSON.stringify({
                title: `My Collection ${Date.now()}`,
                description: "",
                type: "personal",
              }),
              headers: {
                Authorization: localStorage.getItem("access_token")!,
                "Content-Type": "application/json",
              },
            });
            if (response.ok) {
              return await response.json();
            }
          } else {
            return personalCollections[0];
          }
        }
      }
      return null;
    };

    getFirstEditableCollection().then((collection) => {
      if (accessToken) {
        setIsLoading(false);
      }

      if (collection) {
        setCurrentCollection(collection);
      }
    });
  }, [accessToken]);

  useEffect(() => {
    if (currentCollection && accessToken) {
      fetch(forgeUrl(`/collections/${currentCollection.id}/documents`), {
        headers: {
          Authorization: accessToken,
        },
      })
        .then(async (res) => {
          if (res.ok) {
            const docs = await res.json();
            setLiriaeDocuments(docs.items);
          } else {
            setLiriaeDocuments([]);
          }
        })
        .catch((e) => {
          console.error(e);
          setLiriaeDocuments([]);
        });
    }
  }, [currentCollection]);

  useEffect(() => {
    documentsRef.current = liriaeDocuments;
  }, [liriaeDocuments]);

  const updateDocumentsProcessingStatus = async () => {
    // We need to use the ref because this function can be call inside a setInterval
    const documents = [...documentsRef.current];
    let documentsChanged = false;
    for (const doc of documents) {
      // Don't check if we already have a definitive status
      if (
        doc.processing_status !== "SUCCESS" &&
        doc.processing_status !== "FAILED" &&
        doc.processing_task_id
      ) {
        try {
          const res = await fetch(
            forgeUrl(`/documents/${doc.id}/process/status`),
            {
              headers: {
                Authorization: accessToken!,
              },
              method: "POST",
            }
          );
          if (res.ok) {
            const status = (await res.json()).status;
            if (doc.processing_status !== status) {
              doc.processing_status = status;
              documentsChanged = true;
            }
          } else if (res.status === 503 || res.status === 500) {
            setIsProcessingServiceAvailable(false);
            break;
          }
        } catch (e) {
          setIsProcessingServiceAvailable(false);
          break;
        }
      }
    }

    if (documentsChanged) {
      setLiriaeDocuments(documents);
    }
  };

  // Special effect that runs periodically to automatically update processing status
  useEffect(() => {
    if (!isProcessingServiceAvailable || !accessToken) {
      return;
    }
    const intervalInMsBetweenFetchStatus = 2000;
    // Simulate a async interval
    const updateDocOnInterval = async (msToWait: number) => {
      await updateDocumentsProcessingStatus();
      return setTimeout(updateDocOnInterval, msToWait, msToWait);
    };

    updateDocOnInterval(intervalInMsBetweenFetchStatus);
    return () => {};
  }, [isProcessingServiceAvailable, accessToken]);

  return (
    <AuthContext.Provider
      value={{
        accessToken,
        setAccessToken,
        currentCollection,
        liriaeDocuments,
        setLiriaeDocuments,
        isLoading,
        isAuthenticated,
        currentUser,
        isProcessingServiceAvailable,
        setIsProcessingServiceAvailable,
      }}
    >
      {children}
    </AuthContext.Provider>
  );
};
