// Code taken from : https://gitlab.adullact.net/dgfip/projets-ia/caradoc/-/blob/dc346202070924f0fa64ebfd495429682f150722/web/src/app/caradoc/chat/page.tsx
// And adapt to liriae_viewer needs
import { SyntheticEvent, useContext, useEffect, useRef, useState } from "react";
import { Controller, FieldValues, useForm } from "react-hook-form";
import { capitalize } from "lodash";
import { z } from "zod";
import { zodResolver } from "@hookform/resolvers/zod";
import { tss } from "tss-react/mui";
import { fr } from "@codegouvfr/react-dsfr";
import { Button } from "@codegouvfr/react-dsfr/Button";
import Input from "@codegouvfr/react-dsfr/Input";
import { Highlight } from "@codegouvfr/react-dsfr/Highlight";
import { Alert, AlertProps } from "@codegouvfr/react-dsfr/Alert";
import ActionButton from "./components/ActionButton";
import Source from "./interfaces/Source";
import SourceComponent from "./components/Source";
import { answerModeData } from "./data/answer-mode";
import SourceContainer from "@/chat/components/SourceContainer";
import SourceMetadata from "@/chat/components/SourceMetadata";
import AnswerMode from "@/chat/interfaces/AnswerMode";
import SourceDetailModal, {
  sourceDetailModal,
} from "@/chat/components/modals/SourceDetailModal";
import SourceModalProps from "@/chat/interfaces/SourceModalProps";
import SourcesDrawer from "@/chat/components/SourcesDrawer";
import FilesDeletionData from "@/chat/interfaces/FilesDeletionData";
import FilesDeletionModal, {
  clearFilesModal,
} from "@/chat/components/modals/FilesDeletionModal";
import { createRandomString } from "@/chat/utils/string";
import { CustomAbortDOMException } from "@/chat/exceptions/CustomAbortDOMException";
import { REQUIRED_FIELD_ERROR_MESSAGE } from "@/shared/data/form/error-messages";
import { Collection as ChatCollection } from "@/chat/interfaces/Collection";
import { toCamelCase, toSnakeCase } from "@/shared/utils/object";
import { handleSSE } from "@/shared/utils/sse";
import { useIsDark } from "@codegouvfr/react-dsfr/useIsDark";
import { AuthContext } from "@/components/AuthProvider";
import Markdown from "react-markdown";
import { Link } from "react-router-dom";

// We'll display our logo on both the header and the center of the chat page
const logoData = {
  alt: "Logo Liriae Viewer",
  name: "logo-caradoc",
};

// Our tokens will be composed by 32 characters in that format '16chars-16chars'
const PARTIAL_TOKEN_SIZE = 16;

// We'll define the duration for which the alert messages will be displayed in case of error
const ALERT_MESSAGE_DISPLAY_DURATION = 3500;

// We'll limit the number of sources to display
const maxNumberOfSourcesToDisplay = 4;

// Enum for user feedback
enum UserFeedbackValue {
  Up = 1,
  Neutral = 0,
  Down = -1,
}

interface UserFeedback {
  id?: string;
  value: UserFeedbackValue;
}

// Users can orient the chatbot to provide answers according to specific settings
interface Settings {
  isLoading: boolean;
  collections: ChatCollection[];
}

// This is the format expected by the API when sending a user prompt request
interface UserPromptRequest {
  mode: AnswerMode;
  collectionIds: string[];
  collectionName?: string;
  index: string;
  query: string;
  rephrasedQuery?: string;
  token: string;
}

interface UserPromptResponse {
  sources?: Source[];
  content: string;
  generationCompleted?: boolean;
}

interface MessageBlock {
  userPromptRequest: UserPromptRequest;
  userPromptResponse: UserPromptResponse;
  userFeedback: UserFeedback;
  errorMessage?: {
    content: string;
  };
}

// This will be the format of the alert message displayed when an error occurs
interface AlertMessageData {
  message?: string;
  severity: AlertProps.Severity;
  isOpen: boolean;
  description: string;
  title: string;
}

// 'message' fields will benefit from this same validation rule
const validateString = z
  .string({
    errorMap: () => ({ message: REQUIRED_FIELD_ERROR_MESSAGE }),
  })
  .min(1);

const modes = ["collection", "file"] as const;

interface ExtendedFile extends File {
  id?: string;
}

// This schema will be used to validate the form data
const schema = z.object({
  collectionIds: z.array(z.string()).optional(),
  files: z.array(z.custom<ExtendedFile>()).optional(),
  query: validateString,
});

// We can infer the types of the schema in order to re-use them for TS validation
type FormData = z.infer<typeof schema>;

/**
 * Prevents the default action of an event
 * We'll use it as a placeholder for buttons that don't have any action yet
 * @param e
 */
const noAction = (e: SyntheticEvent) => {
  e.preventDefault();
};

/**
 * Generates a token to identify and manage file uploads
 * The 1st half of the token never changes, while the 2nd half is randomly updated when needed
 * @param token
 */
const generateToken = (token?: string) =>
  token
    ? // If a token has already been generated, we only update the second half of it
      `${token.split("-")[0]}-${createRandomString(PARTIAL_TOKEN_SIZE)}`
    : // Otherwise, we generate a new token
      `${createRandomString(PARTIAL_TOKEN_SIZE)}-${createRandomString(
        PARTIAL_TOKEN_SIZE
      )}`;

export default function ChatPage() {
  /* -------------------------------------------------------------------------------------------------------------- */
  /* Hooks and contexts ------------------------------------------------------------------------------------------- */
  /* -------------------------------------------------------------------------------------------------------------- */
  const { isDark } = useIsDark();
  const logoImgSrc = `/logos/${logoData.name}${isDark ? "-dark" : ""}.png`;

  // We'll store a generated token to identify and manage file uploads
  const [token, setToken] = useState("");

  // We'll keep track of the selection collection
  const [selectedCollection, setSelectedCollection] = useState(
    {} as ChatCollection
  );

  const { currentCollection, liriaeDocuments, isProcessingServiceAvailable } =
    useContext(AuthContext);

  const [isDocumentsProcessingStatusOk, setIsDocumentsProcessingStatusOk] =
    useState(false);
  // We'll store our settings data in this state
  const [settings, setSettings] = useState<Settings>({
    isLoading: true,
    collections: [],
  });

  // This variable will contain the source data to display when opening the source modal
  const [sourceModalData, setSourceModalData] = useState<SourceModalProps>(
    {} as SourceModalProps
  );

  // This property controls whether the sources' drawer is open or not
  const [openSourcesDrawer, setOpenSourcesDrawer] = useState(false);

  // When clicking 'See more...', we'll send the entire message which also contains the sources to be displayed by the drawer
  const [drawerMessage, setDrawerMessage] = useState({
    sources: [] as UserPromptResponse["sources"],
    mode: "" as AnswerMode,
  });

  // This represents the messages exchanged between the user and the chat API
  // Save messages in localStorage to be able to retrieve them if we refresh the page or navigate through the application
  const [messages, setMessages] = useState<MessageBlock[]>(() => {
    const savedMessages = localStorage.getItem("chatMessages");
    return savedMessages ? JSON.parse(savedMessages) : [];
  });

  // We'll store some metadata related to files deletion that will help us perform the right actions when clearing the attached files depending on the user's interactions
  const [filesDeletionData, setFilesDeletionData] = useState<FilesDeletionData>(
    {} as FilesDeletionData
  );

  // We'll need a hold of our message input to adjust its height when user types into it
  const inputMessageRef = useRef<HTMLTextAreaElement | null>(null);

  // We'll also need to scroll our messages section to the bottom when user click on scroll to bottom button
  const chatWindowRef = useRef<HTMLElement | null>(null);

  // We'll add a reference to the bottom of the messages section to be able to scroll to the end
  const chatWindowBottomRef = useRef<HTMLDivElement | null>(null);
  const [isChatBottomVisible, setIsChatBottomVisible] = useState(false);

  // We'll use this controller to cancel the user prompt request
  const [formSubmitController, setFormSubmitController] =
    useState<AbortController | null>(null);

  // This is the content that will show up in case of errors
  const [alertMessageData, setAlertMessageData] = useState(
    {} as AlertMessageData
  );

  // We'll use react-hook-form to handle our form validation and submission
  const {
    formState: { errors, isSubmitting },
    handleSubmit,
    reset,
    control,
    setValue,
  } = useForm<FormData>({
    resolver: zodResolver(schema),
  });

  // We retrieve our styles
  const { classes, cx } = useStyles({
    isSubmitting,
  });

  /* -------------------------------------------------------------------------------------------------------------- */
  /* Effects ------------------------------------------------------------------------------------------------------ */
  /* -------------------------------------------------------------------------------------------------------------- */

  // When the page load, we'll generate a random token that will identify the user's uploaded files
  useEffect(() => {
    setToken(generateToken());
  }, []);

  // We fetch our settings data when the component mounts
  useEffect(() => {
    if (!currentCollection) {
      return;
    }
    // Just set value without fetching them
    setSettings((prevState) => ({
      ...prevState,
      collections: [currentCollection],
    }));
    setSelectedCollection(currentCollection);
  }, [currentCollection]);

  // The react-dsfr alert does not disappear automatically after a certain time,
  // so we'll implement the automatic dismissal of the alert message ourselves
  useEffect(() => {
    let timeout: NodeJS.Timeout;

    // When a new alert message is displayed, we'll set a timeout to dismiss it after a certain duration
    if (alertMessageData.isOpen) {
      timeout = setTimeout(() => {
        handleCloseAlert();
      }, ALERT_MESSAGE_DISPLAY_DURATION);
    }

    // We should not forget to clear any existing timeout when the component unmounts
    return () => {
      timeout && clearTimeout(timeout);
    };
  }, [alertMessageData]);

  // We store chat messages in localstorage to keep messages across navigation and
  // page reload
  useEffect(() => {
    localStorage.setItem("chatMessages", JSON.stringify(messages));
  }, [messages]);

  // Use to check if the bottom (we use the input because it always displayed) of the chat is visible or not
  // And therefore display the "scroll to bottom" button
  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        setIsChatBottomVisible(entry.isIntersecting);
      },
      { threshold: 0.1 }
    );

    if (inputMessageRef.current) {
      observer.observe(inputMessageRef.current);
    }

    return () => {
      if (inputMessageRef.current) {
        observer.unobserve(inputMessageRef.current);
      }
    };
  }, []);

  useEffect(() => {
    const hasPendingOperation = liriaeDocuments.some((d) =>
      ["PENDING", "RETRY", "STARTED"].includes(d.processing_status)
    );
    const hasDocumentInSuccess = liriaeDocuments.some((d) =>
      ["SUCCESS"].includes(d.processing_status)
    );
    setIsDocumentsProcessingStatusOk(
      !hasPendingOperation && hasDocumentInSuccess
    );
  }, [liriaeDocuments]);
  /* -------------------------------------------------------------------------------------------------------------- */
  /* Handlers ----------------------------------------------------------------------------------------------------- */
  /* -------------------------------------------------------------------------------------------------------------- */

  /**
   * Formats the alert message data before setting it
   * @param alertMessageData
   */
  const formatAndSetAlertMessageData = (alertMessageData: AlertMessageData) => {
    setAlertMessageData({
      ...alertMessageData,
      // Defaults values will be set if not present
      title: alertMessageData.title ?? "Erreur",
      severity: alertMessageData.severity ?? "error",
      isOpen: !!alertMessageData.isOpen,
    });
  };

  /**
   * Closes the alert message
   */
  const handleCloseAlert = () => {
    setAlertMessageData({} as AlertMessageData);
  };

  const resetChat = () => {
    // Stop SSE ongoing events
    formSubmitController?.abort(
      new CustomAbortDOMException(
        "La génération de réponse a été interrompue.",
        "Arrêt",
        "info",
        "resetButton"
      )
    );

    // We reset the list of messages
    setMessages([]);

    // We reset the form fields back to their default values
    reset();
  };

  /**
   * Takes care of clearing the files after clicking the "New Chat" button
   * This function will be called after confirming files deletion on the warning modal
   */
  const handleResetChatWhileFilesAttached = async () => {
    resetChat();
  };

  /**
   * Clears the chat session
   * @param attachedFiles
   */
  const handleResetChat = (attachedFiles: FormData["files"]) => async () => {
    if (isSubmitting) {
      // When the form is being submitted, we cancel the request
      formSubmitController?.abort(
        new CustomAbortDOMException(
          "La génération de réponse a été interrompue.",
          "Nouveau chat",
          "warning",
          "resetButton"
        )
      );

      // We reset the chat
      resetChat();
    } else {
      // First, if any files are attached, we clear them
      if (Array.isArray(attachedFiles) && attachedFiles.length > 0) {
        // When a user tries to reset the chat while files are attached, we'll warn him before proceeding
        setFilesDeletionData({
          trigger: "resetButton",
        });
        clearFilesModal.open();
      } else {
        // We reset the chat
        resetChat();
      }
    }
  };

  /**
   * Proceeds to form submission
   * @param data
   */
  const handleSubmitForm = async (data: FieldValues) => {
    // Right after form submission, we reset the message field
    setValue("query", "");

    // We resize the message input to its original size
    if (inputMessageRef.current) {
      inputMessageRef.current.style.height = "auto";
    }

    // We prepare our message containing request and response to insert into the feed
    const messageBlock: MessageBlock = {
      userPromptRequest: {
        ...data,
        token,
        collectionName: selectedCollection.name,
        collectionIds: [selectedCollection.id],
      } as UserPromptRequest,
      userPromptResponse: {} as UserPromptResponse,
      userFeedback: {
        value: UserFeedbackValue.Neutral,
      },
    };
    // We update the list of messages with the new user one
    const updatedMessages: MessageBlock[] = [...messages, messageBlock];

    // We update our state with the user message, so it gets immediately displayed on the chat interface
    setMessages(updatedMessages);

    try {
      // We'll use the AbortController to cancel the submission of the form if needed
      const formSubmitAbortController = new AbortController();

      // We keep the controller in memory in case we need to abort the request
      setFormSubmitController(formSubmitAbortController);

      // We can now send the complete user prompt request to our API
      const response = await fetch(
        `${import.meta.env.VITE_API_URL}/api/v1/chat/message`,
        {
          method: "POST",
          body: JSON.stringify(toSnakeCase(messageBlock.userPromptRequest)),
          signal: formSubmitAbortController.signal,
          headers: {
            Accept: "text/event-stream",
            "Content-Type": "application/json",
            Authorization: localStorage.getItem("access_token")!,
          },
        }
      );

      await handleSSE(
        response,
        (currentEvent) => {
          if (currentEvent.event === "rephrased_query") {
            setMessages((messages) => {
              const rephrasedQuery = currentEvent.data.query;
              // We need to target the last message
              const lastMessage = messages[messages.length - 1];
              lastMessage.userPromptRequest.rephrasedQuery = rephrasedQuery;
              return [...messages.slice(0, -1), lastMessage];
            });
          } else if (currentEvent.event === "sources") {
            setMessages((messages) => {
              const sofiaSources = toCamelCase(currentEvent.data.sources);
              const liriaeSources = sofiaSources.map((s: Source) =>
                updateSourceFileNameAndTitle(s)
              );
              // We need to target the last message
              const lastMessage: MessageBlock = {
                ...messages[messages.length - 1],
                userPromptResponse: {
                  content: "",
                  // We extract the sources from the "preData" event
                  sources: liriaeSources,
                },
              };

              return [...messages.slice(0, -1), lastMessage];
            });
          } else if (currentEvent.event === "content") {
            setMessages((messages) => {
              const currentMsg = messages[messages.length - 1];
              // We just need to concatenate the content received with what was previously accumulated
              const newContent = currentMsg.userPromptResponse.content
                ? currentMsg.userPromptResponse.content +
                  currentEvent.data.content
                : currentEvent.data.content;
              // We need to target the last message
              const lastMessage: MessageBlock = {
                ...currentMsg,
                userPromptResponse: {
                  ...currentMsg.userPromptResponse,
                  // We just need to concatenate the content received with what was previously accumulated
                  content: newContent,
                },
              };

              return [...messages.slice(0, -1), lastMessage];
            });
          }
        },
        () => {
          setMessages((messages) => [
            // We grab all the messages except the last one
            ...messages.slice(0, -1),
            {
              // We'll modify the last message by saying generation has been completed
              ...messages[messages.length - 1],
              userPromptResponse: {
                ...messages[messages.length - 1].userPromptResponse,
                generationCompleted: true,
              },
            },
          ]);
        },
        "$$$\n"
      );
    } catch (e: any) {
      console.error(e);
      // When any problem occurs, we display an alert message
      formatAndSetAlertMessageData({
        ...e,
        description: e.message,
        isOpen: true,
      });

      // After that, we can display an error message in the message feed except when the reset button is clicked
      // in which case enriching the messages' feed does not make sense since we'll clear it
      if (e.trigger === "resetButton") {
        return;
      }

      setMessages((messages) => [
        // We grab all the messages except the last one
        ...messages.slice(0, -1),
        {
          // We'll modify the last message by saying generation has been completed
          ...messages[messages.length - 1],
          userPromptResponse: {
            ...messages[messages.length - 1]?.userPromptResponse,
            generationCompleted: true,
          },
          errorMessage: {
            content: `${e.title ?? "Erreur"}: ${e.message}`,
          },
        },
      ]);
    }
  };

  /**
   * Get feedback from user for an AI response
   * @param feedbackValue
   * @param message
   * @param index
   */
  const handleFeedback =
    (feedbackValue: UserFeedbackValue, message: MessageBlock, index: number) =>
    async () => {
      try {
        // If the selected user feedback is identical as the one already selected, we'll neutralize it
        const userFeedbackValue =
          message.userFeedback.value === feedbackValue
            ? UserFeedbackValue.Neutral
            : feedbackValue;

        const userFeedbackResponse = await fetch(
          `${import.meta.env.VITE_API_URL}/api/v1/chat/feedback`,
          {
            method: "POST",
            body: JSON.stringify(
              toSnakeCase({
                id: message.userFeedback.id,
                userPromptRequest: message.userPromptRequest,
                userPromptResponse: message.userPromptResponse,
                userFeedback: userFeedbackValue,
              })
            ),
            headers: {
              "Content-Type": "application/json",
            },
          }
        );

        const updatedUserFeedback = await userFeedbackResponse.json();

        // We update the feedback in the messages feed
        setMessages(
          messages.map((currentMessage, currentMessageIndex) => {
            return currentMessageIndex === index
              ? {
                  ...currentMessage,
                  userFeedback: {
                    id: updatedUserFeedback.data._id,
                    value: userFeedbackValue,
                  },
                }
              : currentMessage;
          })
        );
      } catch (e) {
        // Fixme: handle errors properly
      }
    };
  /**
   * Aborts the submission of the form
   * @param e
   */
  const handleCancelSubmission = (e: SyntheticEvent) => {
    e.preventDefault();
    formSubmitController?.abort(
      new CustomAbortDOMException(
        "La génération de réponse a été interrompue.",
        "Arrêt",
        "info",
        "formSubmitCancelButton"
      )
    );

    // Fixme: maybe send negative feedback to API when the request is aborted
  };

  const handleOnSubmit = (e: SyntheticEvent) => {
    return isSubmitting
      ? handleCancelSubmission(e)
      : handleSubmit(handleSubmitForm)(e);
  };

  /**
   * Loads data into the source modal and opens it
   * @param source
   * @param indexPosition
   * @param answerMode
   */
  const handleOpenSourceModal =
    (source: Source, indexPosition: number, answerMode: AnswerMode) => () => {
      setSourceModalData({
        source,
        indexPosition,
        answerMode,
      });
      sourceDetailModal.open();
    };

  /**
   * Opens the sources' drawer and populates the sources
   * @param message
   */
  const handleOpenSourcesDrawer = (message: MessageBlock) => () => {
    setDrawerMessage({
      sources: message.userPromptResponse.sources,
      mode: message.userPromptRequest.mode,
    });
    setOpenSourcesDrawer(true);
  };

  /**
   * Closes the sources' drawer
   */
  const handleCloseSourcesDrawer = () => {
    setOpenSourcesDrawer(false);
  };

  const handleScrollToBottom = () => {
    chatWindowBottomRef.current?.scrollIntoView({ behavior: "smooth" });
  };

  const updateSourceFileNameAndTitle = (source: Source) => {
    const doc = liriaeDocuments.find((d) => d.id === source.id);
    if (!source.title) {
      source.title = (doc && doc.title) || "Aucun titre pour ce document";
    }
    if (!source.filename) {
      source.filename = doc ? doc.filename : "Document sans nom";
    }
    return source;
  };

  const replaceDocRefWithLink = (
    content: string,
    sources: Source[] | undefined
  ) => {
    // Do nothing if there is no sources
    if (sources) {
      // Simple ref
      let output = content.replace(
        /\[Doc (\d+)\]/g,
        (original, index: string) => {
          const source =
            sources.length >= Number(index) ? sources[Number(index) - 1] : null;
          if (source) {
            return `[[${index}](/?documentId=${source.id}&pageNum=${source.page})]`;
          } else {
            return original;
          }
        }
      );
      // Double refs
      output = output.replace(/\[Doc (\d+(?:, Doc \d+)*)\]/g, (_, docs) => {
        return docs
          .split(", Doc ")
          .map((num: string) => Number(num.trim()))
          .map((num: number, index: number, splitedStr: string[]) => {
            const source = sources.length >= num ? sources[num - 1] : null;
            const prefix = index === 0 ? "[" : "";
            const suffix = index === splitedStr.length - 1 ? "]" : "";
            if (source) {
              return `${prefix}[${num}](/?documentId=${source.id}&pageNum=${source.page})${suffix}`;
            } else {
              return `${prefix}${num}${suffix}`;
            }
          })
          .join(", "); // Recombine les liens avec des virgules
      });

      return output;
    }
    return content;
  };

  return (
    <>
      {/* Files deletion modal */}
      <FilesDeletionModal
        filesDeletionData={filesDeletionData}
        handleConfirmFilesDeletion={handleResetChatWhileFilesAttached}
      />

      {/* Source detail modal */}
      <SourceDetailModal {...sourceModalData} />

      {/* Sources drawer */}
      <SourcesDrawer
        open={openSourcesDrawer}
        handleOpenSourceDetailModal={handleOpenSourceModal}
        handleClose={handleCloseSourcesDrawer}
        sources={drawerMessage.sources!}
        answerMode={drawerMessage.mode}
      />

      {/* Alert snack message */}
      <div className={classes.alert}>
        <Alert
          severity={alertMessageData.severity}
          title={alertMessageData.title}
          description={alertMessageData.description}
          closable
          small
          isClosed={!alertMessageData.isOpen}
          onClose={handleCloseAlert}
        />
      </div>

      {/* Form content */}
      <form className={classes.mainContainer} onSubmit={handleOnSubmit}>
        {/* Chat window */}
        <section className={classes.chatWindow} ref={chatWindowRef}>
          {/* Messages section */}
          <div className={classes.messagesSection}>
            {messages.length === 0 ? (
              // Empty chat window
              <div className={classes.emptyChatWindowContainer}>
                <div>
                  {/* Logo */}
                  {/* <div className={classes.messageLogoContainer}>
                    <img src={logoImgSrc} alt={logoData.alt} />
                  </div> */}

                  {/* Chat instructions */}
                  <div className={classes.chatInstructions}>
                    <p>
                      {!isProcessingServiceAvailable && (
                        <div>
                          <i className="fr-icon-error-warning-line" />
                        </div>
                      )}
                      {isProcessingServiceAvailable &&
                        !isDocumentsProcessingStatusOk && (
                          <div>
                            <i className="fr-icon-time-line" />
                          </div>
                        )}
                      {isProcessingServiceAvailable
                        ? isDocumentsProcessingStatusOk
                          ? "Posez une question sur votre dossier"
                          : "Documents en cours de traitement, merci de patienter..."
                        : "Service de conversation IA momentanément indisponible"}
                    </p>
                  </div>
                </div>
              </div>
            ) : (
              // Chat window with messages
              <div className={classes.messagesContent}>
                {/* Messages contents */}
                {messages.map((message, index) => (
                  <div className={classes.messageContainer} key={index}>
                    {
                      message.userPromptRequest && (
                        // User prompt request message content
                        <div className={classes.userPromptRequestContainer}>
                          {/* Message content */}
                          <h4>{message.userPromptRequest.query}</h4>
                          <h6 className={classes.messageTitle}>
                            <i className="fr-icon-questionnaire-line" />
                            <span>Question reformulée</span>
                          </h6>
                          <p className="fr-text--lead">
                            {message.userPromptRequest.rephrasedQuery ||
                              "Reformulation de la question..."}
                          </p>

                          {/* Metadata related to the user prompt request */}
                          <div className={classes.userPromptRequestMetadata}>
                            {["mode", "collectionName"].map((parameter) => (
                              <div key={parameter}>
                                {message.userPromptRequest[
                                  parameter as keyof UserPromptRequest
                                ] ? (
                                  <div className={classes.promptParameterData}>
                                    <span
                                      className={classes.promptParameterLabel}
                                    >
                                      {parameter === "collectionName"
                                        ? "collection"
                                        : parameter}
                                      :
                                    </span>
                                    <span
                                      className={classes.promptParameterValue}
                                    >
                                      {parameter === "mode"
                                        ? answerModeData[
                                            message.userPromptRequest.mode
                                          ].name
                                        : message.userPromptRequest[
                                            parameter as keyof UserPromptRequest
                                          ]}
                                    </span>
                                  </div>
                                ) : null}
                              </div>
                            ))}
                          </div>
                        </div>
                      )

                      // Assistant response message
                    }

                    {/* We get a successful response */}
                    {message.userPromptResponse !== undefined && (
                      <>
                        {message.userPromptResponse.sources &&
                          message.userPromptResponse.sources.length > 0 && (
                            <>
                              {/* Message source title */}
                              <h6 className={classes.messageTitle}>
                                <i className="fr-icon-draft-line" />
                                <span>Sources</span>
                              </h6>

                              {/* Message source data */}
                              <div
                                className={cx(
                                  classes.answerSectionContainer,
                                  classes.sourcesContainer
                                )}
                              >
                                {/* The 1st [maxNumberOfSourcesToDisplay - 1] sources should be displayed */}
                                {message.userPromptResponse.sources
                                  .slice(0, maxNumberOfSourcesToDisplay - 1)
                                  .map((source, index) => (
                                    // Same document might be refere multiple time in sources
                                    <div key={`${source.id}-${index}`}>
                                      <SourceComponent
                                        source={source}
                                        indexPosition={index + 1}
                                        answerMode={
                                          message.userPromptRequest.mode
                                        }
                                        handleClick={handleOpenSourceModal(
                                          source,
                                          index + 1,
                                          message.userPromptRequest.mode
                                        )}
                                      />
                                    </div>
                                  ))}

                                {/* When we have exactly 'maxNumberOfSourcesToDisplay' sources, we can display all of those */}
                                {message.userPromptResponse.sources.length ===
                                  maxNumberOfSourcesToDisplay && (
                                  <div
                                    key={`${
                                      message.userPromptResponse.sources[
                                        maxNumberOfSourcesToDisplay - 1
                                      ].id
                                    }-${maxNumberOfSourcesToDisplay}`}
                                  >
                                    <SourceComponent
                                      indexPosition={
                                        maxNumberOfSourcesToDisplay
                                      }
                                      source={
                                        message.userPromptResponse.sources[
                                          maxNumberOfSourcesToDisplay - 1
                                        ]
                                      }
                                      answerMode={
                                        message.userPromptRequest.mode
                                      }
                                      handleClick={handleOpenSourceModal(
                                        message.userPromptResponse.sources[
                                          maxNumberOfSourcesToDisplay - 1
                                        ],
                                        maxNumberOfSourcesToDisplay,
                                        message.userPromptRequest.mode
                                      )}
                                    />
                                  </div>
                                )}

                                {/* When more than 'maxNumberOfSourcesToDisplay' sources, we display a show more button */}
                                {message.userPromptResponse.sources.length >
                                  maxNumberOfSourcesToDisplay && (
                                  <div
                                    className={classes.seeMoreSourcesContainer}
                                  >
                                    <SourceContainer
                                      answerMode={
                                        message.userPromptRequest.mode
                                      }
                                      handleClick={handleOpenSourcesDrawer(
                                        message
                                      )}
                                    >
                                      {message.userPromptResponse.sources
                                        .slice(
                                          maxNumberOfSourcesToDisplay - 1,
                                          maxNumberOfSourcesToDisplay + 1
                                        )
                                        .map((source, index) => (
                                          <SourceMetadata
                                            key={`${source.id}-${index}`}
                                            indexPosition={
                                              maxNumberOfSourcesToDisplay +
                                              index
                                            }
                                            filename={source.filename}
                                            answerMode="collection"
                                            showIcon={false}
                                          />
                                        ))}
                                      <div className={classes.moreCharacters}>
                                        ...
                                      </div>
                                      <div
                                        className={
                                          classes.seeMoreSourcesMessage
                                        }
                                      >
                                        Voir plus...
                                      </div>
                                    </SourceContainer>
                                  </div>
                                )}
                              </div>
                            </>
                          )}

                        {message.userPromptResponse.content && (
                          <>
                            {/* Message response */}
                            <h6 className={classes.messageTitle}>
                              <i className="fr-icon-question-answer-line" />
                              <span>Réponse</span>
                            </h6>

                            {/* Message text content */}
                            <div className={classes.answerSectionContainer}>
                              <Markdown
                                components={{
                                  a: ({ href, children }) => {
                                    // Convert internal links to React Link
                                    const isInternalLink =
                                      href && href.startsWith("/");
                                    if (isInternalLink) {
                                      return <Link to={href}>{children}</Link>;
                                    }

                                    return (
                                      <a
                                        href={href}
                                        target="_blank"
                                        rel="noopener noreferrer"
                                      >
                                        {children}
                                      </a>
                                    );
                                  },
                                }}
                              >
                                {replaceDocRefWithLink(
                                  message.userPromptResponse.content,
                                  message.userPromptResponse.sources
                                )}
                              </Markdown>
                            </div>
                          </>
                        )}

                        {/* We get an error response */}
                        {message.errorMessage && (
                          <div className={classes.errorMessageContainer}>
                            <i className="fr-icon-error-warning-line" />
                            {capitalize(message.errorMessage.content)}
                          </div>
                        )}

                        {message.userPromptResponse.generationCompleted && (
                          <>
                            {/* Answer feedback */}
                            <div className={classes.answerFeedbackContainer}>
                              <div>
                                {/* These icons are integrated as a hack to overcome react-dsfr loading only used icons */}
                                <div
                                  style={{ display: "none" }}
                                  className="fr-icon-thumb-down-fill fr-icon-thumb-down-line fr-icon-thumb-up-fill fr-icon-thumb-up-line"
                                />
                                <div className="positiveFeedbackIcon">
                                  <Button
                                    iconId={`fr-icon-thumb-up-${
                                      message.userFeedback.value ===
                                      UserFeedbackValue.Up
                                        ? "fill"
                                        : "line"
                                    }`}
                                    onClick={handleFeedback(
                                      UserFeedbackValue.Up,
                                      message,
                                      index
                                    )}
                                    type="button"
                                    priority="tertiary no outline"
                                    title="Feedback positif"
                                  />
                                </div>
                                <div className="negativeFeedbackIcon">
                                  <Button
                                    iconId={`fr-icon-thumb-down-${
                                      message.userFeedback.value ===
                                      UserFeedbackValue.Down
                                        ? "fill"
                                        : "line"
                                    }`}
                                    onClick={handleFeedback(
                                      UserFeedbackValue.Down,
                                      message,
                                      index
                                    )}
                                    type="button"
                                    priority="tertiary no outline"
                                    title="Feedback négatif"
                                  />
                                </div>
                              </div>
                            </div>
                          </>
                        )}
                      </>
                    )}

                    {/* Separation line between messages */}
                    {messages.length - 1 !== index && (
                      <hr className={classes.answerSectionSeparation} />
                    )}
                  </div>
                ))}
                {/* Bottom div to handle scroll to bottom when a new message is added */}
                <div ref={chatWindowBottomRef} />
              </div>
            )}
          </div>
          <div className={classes.scrollToBottomContainer}>
            {!isChatBottomVisible && messages.length !== 0 && (
              <Button
                iconId="fr-icon-arrow-down-line"
                className={cx(
                  classes.inputIcon,
                  classes.scrollToBottomButton,
                  "fr-ml-2v"
                )}
                priority="tertiary"
                title="Allez en bas du chat"
                onClick={() => handleScrollToBottom()}
              />
            )}
          </div>
          {/* Input section */}
          <div className={classes.inputSection}>
            {/* Warning message */}
            <div className={classes.llmAccuracyWarning}>
              Des erreurs sont possibles. Nous vous invitons à contrôler la
              validité des réponses.
            </div>
            {/* Message input */}
            <div className={classes.inputContainer}>
              {/* Message input */}
              <Controller
                control={control}
                name="files"
                render={({ field }) => (
                  <section>
                    <Controller
                      control={control}
                      name="query"
                      render={({
                        field: {
                          onChange: builtInOnChange,
                          onBlur,
                          value,
                          ref,
                        },
                      }) => (
                        <>
                          <div>
                            <Input
                              label={null}
                              disabled={
                                !(
                                  isProcessingServiceAvailable &&
                                  isDocumentsProcessingStatusOk
                                ) || isSubmitting
                              }
                              textArea
                              className={cx(classes.input)}
                              // state={errors?.message && 'error'}
                              // stateRelatedMessage={errors?.message?.message}
                              nativeTextAreaProps={{
                                placeholder: isProcessingServiceAvailable
                                  ? isDocumentsProcessingStatusOk
                                    ? "Discuter avec votre collection"
                                    : "Veuillez attendre la fin du traitement avant de pouvoir discuter..."
                                  : "Service indisponible",
                                rows: 1,
                                onBlur,
                                value,
                                ref: (e: HTMLTextAreaElement) => {
                                  ref(e);
                                  // We can share the ref provided by react-hook-form and initialize our own ref
                                  inputMessageRef.current = e;
                                },
                                onChange: (e) => {
                                  builtInOnChange(e);
                                  if (inputMessageRef.current) {
                                    inputMessageRef.current.style.height =
                                      "auto";
                                    inputMessageRef.current.style.height = `${e.target.scrollHeight}px`;
                                  }
                                },
                                onKeyPress: (e) => {
                                  // We'll submit the form when pressing 'Enter' without the 'Shift' key
                                  if (e.key === "Enter" && !e.shiftKey) {
                                    handleOnSubmit(e);
                                  }
                                },
                              }}
                            />
                          </div>
                        </>
                      )}
                    />
                  </section>
                )}
              />

              {/* Send message button */}
              <div
                className={cx(
                  classes.inputButtonContainer,
                  classes.sendMessageButtonContainer
                )}
              >
                {/* Send message loading spinner */}
                <ActionButton
                  isLoading={isSubmitting}
                  sx={{
                    color: fr.colors.decisions.background.default.grey.default,
                  }}
                />

                {/* Send message icon button */}
                <Button
                  iconId={
                    isSubmitting
                      ? "fr-icon-stop-circle-fill"
                      : "fr-icon-send-plane-fill"
                  }
                  disabled={
                    !isProcessingServiceAvailable ||
                    !isDocumentsProcessingStatusOk
                  }
                  className={cx(classes.inputIcon, classes.sendMessageIcon)}
                  title={
                    isSubmitting
                      ? "Arrêt de l'envoi du message"
                      : "Envoi du message"
                  }
                  type="submit"
                />
              </div>
            </div>
            <Button
              iconId="fr-icon-refresh-line"
              className={cx(
                classes.inputIcon,
                classes.sendMessageIcon,
                "fr-ml-2v"
              )}
              title="Réinitialiser le chat"
              onClick={() => resetChat()}
            />
          </div>
        </section>
      </form>
    </>
  );
}

const chatWindowMaxWidth = "49rem";
const configurationTitleHeight = 43;
const inputMessageMaxHeight = 550;
const sideBarWidth = 300;
const customBoxShadow = `0px 0px 7px -4px ${fr.colors.decisions.border.plain.blueFrance.default}`;

const useStyles = tss
  .withParams<{ isSubmitting: boolean }>()
  .withNestedSelectors<"seeMoreSourcesMessage">()
  .create(({ theme, classes, isSubmitting }) => ({
    alert: {
      position: "absolute",
      right: fr.spacing("4w"),
      top: fr.spacing("3w"),
      zIndex: 1,
      background: fr.colors.decisions.background.default.grey.default,
    },
    mainContainer: {
      minHeight: "100vh",
      display: "flex",
    },
    sidebar: {
      position: "relative",
      minWidth: sideBarWidth,
      width: sideBarWidth,
      borderRight: `1px solid ${fr.colors.decisions.border.default.grey.default}`,
      overflowY: "auto",
      boxShadow: customBoxShadow,
      ...fr.spacing("padding", { bottom: "4w", rightLeft: "3w" }),
    },
    headerContainer: {
      position: "sticky",
      top: 0,
      left: 0,
      right: 0,
      paddingBottom: fr.spacing("3w"),
      marginLeft: `-${fr.spacing("3w")}`,
      marginRight: `-${fr.spacing("3w")}`,
      zIndex: 1,
      backgroundColor: fr.colors.decisions.background.default.grey.default,
      "@-moz-document url-prefix()": {
        transform: `translateX(-${fr.spacing("3w")})`,
      },
    },
    headerButtons: {
      paddingLeft: fr.spacing("3w"),
      paddingRight: fr.spacing("3w"),
      "> *": {
        marginTop: fr.spacing("2w"),
      },
    },
    configurationContainer: {
      marginTop: fr.spacing("2w"),
    },
    configurationSection: {
      marginBottom: fr.spacing("4w"),
    },
    configurationTitle: {
      display: "flex",
      alignItems: "center",
      maxHeight: configurationTitleHeight,
      ...fr.spacing("padding", { topBottom: "1w", rightLeft: "3w" }),
      marginBottom: fr.spacing("2w"),
      marginLeft: `-${fr.spacing("3w")}`,
      marginRight: `-${fr.spacing("3w")}`,
      fontWeight: 700,
      i: {
        marginRight: fr.spacing("1w"),
      },
    },
    filenamesContainer: {
      ...fr.spacing("margin", { rightLeft: "1w" }),
    },
    filesAttachedErrorMessage: {
      color: fr.colors.decisions.text.default.error.default,
    },
    filename: {
      marginBottom: fr.spacing("1w"),
      maxWidth: 235,
      overflow: "hidden",
      textOverflow: "ellipsis",
      whiteSpace: "nowrap",
    },
    chatWindow: {
      flex: 1,
      display: "flex",
      flexDirection: "column",
      height: "100%",
    },
    emptyChatWindowContainer: {
      display: "flex",
      width: "100%",
      minHeight: "60vh",
      transform: "translateY(-5%)",
      "> div": {
        margin: "auto",
      },
    },
    messageLogoContainer: {
      display: "flex",
      marginBottom: fr.spacing("4w"),
      img: {
        margin: "auto",
        width: 110,
      },
    },
    chatInstructions: {
      ...fr.spacing("padding", { topBottom: "3w", rightLeft: "4w" }),
      color: fr.colors.decisions.text.mention.grey.default,
      boxShadow: customBoxShadow,
      borderRadius: 8,
      p: {
        textAlign: "center",
      },
    },
    modesDescriptionContainer: {
      li: {
        marginBottom: fr.spacing("1w"),
        "> div": {
          display: "flex",
          alignItems: "center",
        },
      },
    },
    answerModeDescription: {
      minWidth: 440,
    },
    messagesSection: {
      height: "100%",
      width: "100%",
      overflowY: "auto",
      display: "flex",
    },
    messagesContent: {
      maxWidth: chatWindowMaxWidth,
      marginLeft: "auto",
      marginRight: "auto",
      width: "100%",
      minHeight: "60vh",
      ...fr.spacing("padding", { top: "6w", rightLeft: "3w" }),
    },
    messageContainer: {
      width: "100%",
      marginBottom: fr.spacing("2w"),
    },
    messageTitle: {
      display: "flex",
      alignItems: "center",
      color: fr.colors.decisions.text.default.grey.default,
      fontSize: "1.2rem",
      i: {
        marginRight: fr.spacing("1w"),
        "::before": {
          width: 20,
          height: 20,
        },
      },
    },
    userPromptRequestContainer: {
      marginBottom: fr.spacing("4w"),
      h4: {
        marginBottom: fr.spacing("3v"),
      },
    },
    userPromptRequestMetadata: {
      fontSize: 12,
      "> div": {
        display: "inline-block",
      },
    },
    promptParameterData: {
      display: "inline-block",
      marginRight: fr.spacing("2w"),
    },
    promptParameterLabel: {
      marginRight: fr.spacing("1v"),
      color: fr.colors.decisions.text.disabled.grey.default,
    },
    promptParameterValue: {
      fontStyle: "italic",
      fontWeight: 500,
    },
    answerSectionContainer: {
      marginBottom: fr.spacing("4w"),
    },
    errorMessageContainer: {
      display: "flex",
      alignItems: "center",
      marginBottom: fr.spacing("2w"),
      i: {
        marginRight: fr.spacing("1w"),
        color: fr.colors.decisions.text.actionHigh.redMarianne.default,
      },
    },
    sourcesContainer: {
      display: "grid",
      gridTemplateColumns: `repeat(${maxNumberOfSourcesToDisplay}, minmax(0, 1fr))`,
      gap: fr.spacing("3w"),
      width: "100%",
    },
    seeMoreSourcesContainer: {
      "&:hover": {
        [`&:hover .${classes.seeMoreSourcesMessage}`]: {
          fontWeight: 500,
        },
      },
    },
    moreCharacters: {
      display: "block",
      textAlign: "center",
    },
    seeMoreSourcesMessage: {
      marginLeft: fr.spacing("5w"),
      transform: `translateY(${fr.spacing("2w")})`,
      fontSize: ".85rem",
    },
    answerFeedbackContainer: {
      display: "none", // "flex", // no feedback for now
      div: {
        marginLeft: "auto",
        display: "flex",
        ".positiveFeedbackIcon button": {
          display: "inline-block",
          color: `${fr.colors.decisions.border.plain.blueFrance.default} !important`,
        },
        ".negativeFeedbackIcon button": {
          display: "inline-block",
          color: `${fr.colors.decisions.border.plain.redMarianne.default} !important`,
        },
        button: {
          borderRadius: "50%",
          marginLeft: fr.spacing("1w"),
        },
      },
    },
    answerSectionSeparation: {
      marginTop: fr.spacing("3w"),
    },
    scrollToBottomContainer: {
      display: "flex",
      width: "100%",
      position: "fixed",
      left: "75%",
      bottom: "50px",
      justifyContent: "center",
      alignItems: "center",
    },
    inputSection: {
      display: "flex",
      width: "100%",
      position: "relative",
      ...fr.spacing("padding", { top: "1w", rightLeft: "3w", bottom: "6w" }),
      justifyContent: "space-between",
      alignItems: "center",
      maxWidth: "49rem",
      marginLeft: "auto",
      marginRight: "auto",
    },
    llmAccuracyWarning: {
      position: "absolute",
      left: "50%",
      transform: "translateX(-50%)",
      width: "max-content",
      bottom: fr.spacing("1w"),
      fontSize: 12,
      color: fr.colors.decisions.text.disabled.grey.default,
    },
    inputContainer: {
      flexGrow: 1,
      maxWidth: chatWindowMaxWidth,
      marginLeft: "auto",
      marginRight: "auto",
      position: "relative",
    },
    scrollToBottomButton: {
      marginLeft: "auto",
      marginRight: "auto",
      backgroundColor: "var(--background-default-grey)",
    },
    input: {
      textarea: {
        maxHeight: inputMessageMaxHeight,
        overflowY: "auto",
        resize: "none",
        paddingRight: fr.spacing("8w"),
        paddingLeft: fr.spacing("8w"),
        paddingTop: fr.spacing("4v"),
      },
    },
    textAreaBorderRadius: {
      textarea: {
        borderRadius: 8,
      },
    },
    dropZoneContainer: {
      borderRadius: 8,
      position: "absolute",
      backgroundColor:
        fr.colors.decisions.background.contrast.redMarianne.default,
      color: fr.colors.decisions.background.active.redMarianne.hover,
      border: `1px dashed ${fr.colors.decisions.border.active.redMarianne.default}`,
      marginTop: 8,
      top: 0,
      left: 0,
      right: 0,
      bottom: 0,
      display: "flex",
      alignItems: "center",
      justifyContent: "center",
      fontWeight: 500,
      fontStyle: "italic",
      i: {
        marginLeft: fr.spacing("1w"),
      },
    },
    inputButtonContainer: {
      position: "absolute",
      bottom: 10,
    },

    sendMessageButtonContainer: {
      right: 16,
    },
    inputIcon: {
      borderRadius: "50%",
      display: "flex",
    },
    sendMessageIcon: {
      "::before": {
        margin: "auto",
        transform: isSubmitting ? undefined : "translate(-1px, 2px)",
      },
      ":disabled": {
        backgroundColor:
          fr.colors.decisions.background.actionHigh.blueFrance.default,
      },
    },
  }));
