import { ReactNode, useContext } from "react";
import { useNavigate } from "react-router-dom";
import Markdown from "react-markdown";
import { createModal } from "@codegouvfr/react-dsfr/Modal";
import Source from "@/chat/interfaces/Source";
import { tss } from "tss-react";
import { fr } from "@codegouvfr/react-dsfr";
import AnswerMode from "@/chat/interfaces/AnswerMode";
import SourceModalProps from "@/chat/interfaces/SourceModalProps";
import { Button } from "@codegouvfr/react-dsfr/Button";
import { forgeUrl } from "@/shared/fetchUtils";
import { AuthContext } from "@/components/AuthProvider";

// We create a modal instance that will provide use properties and methods to interact with the modal
export const sourceDetailModal = createModal({
  id: "source-modal",
  isOpenedByDefault: false,
});

/**
 * Source detail modal component
 * @param source data to display
 * @param indexPosition position of the source to display
 * @param answerMode one of the possible answer modes
 * @constructor
 */
export default function SourceDetailModal({
  source,
  indexPosition,
  answerMode,
}: SourceModalProps) {
  const { classes } = useStyles({ mode: answerMode });
  const navigate = useNavigate();
  const { accessToken } = useContext(AuthContext);

  // Remove Bearer
  const token = accessToken?.split(" ")[1];

  // These metadata will be used to dynamically construct the modal content
  const fileData: {
    label: string;
    key: keyof Source;
    condition?: (source: Source) => boolean;
    node?: (source: Source) => ReactNode;
  }[] = [
    {
      label: "Titre",
      key: "title",
    },
    {
      label: "Fichier",
      key: "filename",
      node: (source: Source) => (
        <div className={classes.filenameAndButtonContainer}>
          <span>{source.filename}</span>
          <Button
            iconId="fr-icon-draft-line"
            className={classes.downloadFileButton}
            onClick={() =>
              navigate(`/?documentId=${source.id}&pageNum=${source.page}`)
            }
            priority="tertiary no outline"
            title="Lire le document"
          />
          <Button
            iconId="fr-icon-file-download-line"
            className={classes.downloadFileButton}
            linkProps={{
              href: forgeUrl(`/documents/${source.id}/file?token=${token}`),
              target: "_blank",
            }}
            priority="tertiary no outline"
            title="Téléchargement du fichier"
          />
        </div>
      ),
    },
    {
      label: "Type",
      key: "filetype",
    },
    { label: "Page", key: "page" },
  ];

  return (
    <sourceDetailModal.Component
      title={
        <span className={classes.modalTitle}>Source n°{indexPosition}</span>
      }
      iconId="fr-icon-article-line"
    >
      {source ? (
        <>
          <div className={classes.modalSection}>
            <h6 className={classes.modalLabelTitle}>Fichier</h6>

            {/* Add similarity score to modal*/}
            <div className={classes.modalField}>
              <span className={classes.modalFileLabel}>Similarité</span>
              <span>{source.score}</span>
            </div>

            {fileData
              .filter((field) => !field.condition || field.condition(source))
              .map((field, index) => (
                <div key={index} className={classes.modalField}>
                  <span className={classes.modalFileLabel}>{field.label}</span>
                  <span>{field?.node?.(source) || source[field.key]}</span>
                </div>
              ))}
          </div>
          <div className={classes.modalSection}>
            <h6 className={classes.modalLabelTitle}>Contenu</h6>
            {/* Content come from a reliable source no problem */}
            <Markdown>{source.content}</Markdown>
          </div>
        </>
      ) : null}
    </sourceDetailModal.Component>
  );
}

const useStyles = tss.withParams<{ mode: AnswerMode }>().create(({ mode }) => ({
  modalTitle: {
    marginLeft: fr.spacing("1w"),
  },
  modalSection: {
    paddingLeft: fr.spacing("2w"),
    marginBottom: fr.spacing("3w"),
    marginTop: fr.spacing("3w"),
    borderLeft: `4px solid`,
    borderColor:
      mode === "collection"
        ? fr.colors.decisions.border.open.blueFrance.default
        : fr.colors.decisions.border.open.redMarianne.default,
  },
  modalField: {
    marginBottom: fr.spacing("1v"),
  },
  modalLabelTitle: {
    marginBottom: fr.spacing("2w"),
    fontSize: "1rem",
  },
  modalFileLabel: {
    display: "inline-block",
    color: fr.colors.decisions.text.disabled.grey.default,
    minWidth: 70,
    fontSize: ".75rem",
    fontWeight: 500,
  },
  filenameAndButtonContainer: {
    display: "inline-flex",
    alignItems: "center",
  },
  downloadFileButton: {
    marginLeft: fr.spacing("1w"),
    borderRadius: "50%",
  },
}));
