import { useContext, useState, ChangeEvent, FormEvent } from "react";
import { Input } from "@codegouvfr/react-dsfr/Input";
import { PasswordInput } from "@codegouvfr/react-dsfr/blocks/PasswordInput";
import { Button } from "@codegouvfr/react-dsfr/Button";
import { forgeUrl } from "./shared/fetchUtils";
import { AuthContext } from "@/components/AuthProvider";

export function LoginForm() {
  const [loginError, setLoginError] = useState<undefined | string>(undefined);
  const { setAccessToken } = useContext(AuthContext);

  const [formData, setFormData] = useState({
    username: "",
    password: "",
  });

  const handleChange = (e: ChangeEvent<HTMLInputElement>) => {
    console.log("change");
    const { name, value } = e.target;
    setFormData({
      ...formData,
      [name]: value,
    });
  };

  const onLoginSubmit = async (e: FormEvent<HTMLFormElement>) => {
    e.preventDefault();
    const data = new FormData();
    data.append("username", formData.username);
    data.append("password", formData.password);
    setLoginError(undefined);
    const res = await fetch(forgeUrl("/login/access-token"), {
      method: "POST",
      body: data,
    });

    if (res.ok) {
      const token = await res.json();
      setAccessToken(`${token.token_type} ${token.access_token}`);
    } else {
      const error = await res.json();
      // details can also be an array of error, we want to keep the code simple
      // since the login form is not meant for end users
      if (typeof error.detail === "string") {
        setLoginError(error.detail);
      } else {
        setLoginError("Erreur merci de renseigner tous les champs");
      }
    }
  };

  return (
    <form onSubmit={onLoginSubmit}>
      <Input
        label="E-mail"
        nativeInputProps={{
          name: "username",
          onChange: handleChange,
          value: formData.username,
        }}
        state={loginError ? "error" : undefined}
        stateRelatedMessage={loginError}
      />
      <PasswordInput
        label="Mot de passe"
        nativeInputProps={{
          name: "password",
          onChange: handleChange,
          value: formData.password,
        }}
      />
      <Button type="submit">Se connecter</Button>
    </form>
  );
}
