import { useState, useEffect, useContext } from "react";
import { fr } from "@codegouvfr/react-dsfr";
import { cx } from "@codegouvfr/react-dsfr/tools/cx";
import { Highlight } from "@codegouvfr/react-dsfr/Highlight";

import PdfViewerComponent from "@/components/PdfViewerComponent";
import ViewerSidePanel from "@/components/ViewerSidePanel";
import { LiriaeDocument } from "@/shared/types";
import { AuthContext } from "@/components/AuthProvider";

export function Home({
  onFullScreen,
  isFullScreen,
}: {
  onFullScreen: () => void;
  isFullScreen: boolean;
}) {
  const [currentDocument, setCurrentDocument] = useState<LiriaeDocument | null>(
    null
  );
  const [initialPageNum, setInitialPageNum] = useState(0);
  const { currentCollection, isProcessingServiceAvailable, liriaeDocuments } =
    useContext(AuthContext);

  const onDocumentSelected = (
    document: LiriaeDocument | null,
    pageNum?: number
  ) => {
    setCurrentDocument(document);
    if (pageNum !== undefined) {
      setInitialPageNum(pageNum);
    }
  };

  if (!currentCollection) {
    return (
      <div className={cx(fr.cx("fr-container"), fr.cx("fr-p-32v"))}>
        Impossible de créer ou récupérer une collection. Essayez d'en créer une
        dans le panel d'administration.
      </div>
    );
  }

  return (
    <>
      {(!isProcessingServiceAvailable && liriaeDocuments.length > 0) && (
        <Highlight
          className={fr.cx("fr-pt-2v")}
          classes={{
            root: "fr-highlight--brown-caramel",
          }}
        >
          Le système de traitement des dossiers par IA est momentanément
          inaccessible. Veuillez nous excusez pour la gêne occasionnée.
        </Highlight>
      )}
      <div
        className={
          isFullScreen ? fr.cx("fr-grid-row", "fr-ml-4v") : fr.cx("fr-grid-row")
        }
      >
        <div className={fr.cx("fr-col-2")}>
          <ViewerSidePanel
            selectedDocument={currentDocument}
            onSelection={onDocumentSelected}
          />
        </div>
        <div className={fr.cx("fr-col-10")}>
          <div className="PDF-viewer">
            {currentDocument ? (
              <PdfViewerComponent
                useDocEngine={import.meta.env.VITE_USE_DOCENGINE !== undefined}
                document={currentDocument}
                onFullScreen={onFullScreen}
                initialPageNum={initialPageNum}
              />
            ) : (
              <div className={cx(fr.cx("fr-container"), fr.cx("fr-p-32v"))}>
                Veuillez cliquer sur un document ou en ajouter un.
              </div>
            )}
          </div>
        </div>
      </div>
    </>
  );
}
