#' graphique en série temporelle à facette
#'
#' @param data Le dataframe avec les données ecln.
#' @param type_logement Appartements ou Maisons.
#' @param indicateurs Un vecteur d'indicateur à visualiser dans le graphique.
#' @param type_zone Liste des types de zonage à intégrer dans la facette.
#' @param titre Le titre du graphique.
#' @param bas_de_page Le bas de page du graphique.
#' @param type_facet Type de facette : classique ou grille pour l'utilisation d'une matrice.
#' @param ncol_facet Le nombre de colonnes dans la facette.
#' @return Un graphique ggplot2.
#' @export
#' @importFrom dplyr filter mutate
#' @importFrom ggplot2 ggplot aes geom_line facet_wrap scale_y_continuous scale_x_date labs
#' @importFrom gouvdown scale_color_gouv_discrete
#' @importFrom scales format_format
#' @importFrom geofacet facet_geo
creer_graphique_series_temporelles <- function(data = indic_ecln,
                                               type_logement = "Appartements",
                                               indicateurs = c("Encours", "Mises en vente", "Ventes"),
                                               type_zone = c("France", "R\u00e9gions", "D\u00e9partements"),
                                               titre = "",
                                               bas_de_page = "",
                                               type_facet = "classique",
                                               ncol_facet = 3) {
  indic <- paste0(indicateurs, " - ", type_logement)
  p <- data %>%
    dplyr::filter(
      .data$Indicateur %in% indic,
      .data$TypeZone %in% type_zone,
      .data$TypeIndicateur == "Cumul annuel"
    ) %>%
    dplyr::mutate(Indicateur = str_split_fixed(.data$Indicateur, " - ", 2)[, 1]) %>%
    ggplot2::ggplot() +
    ggplot2::aes(x = .data$Periode, y = .data$Valeur, color = .data$Indicateur, fill = .data$Indicateur, group = .data$Indicateur) +
    ggplot2::geom_line(size = 1.2) +
    ggplot2::theme(legend.position = c(0.85, 0.15),
                   legend.title = element_blank(),
                   legend.key.size = unit(1, "cm"),
                   panel.grid.major = element_line(color = "lightgrey", linetype = "solid"),
                   panel.grid.minor = element_blank(),
                   strip.text = element_text(hjust = 0.5, face = "bold", size=14),
                   strip.background = element_blank(),
                   axis.text.x = element_text(size=11, hjust=-0.1),
                   axis.line = element_line(colour = "black", size = 0.5, linetype = "solid")) +
    ggplot2::scale_y_continuous(expand = c(0, 0), labels = scales::format_format(big.mark = " "), limits = c(0, NA)) +
    ggplot2::scale_x_date(date_labels = "%y", expand = c(0, 0), date_breaks = "1 year") +
    # gouvdown::scale_color_gouv_discrete(palette = "pal_gouv_qual2") +
    ggplot2::scale_colour_manual(values = c("#E1000F", "#808080", "#000091"), breaks=c("Ventes","Mises en vente","Encours")) +
    ggplot2::labs(
      title = titre,
      x = "",
      y = "",
      fill = "",
      color = "",
      caption = bas_de_page
    )
  if (type_facet == "classique") {
    p <- p +
      ggplot2::facet_wrap(~ .data$Zone, scales = "free", ncol = ncol_facet)
  }

  if (type_facet == "grille") {
    p <- p +
      geofacet::facet_geo(~Zone, grid = mygrid, scales = "free", drop = TRUE)
  }
  return(p)
}
