#' Graphique d'évolution de l'investissement locatif
#'
#' @param data le dataframe avec les données ecln
#' @param type_logement Appartements ou Maisons
#' @param type_indicateur cumul annuel ou valeur trimestrielle
#' @param type_zone liste des types de zonage à intégrer dans la facette
#' @param titre titre du graphique
#' @param soustitre sous titre du graphique
#' @param bas_de_page bas de page
#' @param type_facet Type de facette : classique ou grille pour l'utilisation d'une matrice.
#' @param ncol_facet Le nombre de colonnes dans la facette.
#'
#' @return Un graphique ggplot2.
#' @export
#' @importFrom dplyr filter mutate
#' @importFrom forcats fct_drop
#' @importFrom ggplot2 ggplot aes geom_area scale_x_date scale_y_continuous labs facet_wrap
#' @importFrom scales format_format
#' @importFrom stringr str_split_fixed
#' @importFrom geofacet facet_geo
creer_graphique_investissement_locatif <- function(data = indic_ecln,
                                                   type_logement = "Appartements",
                                                   type_zone = c("R\u00e9gions", "D\u00e9partements"),
                                                   type_indicateur = "Cumul annuel",
                                                   titre = "",
                                                   soustitre = "",
                                                   bas_de_page = "",
                                                   type_facet = "classique",
                                                   ncol_facet = 4) {
  indic <- c(
    "Ventes hors investissement locatif - Appartements",
    "Ventes en investissement locatif - Appartements"
  )
  if (type_logement == "Maisons") {
    indic <- c(
      "Ventes hors investissement locatif - Maisons",
      "Ventes en investissement locatif - Maisons"
    )
  }

  data_prep <- data %>%
    dplyr::filter(
      .data$Indicateur %in% indic,
      .data$TypeIndicateur == type_indicateur,
      .data$TypeZone %in% type_zone
    ) %>%
    dplyr::mutate(
      Zone = forcats::fct_drop(.data$Zone),
      Indicateur = stringr::str_split_fixed(.data$Indicateur, " - ", 2)[, 1]
    )


  p <- ggplot2::ggplot(
    data = data_prep,
    ggplot2::aes(
      x = .data$Periode,
      y = .data$Valeur,
      color = .data$Indicateur,
      fill = .data$Indicateur,
      group = .data$Indicateur
    )
  ) +
    ggplot2::geom_area(alpha = .5) +
    ggplot2::scale_x_date(date_labels = "%y", date_breaks = "1 year") +
    ggplot2::scale_y_continuous(labels = scales::format_format(big.mark = " ")) +
    ggplot2::labs(
      title = titre,
      subtitle = soustitre,
      x = "",
      y = "",
      fill = "",
      color = "",
      caption = bas_de_page
    )
  if (nlevels(data_prep$Zone) > 1) {
    if (type_facet == "classique") {
      p <- p +
        ggplot2::facet_wrap(~ .data$Zone, scales = "free", ncol = ncol_facet)
    }

    if (type_facet == "grille") {
      p <- p +
        geofacet::facet_geo(~Zone, grid = mygrid, scales = "free", drop = TRUE)
    }
  }
  return(p)
}
