#' graphique sur un territoire sur un indicateur en moyenne annuelle et valeur trimestrielle
#'
#' @param data Le dataframe avec les données ecln.
#' @param indicateur  L'indicateur à visualiser.
#' @param zone Le territoire sur lequel visualiser l'indicateur.
#' @param titre Le titre du graphique.
#'
#' @return  un ggplot2
#' @export
#' @importFrom dplyr select distinct pull filter mutate
#' @importFrom forcats fct_recode
#' @importFrom ggforce geom_mark_circle
#' @importFrom ggplot2 ggplot aes geom_line geom_point margin theme labs guides scale_y_continuous scale_x_date element_text element_line
#' @importFrom glue glue
#' @importFrom gouvdown gouv_colors scale_color_gouv_discrete scale_fill_gouv_discrete
#' @importFrom grid unit
#' @importFrom lubridate quarter year
#' @importFrom stringr str_c
creer_graphique_evolution_trim <- function(data = indic_ecln,
                                           indicateur = "Ventes - Logements",
                                           zone = "Pays de la Loire",
                                           titre = "Ventes de logements neufs dans les Pays de la Loire") {
  per <- data %>%
    dplyr::select(.data$Periode) %>%
    dplyr::distinct() %>%
    dplyr::pull(.data$Periode)

  max_date <- max(per)
  num_trimestre <- as.numeric(lubridate::quarter(max_date))
  rang_trimestre <- if( num_trimestre == 1){
    "Premier"
  }else if( num_trimestre == 2){
    "Deuxi\u00E8me"
  }else if( num_trimestre == 3){
    "Troisi\u00E8me"
  }else{
    "Quatri\u00E8me"
  }

  # sous_titre <- glue::glue("Au {label_rang(lubridate::quarter(max_date))} trimestre {lubridate::year(max_date)}")
  sous_titre <- glue::glue("{rang_trimestre} trimestre de {lubridate::year(max_date)}")
  label <- data %>%
    dplyr::filter(
      .data$Zone == zone,
      .data$Indicateur == indicateur
    ) %>%
    dplyr::filter(.data$Periode == max(.data$Periode)) %>%
    dplyr::mutate(
      Valeur = ifelse(.data$TypeIndicateur == "Cumul annuel", .data$Valeur / 4, .data$Valeur),
      TypeIndicateur = forcats::fct_recode(.data$TypeIndicateur,
        "Moyenne annuelle" = "Cumul annuel",
        "Valeur trimestrielle" = "Trimestriel"
      )
    ) %>%
    dplyr::mutate(
      description = stringr::str_c(
        format(round(.data$Valeur), big.mark = " "),
        "\n",
        ifelse(.data$TauxEvolution12Mois > 0, "+ ", ""),
        format(round(.data$TauxEvolution12Mois, 1), big.mark = " ", decimal.mark = ","),
        " %"
      ),
      # label = ifelse(.data$TypeIndicateur %in% c("Valeur trimestrielle"),
      #   FormatCaractere(.data$Periode),
      #   "Moyenne\nannuelle"
      # )
      label = ifelse(.data$TypeIndicateur %in% c("Valeur trimestrielle"),
                     "Valeur\ndu trimestre",
                     "Moyenne\nannuelle"
      )
    )

  gg <- data %>%
    dplyr::filter(
      .data$Zone == zone,
      .data$Indicateur == indicateur
    ) %>%
    dplyr::mutate(
      Valeur = ifelse(.data$TypeIndicateur == "Cumul annuel", .data$Valeur / 4, .data$Valeur),
      TypeIndicateur = forcats::fct_recode(.data$TypeIndicateur,
        "Moyenne annuelle" = "Cumul annuel",
        "Valeur trimestrielle" = "Trimestriel"
      )
    ) %>%
    ggplot2::ggplot() +
    ggplot2::aes(x = .data$Periode, y = .data$Valeur, group = .data$TypeIndicateur, linetype = .data$TypeIndicateur, color = .data$TypeIndicateur) +
    ggplot2::geom_line(size = 1.2) +
    ggplot2::geom_point(data = label, ggplot2::aes(x = .data$Periode, y = .data$Valeur, group = .data$TypeIndicateur, color = .data$TypeIndicateur)) +
    ggforce::geom_mark_circle(
      data = label,
      ggplot2::aes(label = .data$label,
                   fill = .data$TypeIndicateur,
                   description = .data$description),
      label.buffer = grid::unit(5, "mm"),  #longueur du trait
      expand = grid::unit(0, "mm"),  #diametre du cercle sur le dernier point (0 pour le supprimer)
      label.margin = ggplot2::margin(2, 5, 2, 5, "mm"),  #marges autour du texte (haut, gauche, bas, droite)
      # label.colour = c("red", "blue"),   #couleur du texte
      label.colour = gouvdown::gouv_colors("r1"),
      label.fontsize = 9,
      # alpha = 0.5, #transparence du cercle
      label.fill = NA
      # label.fill = gouvdown::gouv_colors("bleu_france") #couleur de la boite d'annotation
    ) +
    ggplot2::guides(linetype = guide_legend(override.aes = list(fill = NA))) +
    ggplot2::theme(title = element_text(colour = "#0f0f0f"),
                   legend.position = "bottom",
                   legend.title = element_blank(),
                   legend.key = element_rect(colour = NA, fill = NA),
                   legend.background=element_blank(),
                   panel.grid.major = element_line(colour = "lightgrey", linetype = "solid"),
                   panel.grid.minor = element_blank(),
                   axis.title = element_blank(),
                   axis.text.x = element_text(size=11, hjust=-0.1),
                   axis.line = element_line(size = 0.5, linetype = "solid")) +
    # gouvdown::scale_color_gouv_discrete(palette = "pal_gouv_qual2") +
    # gouvdown::scale_fill_gouv_discrete(palette = "pal_gouv_qual2") +
    ggplot2::scale_colour_manual(values = c("#E1000F", "#000091")) +
    ggplot2::scale_fill_manual(values = c("#E1000F","#000091")) +
    ggplot2::labs(x = "", y = "", title = titre, subtitle = sous_titre,
                  caption = "Source : SDES - Enqu\u00EAte sur la commercialisation des logements neufs (ECLN)\nMoyenne annuelle : valeur observ\u00e9e en moyenne lors des 4 derniers trimestres") +
    ggplot2::theme(plot.caption = element_text(size = 10))+
    ggplot2::scale_y_continuous(expand = c(0, 0), labels = function(l) format(l, scientific = FALSE, big.mark = " ", decimal.mark = ","), limits = c(0, NA)) +
    ggplot2::scale_x_date(date_breaks = "1 year", date_labels = "%Y", expand = c(0, 0), limits = c(min(per), max(per) + months(12)))

  return(gg)
}

