#' graphique barre sur un an
#'
#' @param data Le dataframe.
#' @param type_logement Collectifs ou individuels.
#' @param type_zone Liste des types de zonage à intégrer dans la facette.
#' @param titre Le titre du graphique.
#' @param bas_de_page Le bas de page du graphique.
#' @param type_facet Type de facette : classique ou grille pour l'utilisation d'une matrice.
#' @param ncol_facet Le nombre de colonnes dans la facette.
#'
#' @return Un graphique ggplot2.
#' @importFrom dplyr filter mutate arrange
#' @importFrom ggplot2 ggplot aes geom_bar coord_flip geom_label facet_wrap scale_y_continuous labs position_dodge
#' @importFrom scales format_format
#' @importFrom lubridate %m-%
#' @importFrom geofacet facet_geo
#' @export
creer_graphique_evolution_annuelle <- function(data = indic_ecln,
                                               type_zone = c("France", "R\u00e9gions", "D\u00e9partements"),
                                               type_logement = "collectif",
                                               titre = "",
                                               bas_de_page = "",
                                               type_facet = "classique",
                                               ncol_facet = 3) {

  indic <- c(
    "Encours - Logements",
    "Mises en vente - Logements",
    "Ventes - Logements"
  )
  if (type_logement == "individuel") {
    indic <- c(
      "Encours - Maisons",
      "Mises en vente - Maisons",
      "Ventes - Maisons"
    )
  }
  if (type_logement == "collectif") {
    indic <- c(
      "Encours - Appartements",
      "Mises en vente - Appartements",
      "Ventes - Appartements"
    )
  }



  df <- data %>%
    dplyr::filter(
      .data$Indicateur %in% indic,
      .data$TypeZone %in% type_zone,
      .data$TypeIndicateur == "Cumul annuel",
      .data$Periode == max(.data$Periode) | .data$Periode == max(.data$Periode) %m-% years(1)
    ) %>%
    dplyr::mutate(
      Periode = FormatCaractere(.data$Periode),
      Indicateur = str_split_fixed(.data$Indicateur, " - ", 2)[, 1]
    ) %>%
    dplyr::arrange(.data$Zone)

  p <- ggplot2::ggplot(data = df, ggplot2::aes(x = .data$Periode, weight = .data$Valeur, fill = .data$Indicateur)) +
    ggplot2::geom_bar(position = ggplot2::position_dodge()) +
    ggplot2::coord_flip() +
    ggplot2::theme(legend.position = c(0.85, 0.15),
                   legend.title = element_blank(),
                   legend.key.size = unit(1, "cm"),
                   panel.grid.major = element_line(color = "lightgrey", linetype = "solid"),
                   panel.grid.minor = element_blank(),
                   strip.text = element_text(hjust = 0.5, face = "bold", size = 14),
                   strip.background = element_blank(),
                   axis.text.x = element_text(size = 9),
                   axis.line = element_line(size = 0.5, linetype = "solid")) +
    ggplot2::geom_text(
      data = df,
      ggplot2::aes(label = format(round(.data$Valeur), big.mark = " "), x = .data$Periode, y = .data$Valeur, group = .data$Indicateur),
      position = position_dodge(0.9), color = "white", size = 4, alpha = 1, fontface = "bold", hjust = 1.2) +
    ggplot2::scale_y_continuous(expand = c(0, 0), labels = scales::format_format(big.mark = " ")) +
    # gouvdown::scale_fill_gouv_discrete(palette = "pal_gouv_qual2") +
    ggplot2::scale_fill_manual(values = c("#E1000F", "#808080", "#000091"), breaks=c("Ventes","Mises en vente","Encours")) +
    ggplot2::labs(title = titre, x = "Trimestre", y = "", caption = bas_de_page, fill = "")
  if (type_facet == "classique") {
    p <- p +
      ggplot2::facet_wrap(~ .data$Zone, scales = "free", ncol = ncol_facet)
  }

  if (type_facet == "grille") {
    p <- p +
      geofacet::facet_geo(~Zone, grid = mygrid, scales = "free", drop = TRUE)
  }
  return(p)
}
