#' Converti une date en trimestre
#'
#' @param date une date
#'
#' @return une chaine de caractère
#' @importFrom dplyr case_when
#' @importFrom lubridate month year
en_trimestre <- function(date) {
  trim <- dplyr::case_when(
    lubridate::month(date) <= 3 ~ "1<sup>er</sup>",
    lubridate::month(date) <= 6 ~ "2<sup>\u00e8me</sup>",
    lubridate::month(date) <= 9 ~ "3<sup>\u00e8me</sup>",
    lubridate::month(date) <= 12 ~ "4<sup>\u00e8me</sup>"
  )
  return(
    paste(trim, "trimestre de", lubridate::year(date))
  )
}

#' Convertit une date en trimestre de l'année n moins un
#'
#' @param date une date
#'
#' @return une chaine de caractère
#' @importFrom dplyr case_when
#' @importFrom lubridate month year
en_trimestre_moins_un <- function(date) {
  trim <- dplyr::case_when(
    lubridate::month(date) <= 3 ~ "1<sup>er</sup>",
    lubridate::month(date) <= 6 ~ "2<sup>\u00e8me</sup>",
    lubridate::month(date) <= 9 ~ "3<sup>\u00e8me</sup>",
    lubridate::month(date) <= 12 ~ "4<sup>\u00e8me</sup>"
  )
  return(
    paste(trim, "trimestre de", lubridate::year(date)-1)
  )
}

#' Fonction utilitaire de formatage en pourcentage pour le français
#'
#' @description fonction utilitaire de formatage en pourcentage pour le français

#' @param x un nombre à formater en pourcentage
#' @param dec un entier désignant le nombre de chiffres après la virgule souhaité (1 par défaut)
#' @param sign TRUE pour avoir le signe +/-
#'
#' @return une chaîne de texte, x %, avec transformation de la décimale en virgule et insertion d'espaces insécables
#'
#' @importFrom attempt stop_if_not
format_fr_pct <- function(x, dec = 1, sign = TRUE) {
  attempt::stop_if_not(x, is.numeric, msg = "x n'est pas un nombre, revoyez la saisie de l'argument de format_fr_pct(x, dec)")
  if (sign == FALSE) {
    res <- paste0(formatC(abs(x), decimal.mark = ",", big.mark = "\u202f", format = "f", digits = dec), "\u00a0%")
  }
  if (sign == TRUE) {
    res <- paste0(formatC(x, decimal.mark = ",", big.mark = "\u202f", format = "f", digits = dec), "\u00a0%")
    if (x > 0) {
      res <- paste0("+", "\u202f", res)
    }
  }
  return(res)
}

#' Fonction utilitaire de formatage de nombre pour le francais
#'
#' @description fonction utilitaire de formatage de nombre pour le francais

#' @param x un nombre à formater en français
#' @param dec un entier désignant le nombre de chiffres après la virgule souhaité (1 par défaut)
#' @param big_mark le séparateur des milliers
#' @param sign TRUE pour avoir le signe +/-
#' @return une chaîne de texte avec transformation de la décimale en virgule et insertion d'un caractère spécifié via big_mark au niveau du séparateur de milliers
#' @importFrom attempt stop_if_not
format_fr_nb <- function(x, dec = 1, big_mark, sign = FALSE) {
  attempt::stop_if_not(x, is.numeric, msg = "x n'est pas un nombre, revoyez la saisie de l'argument de format_fr_pct(x, dec)")
  if (missing("big_mark")) {
    big_mark <- "\u202f"
  }
  if (sign == FALSE) {
    x <- abs(x)
  }
  res <- paste0(formatC(x, decimal.mark = ",", format = "f", digits = dec, big.mark = big_mark))
  if (sign == TRUE & x > 0) {
    res <- paste0("+", res)
  }
  return(res)
}



#' Chiffres clefs pour la publication
#'
#' @param data le dataframe avec les données ecln
#' @param indicateur Ventes, Mises en vente ou encours
#' @param type_logement Appartements ou Maisons
#' @param type_indicateur cumul annuel ou valeur trimestrielle
#' @param texte_log texte en fonction du type de logement
#'
#' @return du code html
#' @export
#' @importFrom dplyr filter
#' @importFrom glue glue
#' @importFrom drealdown my_icon
creer_chiffres_clefs_ecln <- function(data = indic_ecln,
                                      type_logement = "Logements",
                                      indicateur = "Ventes",
                                      type_indicateur = "Trimestriel",
                                      texte_log = "logements neufs vendus") {
  indic <- paste0(indicateur, " - ", type_logement)

  df <- data %>%
    dplyr::filter(
      .data$Indicateur == indic,
      .data$TypeIndicateur == type_indicateur,
      .data$TypeZone == "R\u00e9gions",
      .data$Periode == max(.data$Periode)
    )
  val <- list(
    periode = en_trimestre(df$Periode),
    periode_moins_un = en_trimestre_moins_un(df$Periode),
    valeur = format_fr_nb(df$Valeur, dec = 0),
    evolution = format_fr_pct(df$TauxEvolution12Mois, dec = 1)
  )
  res <- glue::glue("{drealdown::my_icon('la-building', size=2)} <b>{val$valeur}</b> {texte_log}<br>au cours du {val$periode}<br>
                    {drealdown::my_icon('la-line-chart', size=2)} <b>{val$evolution}</b> par rapport au {val$periode_moins_un}")
  return(res)
}
