#' Création de la carte en volume sur les indicateurs de commercialisation de logements neufs
#'
#' @param data Le dataframe.
#' @param code_region Le code de la région.
#' @param indicateurs Nom de l'indicateur.
#' @param type_logement Appartements ou Maisons.
#' @param type_indicateur Cumul annuel ou Trimestriel
#' @param titre Le titre du graphique.
#' @param titre_legende Le titre de la légende.
#' @param bas_de_page Le bas de page du graphique.
#'
#' @return une carte en ggplot2
#' @export
#' @importFrom COGiter filtrer_cog_geo list_epci_in_reg nom_zone
#' @importFrom dplyr filter rename left_join mutate
#' @importFrom ggplot2 ggplot geom_sf stat_sf_coordinates coord_sf scale_size theme element_rect element_blank labs
#' @importFrom ggspatial annotation_north_arrow north_arrow_fancy_orienteering annotation_scale
#' @importFrom gouvdown theme_gouv_map
#' @importFrom mapfactory fond_carto
#' @importFrom sf st_bbox st_crs
#' @importFrom forcats fct_drop fct_relevel
creer_carte_volume<-function(data=indic_ecln,
                             code_region = params$reg,
                             indicateurs = c("Encours","Mises en vente","Ventes"),
                             type_logement="Appartements",
                             type_indicateur="Cumul annuel",
                             titre="",
                             titre_legende="",
                             bas_de_page=""){
  # On ne garde que les données des ventes pour réaliser les classes de valeur
  indic <- paste0(indicateurs," - ",type_logement)

  epci_geo <- COGiter::filtrer_cog_geo(reg = code_region,garder_supra = TRUE)$epci

  data_pour_carte<-data %>%
    dplyr::filter(.data$Indicateur %in% indic,
                  .data$TypeZone == "Epci",
                  .data$Periode == max(.data$Periode),
                  .data$TypeIndicateur == type_indicateur
    ) %>%
    dplyr::rename(EPCI=.data$CodeZone)

  epci_reg <- COGiter::list_epci_in_reg(code_region)

  data_prep<-epci_geo %>%
    dplyr::left_join(data_pour_carte) %>%
    dplyr::mutate(epci_reg = .data$EPCI %in% epci_reg,
                  indicateur_positif = .data$Valeur>0) %>%
    dplyr::filter(!is.na(.data$Source)) %>%
    dplyr::mutate(Indicateur = forcats::fct_drop(.data$Indicateur),
                  Indicateur = forcats::fct_relevel(.data$Indicateur,"Ventes - Appartements",
                                                    "Mises en vente - Appartements","Encours - Appartements",
                                                    "Ventes - Maisons",
                                                    "Mises en vente - Maisons","Encours - Maisons"))


  # Pour la Corse : définition de l'emprise de la carte : on veut 30 km de vide autour de la région et une carte plus large que haute
  bbox_reg <- sf::st_bbox(data_prep %>% filter(epci_reg))
  fond_carte <- mapfactory::fond_carto(nom_reg = COGiter::nom_zone("R\u00e9gions",code_region))

  if(code_region == "94"){
    hauteur = bbox_reg$ymax - bbox_reg$ymin + 30000
    largeur = bbox_reg$xmax - bbox_reg$xmin + 30000
    x_mil = bbox_reg$xmin + largeur/2
    y_min = bbox_reg$ymin - 15000
    y_max = bbox_reg$ymax + 15000
    x_min = ifelse(largeur >= hauteur, bbox_reg$xmin - 15000, x_mil - hauteur/2)
    x_max = ifelse(largeur >= hauteur, bbox_reg$xmax + 15000, x_mil + hauteur/2)
  } else {
    y_min = bbox_reg$ymin
    y_max = bbox_reg$ymax
    x_min = bbox_reg$xmin
    x_max = bbox_reg$xmax
  }


  p<-ggplot2::ggplot() +
    ggplot2::geom_sf(data = fond_carte$epci, fill = "light grey", color = "white", size = .1) +
    ggplot2::geom_sf(data = fond_carte$reg_ombre, fill = "dark grey", color = "light grey") +
    ggplot2::geom_sf(data = fond_carte$regions %>% filter(.data$REG == code_region), fill = "grey", color = "white",size = .2) +
    ggplot2::geom_sf(data=data_prep %>% dplyr::filter(.data$epci_reg), color="white",size=.1,fill = "#B8B8B8")+
    ggplot2::geom_sf(data=data_prep %>% dplyr::filter(.data$epci_reg,.data$indicateur_positif), color="white",size=.1,fill = "#fff5f1")+
    ggplot2::geom_sf(data = fond_carte$departements,fill=NA,color = "white",size = .2) +
    ggplot2::stat_sf_coordinates(data=data_prep %>% dplyr::filter(.data$epci_reg,.data$indicateur_positif), aes(size = .data$Valeur),shape = 21, color = "white", fill = "#000091") +
    ggplot2::coord_sf(
      xlim = c(x_min, x_max),
      ylim = c(y_min, y_max),
      expand = FALSE,
      crs = sf::st_crs(fond_carte$epci),
      datum = NA
    ) +
    ggplot2::scale_size(range = c(5/length(indic),20/length(indic))) +
    ggspatial::annotation_north_arrow(location = "br", style = ggspatial::north_arrow_fancy_orienteering) +
    ggspatial::annotation_scale() +
    gouvdown::theme_gouv_map(base_size=12,
                             strip_text_size = 12,
                             plot_title_size = 20,
                             subtitle_size = 16) +
    ggplot2::theme(panel.background = ggplot2::element_rect(fill = "light blue"),
                   plot.background = ggplot2::element_rect(fill="#ffffff",color="#ffffff"),
                   legend.position = "bottom",
                   legend.title = ggplot2::element_blank(),
                   strip.text = element_text(hjust = 0.5, face = "bold", size=14),
                   strip.background = element_blank(),
                   plot.caption.position = "plot") +
    ggplot2::labs(fill=titre_legende,
         caption=bas_de_page,
         title=titre,
         x = NULL,y=NULL)


  if (length(indic) > 1) {
    p <-p+ facet_wrap(~.data$Indicateur,ncol=3)
  }
  return(p)
}



#' Création de la carte en évolution sur les indicateurs de commercialisation de logements neufs
#'
#' @param data Le dataframe.
#' @param code_region Le code de la région.
#' @param indicateurs Nom de l'indicateur.
#' @param type_logement Appartements ou Maisons.
#' @param type_indicateur Cumul annuel ou Trimestriel
#' @param titre Le titre du graphique.
#' @param titre_legende Le titre de la légende.
#' @param bas_de_page Le bas de page du graphique.
#'
#' @return une carte en ggplot2
#' @export
#' @importFrom COGiter filtrer_cog_geo list_epci_in_reg nom_zone
#' @importFrom dplyr filter rename left_join mutate
#' @importFrom forcats fct_explicit_na fct_relevel
#' @importFrom ggplot2 ggplot geom_sf aes scale_fill_manual coord_sf guides guide_legend theme element_rect element_blank labs
#' @importFrom ggspatial annotation_north_arrow north_arrow_fancy_orienteering annotation_scale
#' @importFrom gouvdown theme_gouv_map
#' @importFrom mapfactory fond_carto
#' @importFrom sf st_bbox st_crs
#' @importFrom stringr str_split_fixed
#'
creer_carte_evolution<-function(data=indic_ecln,
                                code_region = params$reg,
                                indicateurs = c("Encours","Mises en vente","Ventes"),
                                type_logement="Appartements",
                                type_indicateur="Cumul annuel",
                                titre="",
                                titre_legende="",
                                bas_de_page="") {

  indic <- paste0(indicateurs," - ",type_logement)

  epci_geo <- COGiter::filtrer_cog_geo(reg = code_region,garder_supra = TRUE)$epci

  data_pour_carte<-data %>%
    dplyr::filter(.data$Indicateur %in% indic,
                  .data$TypeZone == "Epci",
                  .data$Periode == max(.data$Periode),
                  .data$TypeIndicateur == type_indicateur
    ) %>%
    dplyr::rename(EPCI=.data$CodeZone)

  epci_reg <- COGiter::list_epci_in_reg(code_region)

  data_prep<-epci_geo %>%
    dplyr::left_join(data_pour_carte) %>%
    dplyr::filter(!is.na(.data$Source))

  bks <- c(-Inf,-10,0,10,50,Inf)

  data_prep<-data_prep %>%
    dplyr::mutate(TauxEvolution12Moisc=cut(.data$TauxEvolution12Mois,breaks=bks,
                                           labels=c(
                                             paste("Inf\u00e9rieur \u00e0",round(bks[2],0),"%"),
                                             paste("De",round(bks[2],0),"% \u00e0",round(bks[3],0),"% inclus"),
                                             paste("De",round(bks[3],0),"% \u00e0 +",round(bks[4],0),"% inclus"),
                                             paste("De +",round(bks[4],0),"% \u00e0 +",round(bks[5],0),"% inclus"),
                                             paste("Plus de +",round(bks[5],0),"%")),
                                           # labels=c("-10%","0%","10%","50%",""),
                                           ordered_result = T,include.lowest = T),
                  TauxEvolution12Moisc=forcats::fct_explicit_na(.data$TauxEvolution12Moisc,na_level="Secret statistique ou\n\u00e9volution non significative"),
                  epci_reg = .data$EPCI %in% epci_reg,
                  indicateur_positif = .data$Valeur>0)

  liste_facteurs <- c("Aucune activit\u00e9",levels(data_prep$TauxEvolution12Moisc))

  data_prep<-data_prep  %>%
    dplyr::mutate(TauxEvolution12Moisc= dplyr::case_when(
      .data$TauxEvolution12Mois == Inf ~ "Secret statistique ou\n\u00e9volution non significative",
      .data$TauxEvolution12Mois == "NaN" ~ "Aucune activit\u00e9",
      TRUE ~ .data$TauxEvolution12Moisc  )) %>%
    dplyr::mutate(Indicateur = forcats::fct_drop(.data$Indicateur),
                  Indicateur = forcats::fct_relevel(.data$Indicateur,"Ventes - Appartements",
                                                    "Mises en vente - Appartements","Encours - Appartements",
                                                    "Ventes - Maisons",
                                                    "Mises en vente - Maisons","Encours - Maisons"))%>%
    dplyr::mutate(TauxEvolution12Moisc=forcats::fct_relevel(.data$TauxEvolution12Moisc,liste_facteurs))%>%
    dplyr::mutate(Indicateur2 = .data$Indicateur %>%
                    stringr::str_split_fixed(., pattern = "-", n = 2) %>% # scinde la colonne en 2
                    .[, 1]) #garde la 1ere valeur

  color<-c("white",
           gouvdown::gouv_palettes$pal_gouv_f[5],
           gouvdown::gouv_palettes$pal_gouv_f[4],
           gouvdown::gouv_palettes$pal_gouv_f[3],
           gouvdown::gouv_palettes$pal_gouv_f[2],
           gouvdown::gouv_palettes$pal_gouv_f[1],
           "dark grey")

  fond_carte <- mapfactory::fond_carto(nom_reg = COGiter::nom_zone("R\u00e9gions",code_region))

  p <- ggplot2::ggplot() +
    ggplot2::geom_sf(data = fond_carte$epci, fill = "light grey", color = "white", size = .1) +
    ggplot2::geom_sf(data = fond_carte$regions %>% filter(.data$REG == code_region), fill = "grey", color = "white",size = .2) +
    ggplot2::geom_sf(data=data_prep %>% filter(!.data$epci_reg),
            fill = "light grey",
            color="white",size=.1)+
    # ggplot2::geom_sf(data = fond_carte$reg_ombre, fill = "dark grey", color = "light grey") +
    ggplot2::geom_sf(data=data_prep %>% filter(.data$epci_reg),
                     ggplot2::aes(fill=.data$TauxEvolution12Moisc,
                color=""),
            color="white",size=.1)+
    ggplot2::scale_fill_manual(values=color)+
    ggplot2::geom_sf(data = fond_carte$departements,fill=NA,color = "dark grey",size = .2) +
    ggplot2::coord_sf(
      # xlim = c(x_min, x_max),
      # ylim = c(y_min, y_max),
      expand = FALSE,
      crs = sf::st_crs(fond_carte$epci),
      datum = NA
    ) +
    gouvdown::theme_gouv_map(base_size=12,
                             strip_text_size = 12,
                             plot_title_size = 20,
                             subtitle_size = 16) +
    ggplot2::theme(panel.background = ggplot2::element_rect(fill = "light blue",color="light blue"),
          plot.background=ggplot2::element_rect(fill="#ffffff",color="#ffffff"),
          legend.position = "bottom",
          legend.title = ggplot2::element_blank(),
          strip.text = element_text(hjust = 0.5, face = "bold", size=14),
          strip.background = element_blank(),
          plot.caption.position = "plot") +
    ggplot2::labs(fill=titre_legende,
         caption=bas_de_page,
         title=titre)

  if (length(indic) > 1) {
    p <-p+ facet_wrap(~Indicateur2,ncol=3)+
      ggplot2::theme(panel.spacing = unit(1, "lines"))
  }
  return(p)

}

#' Création de la carte en volume en aplats sur les indicateurs de commercialisation de logements neufs
#'
#' @param data Le dataframe.
#' @param code_region Le code de la région.
#' @param indicateurs Nom de l'indicateur.
#' @param type_logement Appartements ou Maisons.
#' @param type_indicateur Cumul annuel ou Trimestriel
#' @param titre Le titre du graphique.
#' @param titre_legende Le titre de la légende.
#' @param bas_de_page Le bas de page du graphique.
#'
#' @return une carte en ggplot2
#' @export
#' @importFrom COGiter filtrer_cog_geo list_epci_in_reg nom_zone
#' @importFrom dplyr filter rename left_join mutate pull case_when
#' @importFrom ggplot2 ggplot geom_sf stat_sf_coordinates coord_sf scale_size theme element_rect element_blank labs facet_wrap theme
#' @importFrom ggspatial annotation_north_arrow north_arrow_fancy_orienteering annotation_scale
#' @importFrom gouvdown theme_gouv_map
#' @importFrom mapfactory fond_carto
#' @importFrom sf st_bbox st_crs
#' @importFrom forcats fct_drop fct_relevel
#' @importFrom cartography getBreaks
#' @importFrom stringr str_split_fixed
#'
#'
creer_carte_volume2<-function(data=indic_ecln,
                               code_region = params$reg,
                               indicateurs = c("Encours","Mises en vente","Ventes"),
                               type_logement="Appartements",
                               type_indicateur="Cumul annuel",
                               titre="",
                               titre_legende="",
                               bas_de_page=""){

  # color<-c(drealthemes::dreal_pal("continuous")(5),"dark grey")
  # color<-c(rev(gouvdown::gouv_palettes$pal_gouv_m[1:nlevels(data_prep$TauxEvolution12Moisc)-1]),"dark grey")
  # color<-c(rev(gouvdown::gouv_palettes$pal_gouv_m[2:5]),gouvdown::gouv_palettes$pal_gouv_q[2],"dark grey")
  color<-c("white",
           gouvdown::gouv_palettes$pal_gouv_f[5],
           gouvdown::gouv_palettes$pal_gouv_f[4],
           gouvdown::gouv_palettes$pal_gouv_f[3],
           gouvdown::gouv_palettes$pal_gouv_f[2],
           gouvdown::gouv_palettes$pal_gouv_f[1],
           "dark grey")
  # color<-c(rev(gouvdown::gouv_palettes$pal_gouv_a[1:5]),"dark grey")


  indic <- paste0(indicateurs," - ",type_logement)


  epci_geo <- COGiter::filtrer_cog_geo(reg = code_region,garder_supra = TRUE)$epci

  data_pour_carte<-data %>%
    dplyr::filter(.data$Indicateur %in% indic,
                  .data$TypeZone == "Epci",
                  .data$Periode == max(.data$Periode),
                  .data$TypeIndicateur == type_indicateur
    ) %>%
    dplyr::rename(EPCI=.data$CodeZone)

  epci_reg <- COGiter::list_epci_in_reg(code_region)

  data_prep<- epci_geo %>%
    dplyr::left_join(data_pour_carte) %>%
    dplyr::mutate(epci_reg = .data$EPCI %in% epci_reg) %>%
    dplyr::filter(!is.na(.data$Source)) %>%
    dplyr::mutate(Indicateur = forcats::fct_drop(.data$Indicateur),
                  Indicateur = forcats::fct_relevel(.data$Indicateur,"Ventes - Appartements",
                                                    "Mises en vente - Appartements","Encours - Appartements",
                                                    "Ventes - Maisons",
                                                    "Mises en vente - Maisons","Encours - Maisons"))

  data_prep2 <- data_prep %>%
    dplyr::mutate(Valeur2 = dplyr::case_when(
      Valeur == 0 ~ NA,
      TRUE ~ Valeur
    ))

  bks<-cartography::getBreaks(data_prep2 %>%
                              dplyr::filter(!is.na(.data$Valeur)) %>%
                              dplyr::filter(.data$Indicateur == paste0("Ventes - ",type_logement)) %>%
                              dplyr::pull(.data$Valeur),
                              method="fisher-jenks",
                              nclass=4)
  # bks2 <- c(bks,Inf)
  bks2 <- c(0,0.9,bks[2:5],Inf)

  data_prep<- data_prep %>%
    dplyr::mutate(Valeur=cut(Valeur,breaks=bks2,
                              # labels=c(paste("moins de ",round(bks[2:length(bks)],0)),paste("plus de",max(bks))),
                              labels=c("Aucune activit\u00e9",
                                       paste("De 1 \u00e0",round(bks[2],0)),
                                       paste("De",round(bks[2],0)+1,"\u00e0",round(bks[3],0)),
                                       paste("De",round(bks[3],0)+1,"\u00e0",round(bks[4],0)),
                                       paste("De",round(bks[4],0)+1,"\u00e0",round(bks[5],0)),
                                       paste("Plus de",max(bks))),
                              include.lowest = T,
                              ordered_result = T),
                  Valeur=forcats::fct_explicit_na(Valeur,na_level="Secret statistique"))%>%
    dplyr::mutate(Indicateur2 = .data$Indicateur %>%
                    stringr::str_split_fixed(., pattern = "-", n = 2) %>% # scinde la colonne en 2
                    .[, 1]) #garde la 1ere valeur

  fond_carte <- mapfactory::fond_carto(nom_reg = COGiter::nom_zone("R\u00e9gions",code_region))

  p<-ggplot2::ggplot() +
    ggplot2::geom_sf(data = fond_carte$epci, fill = "light grey", color = "white", size = .1) +
    ggplot2::geom_sf(data = fond_carte$regions %>% filter(.data$REG == code_region), fill = "grey", color = "white",size = .2) +
    ggplot2::geom_sf(data=data_prep %>% filter(!.data$epci_reg),
                     fill = "light grey",
                     color="white",size=.1)+
    # ggplot2::geom_sf(data = fond_carte$reg_ombre, fill = "dark grey", color = "light grey") +
    ggplot2::geom_sf(data=data_prep,
                     ggplot2::aes(fill=.data$Valeur,
                                  color=""),
                     color="white",size=.1)+
    ggplot2::scale_fill_manual(values=color)+
    ggplot2::geom_sf(data = fond_carte$departements,fill=NA,color = "dark grey",size = .2) +
    ggplot2::coord_sf(
      expand = FALSE,
      crs = sf::st_crs(fond_carte$epci),
      datum = NA
    ) +
    gouvdown::theme_gouv_map(base_size=12,
                             strip_text_size = 12,
                             plot_title_size = 20,
                             subtitle_size = 16) +
    ggplot2::theme(panel.background = ggplot2::element_rect(fill = "light blue",color="light blue"),
                   plot.background = ggplot2::element_rect(fill="#ffffff",color="#ffffff"),
                   legend.position = "bottom",
                   legend.title = ggplot2::element_blank(),
                   strip.text = element_text(hjust = 0.5, face = "bold", size=14),
                   strip.background = element_blank(),
                   plot.caption.position = "plot")+
    ggplot2::labs(fill=titre_legende,
         caption = bas_de_page,
         title=titre)

  if (length(indic) > 1) {
    p <-p+ ggplot2::facet_wrap(~Indicateur2,ncol=3)+
      ggplot2::theme(panel.spacing = unit(1, "lines"))
  }
  return(p)
}

