library(RcppRoll)
#
# # Calcul données annuelles----
#
# ecln_annuel <- Indic_final %>%
#   filter(!grepl("Encours", Indicateur)) %>%  #supprime les "en cours"
#   arrange(TypeZone,CodeZone,Indicateur,Periode) %>%
#   group_by(TypeZone,Zone,CodeZone,Indicateur) %>%
#   mutate(Valeur2 = roll_sumr(Valeur,4)) %>%  # additionne les 4 derniers trimestres. NA si au moins un trimestre = NA
#   ungroup() %>%
#   select(-Valeur) %>%
#   rename (Valeur = Valeur2) %>%
#   mutate(Indicateur = paste0("cumul 12 mois - ",Indicateur)) %>%
#   mutate_if(is.character,as.factor)
#
# ecln_avec_annuel <- bind_rows(Indic_final,ecln_annuel)


# 5. Calcul des indicateurs annuels glissants
# -----------------------------------------------------------------


# Attention les encours sont un stock, ils ne sont donc pas sommable sur une année

indicateurs_non_sommables <- str_subset(levels(Indic_final$Indicateur), "Encours")

# On ajoute une colonne valeur_cumul qui correspond au cumul annuel glissant sauf pour les encours = valeur du trimestre

Indic_final2<- Indic_final %>%
  arrange(TypeZone,CodeZone,Indicateur,Periode) %>%
  # lazy_dt() %>%
  group_by(TypeZone, Zone, CodeZone, Indicateur) %>%
  mutate(Valeur_cumul = Valeur + lag(Valeur) + lag(Valeur, 2) + lag(Valeur, 3)) %>%
  # mutate(Valeur_cumul = roll_sumr(Valeur,4)) %>%  # additionne les 4 derniers trimestres. NA si au moins un trimestre = NA
  ungroup() %>%
  mutate(Valeur_cumul = ifelse(Indicateur %in% indicateurs_non_sommables, Valeur, Valeur_cumul)) %>%
  as_tibble()


# 5. Calcul des valeurs trimestrielles et cumul annuel
# ----------------------------------------------------
# L'objectif est de calculer :
# - des indicateurs annuels glissants et de différencier les indicateurs trimestriels / annuels
# - des taux d'évolution sur 12 mois pour chauqe indicateur
# - d'autres indicateurs:
#     PrixM2_Maisons,PrixM2_Appartements,PrixM2_Logements,PrixPar_Maisons,PrixPar_Appartements,PrixPar_Logements

# Dans la table finale, on aura la source, le type de l'indicateur (trimestriel, annuel)

# Appel de la fonction FormatDate
source("R/FormatDate.R")

indic_ecln_trim <- Indic_final2 %>%
  select(-Valeur_cumul) %>%
  mutate(Indicateur = str_replace_all(Indicateur, " - ", "_")) %>%
  spread(Indicateur, Valeur) %>%
  mutate(
    PrixM2_Maisons = Prix_Maisons / Surface_Maisons,
    PrixM2_Appartements = Prix_Appartements / Surface_Appartements,
    PrixM2_Logements = Prix_Logements / Surface_Logements,
    PrixPar_Maisons = Prix_Maisons / Ventes_Maisons,
    PrixPar_Appartements = Prix_Appartements / Ventes_Appartements,
    PrixPar_Logements = Prix_Logements / Ventes_Logements
  ) %>%
  gather(Indicateur, Valeur, Encours_Appartements:PrixPar_Logements) %>%
  mutate(
    Indicateur = str_replace_all(Indicateur, "_", " - "),
    Periode = FormatDate(Periode, Periodicite = "Trim", sep = "T"),
    Source = "ECLN",
    TypeIndicateur = "Trimestriel"
  )

indic_ecln_cumul <- Indic_final2 %>%
  select(-Valeur) %>%
  mutate(Indicateur = str_replace_all(Indicateur, " - ", "_")) %>%
  spread(Indicateur, Valeur_cumul) %>%
  mutate(
    PrixM2_Maisons = Prix_Maisons / Surface_Maisons,
    PrixM2_Appartements = Prix_Appartements / Surface_Appartements,
    PrixM2_Logements = Prix_Logements / Surface_Logements,
    PrixPar_Maisons = Prix_Maisons / Ventes_Maisons,
    PrixPar_Appartements = Prix_Appartements / Ventes_Appartements,
    PrixPar_Logements = Prix_Logements / Ventes_Logements
  ) %>%
  gather(Indicateur, Valeur, Encours_Appartements:PrixPar_Logements) %>%
  mutate(
    Indicateur = str_replace_all(Indicateur, "_", " - "),
    Periode = FormatDate(Periode, Periodicite = "Trim", sep = "T"),
    Source = "ECLN",
    TypeIndicateur = "Cumul annuel"
  )
# %>%
#  filter(Periode > "2011-01-01")

# Aggrétation finale -----

TauxEvolution12Mois <- function(.data) {
  .data %>%
    arrange(Source, TypeZone, Zone, CodeZone, TypeIndicateur, Indicateur,Periode) %>%
    select(Source, TypeZone, Zone, CodeZone, TypeIndicateur, Indicateur, Periode, Valeur) %>%
    group_by(Source, TypeZone, Zone, CodeZone, TypeIndicateur, Indicateur) %>%
    mutate(TauxEvolution12Mois = round(100 * Valeur / lag(Valeur, 4) - 100, 1)) %>%
    ungroup()
}
indic_ecln_cumul <- TauxEvolution12Mois(indic_ecln_cumul)
indic_ecln_trim <- TauxEvolution12Mois(indic_ecln_trim)

indic_ecln <- bind_rows(indic_ecln_cumul, indic_ecln_trim) %>%
  # refaire les facteurs en enlevant les communes
  mutate_if(is.factor, as.character) %>%   #enleve les factor
  arrange(TypeZone, Zone, TypeIndicateur, Indicateur, Periode) %>%
  mutate_if(is.character, as.factor) %>%
  mutate(TypeZone = fct_relevel(TypeZone, "France", "Régions", "Départements", "Epci"))

rm(indicateurs_non_sommables,Indic_final2,Indic_final,indic_ecln_trim,indic_ecln_cumul,
   TauxEvolution12Mois,FormatDate,FormatCaractere)

usethis::use_data(indic_ecln, overwrite = TRUE)
