#' graphique sur l'évolution du prix de vente des biens.
#'
#' @param data Le dataframe.
#' @param type_logement Appartements ou Maisons.
#' @param type_indicateur Cumul annuel ou valeur trimestrielle.
#' @param type_zone Liste des types de zonage à intégrer dans la facette.
#' @param titre Le titre du graphique.
#' @param bas_de_page Le bas de page du graphique.
#' @param type_facet Type de facette : classique ou grille pour l'utilisation d'une matrice.
#' @param ncol_facet Le nombre de colonnes dans la facette.
#'
#' @return Un graphique ggplot2.
#' @export
#' @importFrom dplyr filter mutate
#' @importFrom forcats fct_drop
#' @importFrom ggplot2 ggplot aes geom_line geom_point theme scale_x_date scale_y_continuous labs facet_wrap element_line
#' @importFrom scales dollar_format
#' @importFrom stringr str_split_fixed
#' @importFrom gouvdown scale_color_gouv_discrete
creer_graphique_series_temporelles_prix <- function(data = indic_ecln,
                                                    type_logement = "Logements",
                                                    type_zone = c("R\u00e9gions", "D\u00e9partements"),
                                                    type_indicateur = "Cumul annuel",
                                                    titre = "",
                                                    bas_de_page = "",
                                                    type_facet = "classique",
                                                    ncol_facet = 3) {
  indic <- c("PrixM2 - Logements")
  if (type_logement == "Maisons") {
    indic <- c("PrixM2 - Maisons")
  }
  if (type_logement == "Appartements") {
    indic <- c("PrixM2 - Appartements")
  }
  data_prep <- data %>%
    dplyr::filter(
      .data$Indicateur %in% indic,
      .data$TypeIndicateur == type_indicateur,
      .data$TypeZone %in% type_zone
    ) %>%
    dplyr::mutate(
      Zone = forcats::fct_drop(.data$Zone),
      Indicateur = stringr::str_split_fixed(.data$Indicateur, " - ", 2)[, 1]
    )


  p <- ggplot2::ggplot(
    data_prep,
    ggplot2::aes(
      x = .data$Periode,
      y = .data$Valeur,
      color = .data$Indicateur,
      fill = .data$Indicateur,
      group = .data$Indicateur
    )
  ) +
    ggplot2::geom_line(size = 1.2) +
    # ggplot2::geom_point(data = data %>%
    #   dplyr::filter(
    #     .data$Indicateur %in% indic,
    #     .data$TypeIndicateur == type_indicateur,
    #     .data$TypeZone %in% type_zone,
    #     .data$Periode == max(.data$Periode)
    #   )) +
    ggplot2::theme(legend.position = "none",
                   legend.title = element_blank(),
                   panel.grid.major = element_line(color = "lightgrey", linetype = "solid"),
                   panel.grid.minor = element_blank(),
                   strip.text = element_text(hjust = 0.5, face = "bold", size=14),
                   strip.background = element_blank(),
                   axis.text.x = element_text(size=11, hjust=-0.1),
                   axis.line = element_line(size = 0.5, linetype = "solid")) +
    ggplot2::scale_x_date(date_labels = "%y", expand = c(0, 0), date_breaks = "1 year") +
    ggplot2::scale_y_continuous(labels = scales::dollar_format(big.mark = " ", decimal_mark = ",", prefix = "", suffix = " \u20ac")) +
    gouvdown::scale_color_gouv_discrete(palette = "pal_gouv_fr") +
    ggplot2::labs(
      title = titre,
      x = "",
      y = "",
      fill = "",
      color = "",
      caption = bas_de_page
    )
  if (nlevels(data_prep$Zone) > 1) {
    if (type_facet == "classique") {
      p <- p +
        ggplot2::facet_wrap(~ .data$Zone, scales = "free", ncol = ncol_facet)
    }

    if (type_facet == "grille") {
      p <- p +
        geofacet::facet_geo(~Zone, grid = mygrid, scales = "free", drop = TRUE)
    }
  }
  return(p)
}

