
# librairies -------------
library(dplyr)
library(stringr)
library(COGiter)
library(lubridate)
library(datalibaba)

rm(list = ls())


# téléchargement des donnees Teruti -----------------------------
# issues du site  https://agreste.agriculture.gouv.fr/agreste-web/disaron/W0020/detail/
# L'indicateur "Surface" est en hectares.
# L'indicateur "Précision" est le coefficient de variation (CV) en %. Plus le CV est faible, plus l'estimation est précise.
# L'indicateur "Demi-intervalle de confiance" est la valeur en hectares permettant de calculer la borne inférieure et la borne supérieure de l'intervalle de confiance. Cet intervalle a 95 % de chances de contenir la "vraie" surface.
# L'Année de référence est l'année médiane de 3 années successives.

teruti <- read.csv2("extdata/FDS_W0020_Moy-2017-2018-2019.csv", as.is = TRUE, encoding = "UTF-8") %>%
  select(ANNREF, GEOGRAPHIE, N027_LIB, N613_LIB, VALEUR) %>%
  mutate(
    TypeZone = case_when(
      grepl("^NR", GEOGRAPHIE, perl = TRUE) ~ "Régions",
      GEOGRAPHIE == "DOM" ~ "France",
      GEOGRAPHIE == "METRO" ~ "France",
      GEOGRAPHIE == "FR" ~ "France",
      TRUE ~ "Départements"
      ),
    GEOGRAPHIE = case_when(
      GEOGRAPHIE == "DOM" ~ "DROM",
      GEOGRAPHIE == "METRO" ~ "FRMETRO",
      GEOGRAPHIE == "FR" ~ "FRMETRODROM",
      TRUE ~ GEOGRAPHIE
      ),
    ) %>%
  mutate(
    GEOGRAPHIE = str_replace(GEOGRAPHIE, "NR", ""),
    date = make_date(ANNREF, "01", "01")
    ) %>%
  filter(N027_LIB == "Surface") %>%
  rename(
    CodeZone = GEOGRAPHIE,
    variable = N613_LIB,
    valeur = VALEUR
    ) %>%
  left_join(COGiter::liste_zone %>% select(TypeZone, Zone, CodeZone)) %>%
  mutate(Zone = case_when(
    CodeZone == "DROM" ~ "Départements et régions d'outre-mer",
    CodeZone == "FRMETRO" ~ "France métropolitaine",
    CodeZone == "FRMETRODROM" ~ "France métropolitaine et DROM",
    TRUE ~ as.character(Zone)
    ) %>%
      as.factor()
    ) %>%
  select(TypeZone, CodeZone, Zone, date, variable, valeur)


# téléchargement des tables sur le SGBD -------------
observatoire_artificialisation <- importer_data(db = "datamart",    #surface en m2
                                                schema = "portrait_territoires",
                                                table = "cogifiee_observatoire_artificialisation")

etalement_urbain <- importer_data(db = "datamart",
                                  schema = "portrait_territoires",
                                  table = "indicateur_ff_etalement_urbain")

population_legale <- importer_data(db = "datamart",
                                   schema = "portrait_territoires",
                                   table = "cogifiee_population_legale")

ocsge <- importer_data(db = "datamart",    #surface en m2
                       schema = "portrait_territoires",
                       table = "cogifiee_ocsge")

stock_artificialise <- importer_data(db = "datamart",
                       schema = "portrait_territoires",
                       table = "cogifiee_fichier_foncier_artificialisation") %>%
  rename(surface_artificialisee = surf_artificialisee)


# metadonnées (à revoir) -------------
metadata_donnee <- read.csv2("extdata/indicateurs.csv", as.is = TRUE, encoding = "UTF-8") %>%
  mutate(source = str_replace(source, "chargement_", "") %>%
           str_replace("indicateur_", "") %>%
           str_replace("_cogiter", ""))


# formatage des tables du sgbd  ---------
select_donnees_region <- function(indicateur = NULL){
 selection <- indicateur %>%
   mutate(across(where(is.character),as.factor))
}

ocsge <- select_donnees_region (indicateur = ocsge)


# creation data des millesimes disponibles ---------
millesimes_disponibles <- tibble::tibble(donnee = c("teruti","observatoire_artificialisation","etalement_urbain","population_legale","ocsge","stock_artificialise"),
                                  date = c(max(teruti$date),max(observatoire_artificialisation$date),max(etalement_urbain$date),max(population_legale$date),max(ocsge$date),max(stock_artificialise$date)))


# production des data/rda ------------
usethis::use_data(teruti, overwrite = TRUE, internal = FALSE)
usethis::use_data(observatoire_artificialisation, overwrite = TRUE, internal = FALSE)
usethis::use_data(etalement_urbain, overwrite = TRUE, internal = FALSE)
usethis::use_data(population_legale, overwrite = TRUE, internal = FALSE)
usethis::use_data(ocsge, overwrite = TRUE, internal = FALSE)
usethis::use_data(stock_artificialise, overwrite = TRUE, internal = FALSE)
usethis::use_data(metadata_donnee, overwrite = TRUE, internal = FALSE)
usethis::use_data(millesimes_disponibles, overwrite = TRUE, internal = FALSE)
