#' Creation du tableau de synthese.
#' @description Tableau de synthese.
#'
#' @param millesime_stock_artif une annee parmi les millesimes selectionnables par l'utilisateur, au format numerique.
#' @param millesime_population une annee parmi les millésimes selectionnables par l'utilisateur, au format numerique.
#' @param code_reg code insee de la region sur laquelle construire le graphique
#' @param police police des titres et sous titre
#' @return Un tableau
#'
#' @importFrom attempt stop_if stop_if_not
#' @importFrom COGiter charger_zonage filtrer_cog ajouter_typologie
#' @importFrom dplyr mutate filter rename select left_join arrange pull
#' @importFrom glue glue
#' @importFrom kableExtra kable_styling row_spec pack_rows footnote
#' @importFrom knitr kable
#' @importFrom lubridate year
#' @importFrom stringr str_detect str_split_fixed
#'
#'
#' @export
#' @examples
#' creer_tableau_6_1(millesime_stock_artif = 2020, millesime_population = 2019, code_reg = '52')


creer_tableau_6_1 <- function(millesime_stock_artif = NULL, millesime_population = NULL, code_reg = NULL, police = "sans") {

  attempt::stop_if(millesime_stock_artif, is.null, msg = "millesime_stock_artif n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_stock_artif, is.numeric, msg = "millesime_stock_artif n\'est pas un nombre")
  attempt::stop_if(millesime_population, is.null, msg = "millesime_population n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_population, is.numeric, msg = "millesime_population n\'est pas un nombre")
  attempt::stop_if(code_reg, is.null, msg = "code_reg n\'est pas renseign\u00e9")
  attempt::stop_if_not(code_reg, ~ .x %in% levels(COGiter::regions$REG), msg = "code_reg n\'est pas un code r\u00e9gion valide")

  millesime_debut <- 2016
  millesime_population_debut <- millesime_population - (millesime_stock_artif - millesime_debut)
  if (is.numeric(code_reg)) {
    code_reg = as.character(code_reg)
  }

  # seuils population
  pop_3 <- 40000
  pop_2 <- 10000
  pop_1 <- 2000

  # creation de la fonction inverse
  `%notin%` <- Negate(`%in%`)

  if (code_reg %in% c('52')) {

  # population du millesime
  population <- population_legale %>%
    dplyr::mutate(date=lubridate::year(.data$date)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$date == millesime_population,
                  .data$TypeZone =="Communes") %>%
    dplyr::mutate(seuil_pop = dplyr::case_when(
      .data$population_municipale > pop_3  ~ glue::glue("communes de plus\nde ",pop_3," habitants"),
      .data$population_municipale > pop_2  ~ glue::glue("communes entre\n",pop_2," et ",pop_3, " habitants"),
      .data$population_municipale > pop_1  ~ glue::glue("communes entre\n",pop_1," et ",pop_2, " habitants"),
      .data$population_municipale > 0  ~ glue::glue("communes de moins\nde ",pop_1," habitants"),
      TRUE ~ "")
    ) %>%
    dplyr::mutate(seuil_code = dplyr::case_when(
      .data$population_municipale > pop_3  ~ "D",
      .data$population_municipale > pop_2  ~ "C",
      .data$population_municipale > pop_1  ~ "B",
      .data$population_municipale > 0  ~ "A",
      TRUE ~ "")
    ) %>%
    dplyr::rename ("population_n"="population_municipale") %>%
    dplyr::select (-.data$date)

  # table des seuils
  seuil_population <- population %>%
    dplyr::select (.data$CodeZone, .data$seuil_pop)


  # preparation des donnees evolution_artificialisation

  evol_artif <- stock_artificialise %>%
    dplyr::mutate(date = lubridate::year(.data$date)
                  # surface_artificialisee = surface_artificialisee / 10000, # surface déjà convertie en ha !
                  # surf_cadastree = surf_cadastree / 10000
                  ) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$TypeZone == "Communes",
                  .data$date == millesime_stock_artif | .data$date == millesime_debut)

  evol_artif1 <- evol_artif %>%
    dplyr::left_join(seuil_population) %>%
    dplyr::mutate(dep = substr(.data$CodeZone, 1, 2)) %>%
    dplyr::filter (.data$date == millesime_stock_artif) %>%
    dplyr::group_by(.data$dep, .data$seuil_pop, .data$date) %>%
    dplyr::summarise(surface_artificialisee = sum(.data$surface_artificialisee, na.rm = T),
                     surf_cadastree = sum(.data$surf_cadastree, na.rm = T)
                     ) %>%
    dplyr::ungroup() %>%
    dplyr::left_join(COGiter::departements %>%
                       dplyr::filter (.data$REG == code_reg) %>%
                       dplyr::select(.data$DEP, .data$NOM_DEP),
                     by=c("dep"= "DEP")) %>%
    dplyr::rename("Zone" = "NOM_DEP") %>%
    dplyr::select(-.data$dep)

  evol_artif1_2 <- evol_artif1 %>%
    dplyr::group_by(.data$seuil_pop, .data$date) %>%
    dplyr::summarise(surface_artificialisee = sum(.data$surface_artificialisee, na.rm = T),
                     surf_cadastree = sum(.data$surf_cadastree, na.rm = T)
                     ) %>%
    dplyr::ungroup() %>%
    dplyr::mutate(Zone = COGiter::regions %>%
                    dplyr::filter (.data$REG == code_reg) %>%
                    dplyr::pull(.data$NOM_REG))


  evol_artif1 <-dplyr::bind_rows(evol_artif1,evol_artif1_2) %>%
    dplyr::select(.data$seuil_pop, .data$Zone, .data$date, .data$surface_artificialisee, .data$surf_cadastree) %>%
    dplyr::mutate (part_artificialisee_du_territoire_cadastre = paste0( round((.data$surface_artificialisee / .data$surf_cadastree)*100 , 1), " %"))

  evol_artif2 <- evol_artif %>%
    dplyr::left_join(seuil_population) %>%
    dplyr::mutate(dep = substr(.data$CodeZone, 1, 2)) %>%
    dplyr::filter (.data$date == millesime_debut) %>%
    dplyr::group_by(.data$dep, .data$seuil_pop, .data$date) %>%
    dplyr::summarise(surface_artificialisee_ancienne = sum(.data$surface_artificialisee, na.rm = T)) %>%
    dplyr::ungroup() %>%
    dplyr::left_join(COGiter::departements %>%
                       dplyr::filter (.data$REG == code_reg) %>%
                       dplyr::select(.data$DEP, .data$NOM_DEP),
                     by=c("dep"= "DEP")) %>%
    dplyr::rename("Zone" = "NOM_DEP") %>%
    dplyr::select(-.data$dep)

  evol_artif2_2 <- evol_artif2 %>%
    dplyr::group_by(.data$seuil_pop, .data$date) %>%
    dplyr::summarise(surface_artificialisee_ancienne = sum(.data$surface_artificialisee_ancienne, na.rm = T)) %>%
    dplyr::ungroup() %>%
    dplyr::mutate(Zone = COGiter::regions %>%
                    dplyr::filter (.data$REG == code_reg) %>%
                    dplyr::pull(.data$NOM_REG))

  evol_artif2 <-dplyr::bind_rows(evol_artif2,evol_artif2_2) %>%
    dplyr::select(.data$seuil_pop, .data$Zone, .data$surface_artificialisee_ancienne)


  # preparation des donnees evol_population

  evol_popul <- population_legale %>%
    dplyr::mutate(date = lubridate::year(.data$date)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$TypeZone == "Communes",
                  .data$date == millesime_population | .data$date == millesime_population_debut)

  evol_popul1 <- evol_popul %>%
    dplyr::left_join(seuil_population) %>%
    dplyr::mutate(dep = substr(.data$CodeZone, 1, 2)) %>%
    dplyr::filter (.data$date == millesime_population) %>%
    dplyr::group_by(.data$dep, .data$seuil_pop, .data$date) %>%
    dplyr::summarise(population_municipale = sum(.data$population_municipale, na.rm = T)) %>%
    dplyr::ungroup() %>%
    dplyr::left_join(COGiter::departements %>%
                       dplyr::filter (.data$REG == code_reg) %>%
                       dplyr::select(.data$DEP, .data$NOM_DEP),
                     by=c("dep"= "DEP")) %>%
    dplyr::rename("Zone" = "NOM_DEP") %>%
    dplyr::select(-.data$dep)

  evol_popul1_2 <- evol_popul1 %>%
    dplyr::group_by(.data$seuil_pop, .data$date) %>%
    dplyr::summarise(population_municipale = sum(.data$population_municipale, na.rm = T)) %>%
    dplyr::ungroup() %>%
    dplyr::mutate(Zone = COGiter::regions %>%
                    dplyr::filter (.data$REG == code_reg) %>%
                    dplyr::pull(.data$NOM_REG))

  evol_popul1 <-dplyr::bind_rows(evol_popul1, evol_popul1_2) %>%
    dplyr::select(.data$seuil_pop, .data$Zone, .data$population_municipale)


  evol_popul2 <- evol_popul %>%
    dplyr::left_join(seuil_population) %>%
    dplyr::mutate(dep = substr(.data$CodeZone,1,2)) %>%
    dplyr::filter (.data$date == millesime_population_debut) %>%
    dplyr::group_by(.data$dep, .data$seuil_pop, .data$date) %>%
    dplyr::summarise(population_municipale_ancienne = sum(.data$population_municipale, na.rm = T)) %>%
    dplyr::ungroup() %>%
    dplyr::left_join(COGiter::departements %>%
                       dplyr::filter (.data$REG == code_reg) %>%
                       dplyr::select(.data$DEP, .data$NOM_DEP),
                     by=c("dep"= "DEP")) %>%
    dplyr::rename("Zone" = "NOM_DEP") %>%
    dplyr::select(-.data$dep)

  evol_popul2_2 <- evol_popul2 %>%
    dplyr::group_by(.data$seuil_pop, .data$date) %>%
    dplyr::summarise(population_municipale_ancienne = sum(.data$population_municipale_ancienne, na.rm = T)) %>%
    dplyr::ungroup() %>%
    dplyr::mutate(Zone = COGiter::regions %>%
                    dplyr::filter (.data$REG == code_reg) %>%
                    dplyr::pull(.data$NOM_REG))

  evol_popul2 <-dplyr::bind_rows(evol_popul2, evol_popul2_2) %>%
    dplyr::select(.data$seuil_pop, .data$Zone, .data$population_municipale_ancienne)


# regroupement des donnees

  data_regroupe <- evol_artif1 %>%
    dplyr::left_join(evol_artif2) %>%
    dplyr::left_join(evol_popul1) %>%
    dplyr::left_join(evol_popul2) %>%
    dplyr::mutate (evolution_population = .data$population_municipale - .data$population_municipale_ancienne,
                   evolution_surf_artificialisee = .data$surface_artificialisee - .data$surface_artificialisee_ancienne,
                   tc_surfaces_artificialisees = paste0( round(((.data$surface_artificialisee / .data$surface_artificialisee_ancienne)-1 )*100,1)," %"),
                   tc_population = paste0( round(((.data$population_municipale / .data$population_municipale_ancienne)-1 )*100,1)," %"),
                   surf_artif_par_an = .data$evolution_surf_artificialisee / ( millesime_stock_artif - millesime_debut),
                   nb_annees_pour_doubler = ceiling(.data$surface_artificialisee_ancienne / .data$surf_artif_par_an),
                   surf_artif_par_hab_an_x= round(.data$surface_artificialisee_ancienne / .data$population_municipale_ancienne *10000 , 1) ,
                   surf_artif_par_nouv_hab_entre_x_y =round(.data$evolution_surf_artificialisee / .data$evolution_population *10000 , 1)) %>%
    dplyr::mutate(seuil_pop = factor(.data$seuil_pop, levels = c(glue::glue("communes de moins\nde ",pop_1," habitants"),
                                                           glue::glue("communes entre\n",pop_1," et ",pop_2, " habitants"),
                                                           glue::glue("communes entre\n",pop_2," et ",pop_3, " habitants"),
                                                           glue::glue("communes de plus\nde ",pop_3," habitants")))) %>%
    dplyr::arrange(.data$seuil_pop) %>%
    dplyr::select (.data$seuil_pop,
                   .data$Zone,
                   .data$surf_cadastree,
                   .data$surface_artificialisee,
                   .data$population_municipale,
                   .data$evolution_surf_artificialisee,
                   .data$part_artificialisee_du_territoire_cadastre,
                   .data$tc_surfaces_artificialisees,
                   .data$tc_population,
                   .data$nb_annees_pour_doubler,
                   .data$surf_artif_par_hab_an_x,
                   .data$surf_artif_par_nouv_hab_entre_x_y)

# separation des tableaux

  tab1 <- data_regroupe %>%
    dplyr::filter(.data$seuil_pop == glue::glue("communes de moins\nde ",pop_1," habitants"))

  tab2 <- data_regroupe %>%
    dplyr::filter(.data$seuil_pop == glue::glue("communes entre\n",pop_1," et ",pop_2, " habitants"))

  tab3 <- data_regroupe %>%
    dplyr::filter(.data$seuil_pop == glue::glue("communes entre\n",pop_2," et ",pop_3, " habitants"))

  tab4 <- data_regroupe %>%
    dplyr::filter(.data$seuil_pop == glue::glue("communes de plus\nde ",pop_3," habitants"))

  nb_lign1 <- nrow(tab1)
  nb_lign2 <- nrow(tab2)
  nb_lign3 <- nrow(tab3)
  nb_lign4 <- nrow(tab4)
  n = nb_lign1 + nb_lign2 + nb_lign3 + nb_lign4


tableau_6_1 <- data_regroupe %>%
  dplyr::select(-.data$seuil_pop) %>%
  knitr::kable("html", caption = "Tableau de synth\u00e8se",
               digits= c(0,0,0,0,0,1,1,1,0,1,1),
               align = "c",
               col.names = c("Territoire",
                             glue::glue("Surface totale cadastr\u00e9e au 1/1/",millesime_stock_artif," (ha)"),
                             glue::glue("Surface artificialis\u00e9e au 1/1/",millesime_stock_artif," (ha)"),
                             glue::glue("Population municipale ",millesime_population),
                             glue::glue("Evolution des surface artificialis\u00e9es entre le 1/1/",millesime_debut," et le 1/1/",millesime_stock_artif),
                             glue::glue("Part artificialis\u00e9e du territoire cadastr\u00e9 au 1/1/",millesime_stock_artif),
                             glue::glue("Taux de croissance des surfaces artificialis\u00e9es entre ",millesime_debut," et ",millesime_stock_artif),
                             glue::glue("Taux de croissance de la population entre ",millesime_population_debut , " et ",millesime_population),
                             glue::glue("Nombre d\'ann\u00e9es n\u00e9cessaires pour doubler la surface artificialis\u00e9e ", millesime_debut),
                             glue::glue("Surface artificialis\u00e9e par habitant en ",millesime_population_debut," (m2)" ),
                             glue::glue("Surface artificialis\u00e9e pour un nouvel habitant suppl\u00e9mentaire\n entre ",millesime_debut," et ",millesime_stock_artif," (m\u00b2)")
                                        ),
               row.names = FALSE,
               format.args=list(big.mark=" ",decimal.mark=","))%>%
  kableExtra::kable_styling(bootstrap_options = c("condensed"), font_size = 10, full_width = F, position = "float_left") %>%
  kableExtra::row_spec(0, bold = T, background = "#f0f0f5") %>%

  kableExtra::pack_rows(glue::glue("\n \n communes de moins de ",pop_1," habitants"), 1, nb_lign1) %>%
  kableExtra::row_spec(nb_lign1, bold = T, background = "#f0f0f5") %>%

  kableExtra::pack_rows(glue::glue("\n \n communes entre ",pop_1," et ",pop_2, " habitants"), nb_lign1+1,nb_lign1+nb_lign2) %>%
  kableExtra::row_spec(nb_lign1+nb_lign2, bold = T, background = "#f0f0f5") %>%

  kableExtra::pack_rows(glue::glue("\n \n communes entre ",pop_2," et ",pop_3, " habitants"), nb_lign1+nb_lign2+1,nb_lign1+nb_lign2+nb_lign3) %>%
  kableExtra::row_spec(nb_lign1+nb_lign2+nb_lign3, bold = T, background = "#f0f0f5") %>%

  kableExtra::pack_rows(glue::glue("\n \n communes de plus de ",pop_3," habitants"), nb_lign1+nb_lign2+nb_lign3+1,nb_lign1+nb_lign2+nb_lign3+nb_lign4) %>%
  kableExtra::row_spec(nb_lign1+nb_lign2+nb_lign3+nb_lign4, bold = T, background = "#f0f0f5") %>%

  kableExtra::row_spec( 1:n, extra_css = 'vertical-align: middle ;') %>%
  kableExtra::footnote(general_title = glue::glue("Source : Cerema Fichiers fonciers / Insee RP {millesime_population}"),
                       general ="")

return(tableau_6_1)
}
if (code_reg  %notin% c('52')) {
  return(invisible(NULL))
}

}
