#' Création du graphique de simulation des objectifs de la loi 3C sur le rythme d'artificialisation des sols
#'
#' @param millesime_obs_artif une annee parmi les millesimes selectionnables par l'utilisateur, au format numerique.
#' @param code_reg code de la région sélectionnable par l'utilisateur.
#'
#' @return une liste de deux graphiques ggplot
#' @importFrom COGiter filtrer_cog
#' @importFrom dplyr filter select arrange mutate group_by summarise ungroup row_number lead bind_rows
#' @importFrom ggplot2 ggplot geom_line aes labs scale_y_continuous coord_cartesian theme
#' @importFrom ggtext geom_textbox
#' @importFrom glue glue
#' @importFrom gouvdown scale_color_gouv_discrete scale_fill_gouv_discrete
#' @importFrom lubridate ymd year years
#' @importFrom mapfactory format_fr_nb
#' @importFrom scales label_number
#' @importFrom tibble tibble
#' @importFrom tidyr crossing
#' @export
#'
#' @examples
#' creer_graphe_5_2(2019,'52')
creer_graphe_5_2 <- function(millesime_obs_artif,code_reg) {
  if (is.numeric(code_reg)) {
    code_reg = as.character(code_reg)
  }

  data <- observatoire_artificialisation %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$TypeZone %in% c("R\u00e9gions","D\u00e9partements")) %>%
    dplyr::select(.data$TypeZone,.data$Zone,.data$date,.data$flux_naf_artificialisation_total) %>%
    dplyr::arrange(.data$TypeZone,.data$Zone,.data$date) %>%
    dplyr::mutate(simul=FALSE,
                  flux_naf_artificialisation_total = .data$flux_naf_artificialisation_total / 10000)

  datemax <- lubridate::ymd(paste0(millesime_obs_artif,'0101'))
  datemin <- min(data$date)
  pas <- millesime_obs_artif - lubridate::year(datemin)

  simul <- data %>%
    dplyr::filter(.data$date %in% c(datemin,datemax)) %>%
    dplyr::group_by(.data$TypeZone,.data$Zone) %>%
    dplyr::summarise(flux_naf_artificialisation_total_tendance = mean(.data$flux_naf_artificialisation_total),
                     flux_naf_artificialisation_total_objectif = mean(.5 * .data$flux_naf_artificialisation_total)) %>%
    dplyr::mutate(simul=TRUE)

  seq_prochaines_annees <- seq(datemax + lubridate::years(1),datemax + lubridate::years(11), by = '1 year')


  data_prochaines_annees <- tibble::tibble(date = seq_prochaines_annees) %>%
    tidyr::crossing(simul)

  data <- data %>%
    dplyr::mutate(flux_naf_artificialisation_total_tendance = .data$flux_naf_artificialisation_total,
           flux_naf_artificialisation_total_objectif = .data$flux_naf_artificialisation_total) %>%
    dplyr::bind_rows(data_prochaines_annees) %>%
    dplyr::arrange(.data$TypeZone,.data$Zone,.data$date)


  data <- data %>%
    dplyr::group_by(.data$TypeZone,.data$Zone) %>%
    dplyr::mutate(simul_ou_dernier_mesure = case_when(
      simul ~ TRUE,
      !simul & dplyr::lead(simul) ~ TRUE,
      TRUE ~ FALSE
    ),
    valeur = cumsum(.data$flux_naf_artificialisation_total),
    valeur_tendance = cumsum(.data$flux_naf_artificialisation_total_tendance),
    valeur_objectif = cumsum(.data$flux_naf_artificialisation_total_objectif),
    valeur_tendance = ifelse(.data$simul_ou_dernier_mesure,.data$valeur_tendance,NA),
    valeur_objectif = ifelse(.data$simul_ou_dernier_mesure,.data$valeur_objectif,NA),
    ) %>%
    dplyr::ungroup()

  data_departements <-  data%>%
    dplyr::filter(.data$TypeZone == "D\u00e9partements")
  data_region <-  data%>%
    dplyr::filter(.data$TypeZone == "R\u00e9gions")

  data_segment_departements = data_departements %>%
    dplyr::filter(.data$date ==max(date)) %>%
    dplyr::select(.data$TypeZone,.data$Zone,.data$date,.data$valeur_objectif,.data$valeur_tendance) %>%
    dplyr::mutate(y = .data$valeur_objectif+0.5*(.data$valeur_tendance-.data$valeur_objectif),
                  rank = rank(.data$y),
                  nrow = max(dplyr::row_number()),
                  range = max(.data$valeur_tendance)-min(.data$valeur_objectif),
                  position = min(.data$valeur_objectif)+.data$range*.data$rank/.data$nrow,
                  label = glue::glue("**{Zone}** :
                                   {mapfactory::format_fr_nb(valeur_tendance-valeur_objectif,0)} ha en moins \u00e0 r\u00e9aliser"))
  data_segment_region = data_region %>%
    dplyr::filter(.data$date ==max(date)) %>%
    dplyr::select(.data$TypeZone,.data$Zone,.data$date,.data$valeur_objectif,.data$valeur_tendance) %>%
    dplyr::mutate(y = .data$valeur_objectif+0.5*(.data$valeur_tendance-.data$valeur_objectif),
                  position = .data$y,
                  label = glue::glue("**{Zone}** :
                                   {mapfactory::format_fr_nb(valeur_tendance-valeur_objectif,0)} ha en moins \u00e0 r\u00e9aliser"))




  note_de_lecture_zone_departements <- as.character(data_segment_departements[1,]$Zone)
  note_de_lecture_date_departements <- lubridate::year(data_segment_departements[1,]$date)
  note_de_lecture_valeur_en_date_departements <- max(data_departements %>% dplyr::filter(.data$Zone==note_de_lecture_zone_departements) %>% pull(valeur),na.rm=TRUE)
  note_de_lecture_valeur_tendance_departements <- data_segment_departements[1,]$valeur_tendance
  note_de_lecture_valeur_objectif_departements <- data_segment_departements[1,]$valeur_objectif

  note_de_lecture_zone_region <- as.character(data_segment_region[1,]$Zone)
  note_de_lecture_date_region <- lubridate::year(data_segment_region[1,]$date)
  note_de_lecture_valeur_en_date_region <- max(data_region %>% dplyr::filter(.data$Zone==note_de_lecture_zone_region) %>% pull(valeur),na.rm=TRUE)
  note_de_lecture_valeur_tendance_region <- data_segment_region[1,]$valeur_tendance
  note_de_lecture_valeur_objectif_region <- data_segment_region[1,]$valeur_objectif

  res_departement <- ggplot2::ggplot(data_departements) +
    ggplot2::geom_line(ggplot2::aes(x=.data$date,y=.data$valeur,group = .data$Zone,color=.data$Zone)) +
    ggplot2::geom_line(ggplot2::aes(x=.data$date,y=.data$valeur_tendance,group = .data$Zone,color=.data$Zone),linetype = 2) +
    ggplot2::geom_line(ggplot2::aes(x=.data$date,y=.data$valeur_objectif,group = .data$Zone,color=.data$Zone),linetype = 3) +
    ggtext::geom_textbox(data = data_segment_departements,
                         ggplot2::aes(x = .data$date,y=.data$position,label = .data$label,fill = .data$Zone),color = "white",hjust = 0,vjust = .5) +
    ggplot2::labs(
      title = "Impact de l\u2019objectif de r\u00e9duction par 2 du rythme de consommation",
      subtitle = "entre 2020 et 2030",
      x = "", y = "surfaces artificialis\u00e9s (en ha)",
      fill = "",
      caption = glue::glue('
                         Source : Observatoire de l\'artificialisation {millesime_obs_artif}
                         Note de lecture : D\'ici {note_de_lecture_date_departements}, la {note_de_lecture_zone_departements} au rythme actuel aura artificialis\u00e9 {mapfactory::format_fr_nb(note_de_lecture_valeur_tendance_departements-note_de_lecture_valeur_en_date_departements,0)} hectares suppl\u00e9mentaires, soit un total de {mapfactory::format_fr_nb(note_de_lecture_valeur_tendance_departements,0)} hectares depuis 2010
                         Une division par deux du rythme ram\u00e8nerait ce total \u00e0 {mapfactory::format_fr_nb(note_de_lecture_valeur_objectif_departements,0)}
                         ')
    ) +
    ggplot2::scale_y_continuous(
      limits = c(0, NA),
      labels = scales::label_number(big.mark = " ", decimal.mark = ",")
    ) +
    ggplot2::scale_x_date(
      date_breaks = "5 years",
      date_labels = "%Y"
    )+
    ggplot2::coord_cartesian(clip = "off",expand = FALSE) +
    ggplot2::theme(
      legend.position = "none",
      plot.margin = unit(c(1, 9, 1, 1), "lines")
    ) +
    gouvdown::scale_color_gouv_discrete(palette = "pal_gouv_qual1") +
    gouvdown::scale_fill_gouv_discrete(palette = "pal_gouv_qual1")

  res_region <- ggplot2::ggplot(data_region) +
    ggplot2::geom_line(ggplot2::aes(x=.data$date,y=.data$valeur,group = .data$Zone,color=.data$Zone)) +
    ggplot2::geom_line(ggplot2::aes(x=.data$date,y=.data$valeur_tendance,group = .data$Zone,color=.data$Zone),linetype = 2) +
    ggplot2::geom_line(ggplot2::aes(x=.data$date,y=.data$valeur_objectif,group = .data$Zone,color=.data$Zone),linetype = 3) +
    ggtext::geom_textbox(data = data_segment_region,
                         ggplot2::aes(x = .data$date,y=.data$position,label = .data$label,fill = .data$Zone),color = "white",hjust = 0,vjust = .5) +
    ggplot2::labs(
      title = "Impact de l\u2019objectif de r\u00e9duction par 2 du rythme de consommation",
      subtitle = "entre 2020 et 2030",
      x = "", y = "surfaces cadastr\u00e9es consomm\u00e9s (en ha)",
      fill = "",
      caption = glue::glue('
                         Source : Observatoire de l\'artificialisation {millesime_obs_artif}
                         Note de lecture :
                         D\'ici {note_de_lecture_date_region}, La r\u00e9gion {note_de_lecture_zone_region} au rythme actuel aura artificialis\u00e9 {mapfactory::format_fr_nb(note_de_lecture_valeur_tendance_region-note_de_lecture_valeur_en_date_region,0)} hectares suppl\u00e9mentaires, soit un total de {mapfactory::format_fr_nb(note_de_lecture_valeur_tendance_region,0)} hectares depuis 2010
                         Une division par deux du rythme ram\u00e8nerait ce total \u00e0 {mapfactory::format_fr_nb(note_de_lecture_valeur_objectif_region,0)}
                         ')
    ) +
    ggplot2::scale_y_continuous(
      limits = c(0, NA),
      labels = scales::label_number(big.mark = " ", decimal.mark = ",")
    ) +
    ggplot2::scale_x_date(
      date_breaks = "5 years",
      date_labels = "%Y"
    )+
    ggplot2::coord_cartesian(clip = "off",expand = FALSE) +
    ggplot2::theme(
      legend.position = "none",
      plot.margin = unit(c(1, 8, 1, 1), "lines")
    ) +
    gouvdown::scale_color_gouv_discrete(palette = "pal_gouv_qual1") +
    gouvdown::scale_fill_gouv_discrete(palette = "pal_gouv_qual1")

  return(list(graphique_departement=res_departement,graphique_region=res_region))

}
