#' Creation du graphique en barres par type de territoire des surfaces artificialisees pour un nouvel habitant.
#' @description Graphique en barres par type de territoire des surfaces artificialisees pour un nouvel habitant.
#'
#' @param millesime_stock_artif une annee parmi les millesimes selectionnables par l'utilisateur, au format numerique.
#' @param millesime_population une annee parmi les millesimes selectionnables par l'utilisateur, au format numerique.
#' @param code_reg code de la région sélectionnable par l'utilisateur.
#'
#' @return Un graphique en barres
#'
#' @importFrom attempt stop_if stop_if_not
#' @importFrom COGiter filtrer_cog
#' @importFrom dplyr mutate filter case_when rename select group_by summarise ungroup left_join arrange
#' @importFrom forcats fct_relevel fct_drop fct_inorder
#' @importFrom ggplot2 ggplot aes geom_bar position_dodge coord_flip geom_text scale_color_manual scale_fill_manual labs scale_y_continuous theme
#' @importFrom ggtext element_markdown
#' @importFrom glue glue
#' @importFrom lubridate year
#' @importFrom scales label_number
#' @importFrom tidyr gather
#'
#' @export
#'
#' @examples
#' creer_graphe_4_2(millesime_stock_artif = 2020,
#' millesime_population = 2018, code_reg = '52')


creer_graphe_4_2 <- function(millesime_stock_artif = NULL,millesime_population = NULL, code_reg = NULL){

  attempt::stop_if(millesime_stock_artif, is.null, msg = "millesime_stock_artif n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_stock_artif, is.numeric, msg = "millesime_stock_artif n\'est pas un nombre")
  attempt::stop_if(millesime_population, is.null, msg = "millesime_population n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_population, is.numeric, msg = "millesime_population n\'est pas un nombre")
  attempt::stop_if(code_reg, is.null, msg = "code_reg n\'est pas renseign\u00e9")
  attempt::stop_if_not(code_reg, ~ .x %in% levels(COGiter::regions$REG), msg = "code_reg n\'est pas un code r\u00e9gion valide")

  # # calcul des millesimes utilises
  millesime_debut <- 2016
  millesime_debut_population <- millesime_population - (millesime_stock_artif - millesime_debut)
  millesime_fin_population <- millesime_population

  # seuils population
  pop_3 <- 40000
  pop_2 <- 10000
  pop_1 <- 2000

  if (is.numeric(code_reg)) {
    code_reg = as.character(code_reg)
  }

  # creation de la fonction inverse
  `%notin%` <- Negate(`%in%`)

  if (code_reg %in% c('52')) {

  # population du dernier millesime
  population <- population_legale %>%
    dplyr::mutate(date=lubridate::year(.data$date)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$date == millesime_population,
                  .data$TypeZone =="Communes") %>%
    dplyr::mutate(seuil_pop = dplyr::case_when(
      .data$population_municipale > pop_3  ~ glue::glue("communes de plus\nde ",pop_3," habitants"),
      .data$population_municipale > pop_2  ~ glue::glue("communes entre\n",pop_2," et ",pop_3, " habitants"),
      .data$population_municipale > pop_1  ~ glue::glue("communes entre\n",pop_1," et ",pop_2, " habitants"),
      .data$population_municipale > 0  ~ glue::glue("communes de moins\nde ",pop_1," habitants"),
      TRUE ~ "")
    ) %>%
    dplyr::mutate(seuil_code = dplyr::case_when(
      .data$population_municipale > pop_3  ~ "D",
      .data$population_municipale > pop_2  ~ "C",
      .data$population_municipale > pop_1  ~ "B",
      .data$population_municipale > 0  ~ "A",
      TRUE ~ "")
    ) %>%
    dplyr::rename ("population_n"="population_municipale") %>%
    dplyr::select (-.data$date)

  # table des seuils
  seuil_population <- population %>%
    dplyr::select (.data$CodeZone,.data$seuil_pop)

  # population du dernier millesime
  population_n <- population %>%
    dplyr::select (.data$seuil_pop,.data$seuil_code, .data$population_n) %>%
    dplyr::group_by(.data$seuil_pop,.data$seuil_code) %>%
    dplyr::summarise(population_n = sum(.data$population_n, na.rm = T)) %>%
    dplyr::ungroup()

  # population du ancien millesime
  population_ancienne <- population_legale %>%
    dplyr::mutate(date=lubridate::year(.data$date)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$date == millesime_debut_population,
                  .data$TypeZone =="Communes") %>%
    dplyr::left_join(seuil_population) %>%
    dplyr::select (.data$seuil_pop, .data$population_municipale) %>%
    dplyr::group_by(.data$seuil_pop) %>%
    dplyr::summarise(population_ancienne = sum(.data$population_municipale, na.rm = T)) %>%
    dplyr::ungroup()


  # preparation donnees du stock
  # stock_artif_gk3 <- observatoire_artificialisation_gk3 %>%
  stock_depart <- stock_artificialise %>%
    dplyr::mutate(date=lubridate::year(.data$date)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$TypeZone == "Communes",
                  .data$date == millesime_debut) %>%
    dplyr::left_join(seuil_population) %>%
    dplyr::select (.data$seuil_pop, .data$surface_artificialisee) %>%
    dplyr::group_by(.data$seuil_pop) %>%
    dplyr::summarise(stock_depart = sum(.data$surface_artificialisee, na.rm = T)) %>%
    dplyr::ungroup()

  # preparation des flux
  flux <- stock_artificialise %>%
    dplyr::mutate(date=lubridate::year(.data$date)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter((.data$TypeZone == "Communes"),
                  (.data$date == millesime_debut) |(.data$date == millesime_stock_artif)) %>%
     dplyr::left_join(seuil_population) %>%
    dplyr::select ( .data$seuil_pop,.data$date, .data$surface_artificialisee) %>%
    dplyr::group_by(.data$seuil_pop , .data$date) %>%
    dplyr::summarise(surface_artificialisee = sum(.data$surface_artificialisee, na.rm = T)) %>%
    dplyr::ungroup()  %>%
    dplyr::arrange(.data$seuil_pop, .data$date) %>%
    dplyr::group_by(.data$seuil_pop) %>%
    dplyr::mutate(flux = .data$surface_artificialisee - dplyr::lag(.data$surface_artificialisee) ) %>%
    dplyr::filter(.data$date == millesime_stock_artif) %>%
    dplyr::select (.data$seuil_pop , .data$flux)


  data <-  population_n %>%
    dplyr::left_join(population_ancienne) %>%
    dplyr::left_join(stock_depart) %>%
    dplyr::left_join(flux) %>%
    dplyr::mutate(seuil_pop=as.factor(.data$seuil_pop),seuil_code=as.factor(.data$seuil_code)) %>%
    dplyr::mutate(surf_artif_par_hab_an_x=.data$stock_depart / .data$population_ancienne *10000) %>%
    dplyr::mutate(surf_artif_par_nouv_hab_entre_x_y =.data$flux / (.data$population_n - .data$population_ancienne) *10000)  %>%
    dplyr::select (.data$seuil_pop,.data$seuil_code,.data$surf_artif_par_hab_an_x,.data$surf_artif_par_nouv_hab_entre_x_y) %>%
    tidyr::gather(variable, valeur, 3:4) %>%
    dplyr::arrange(desc(.data$seuil_code)) %>%
    dplyr::mutate(variable=forcats::fct_relevel(.data$variable,"surf_artif_par_nouv_hab_entre_x_y","surf_artif_par_hab_an_x")) %>%
    dplyr::mutate(seuil_pop = forcats::fct_drop(.data$seuil_pop) %>%
                    forcats::fct_inorder())


  # creation du graphique
  valeur_max_graphique <- max(data$valeur,na.rm=T)+500

  graphe_4_2 <- data %>% ggplot2::ggplot(ggplot2::aes(x=.data$seuil_pop,y=.data$valeur,fill=.data$variable), width = 0.9) +
    ggplot2::geom_bar(stat="identity",position = ggplot2::position_dodge())+
    ggplot2::coord_flip()  +
    ggplot2::geom_text(ggplot2::aes(label = format(round(.data$valeur,0), big.mark = " "),
                                    color = .data$variable),
                       position = ggplot2::position_dodge(width=1),
                       hjust=-0.1) +
    ggplot2::scale_color_manual(values = c("#82534b","#FF8D7E")) +
    ggplot2::scale_fill_manual(values = c("#82534b","#FF8D7E")) +
    ggplot2::labs(
      title = glue::glue("Surfaces cadastr\u00e9es consomm\u00e9es (en m\u00b2)", width = 60),
      subtitle = glue::glue("<span style = \'color:#FF8D7E\'>par habitant pr\u00e9sent en {millesime_debut_population}</span> et <span style = \'color:#82534b\'>par nouvel habitant entre {millesime_debut_population} et {millesime_fin_population}</span> "),
      x = "",
      y = "",
      fill = "",
      caption = glue::glue("Source : Cerema Fichiers fonciers / Insee RP  {millesime_population}")
    ) +
    ggplot2::scale_y_continuous(
      limits = c(0, max(data$valeur) +500),
      labels = scales::label_number(big.mark = " ", decimal.mark = ",")
    ) +
    ggplot2::theme(
      legend.position = "none",
      plot.subtitle = ggtext::element_markdown(size = 12, lineheight = 1.2)
    )


  return(graphe_4_2)
}
if (code_reg  %notin% c('52')) {
  return(invisible(NULL))
}

}
