#' Creation du graphique en barres par departement des surfaces artificialisees pour un nouvel habitant.
#' @description Graphique en barres par departement des surfaces artificialisees pour un nouvel habitant.
#'
#' @param millesime_stock_artif une annee parmi les millesimes selectionnables par l'utilisateur, au format numerique.
#' @param millesime_population une annee parmi les millesimes selectionnables par l'utilisateur, au format numerique.
#' @param code_reg code de la région sélectionnable par l'utilisateur.
#'
#' @return Un graphique en barres
#'
#' @importFrom dplyr mutate filter select rename group_by summarise ungroup left_join arrange
#' @importFrom forcats fct_relevel fct_inorder
#' @importFrom ggplot2 ggplot aes geom_bar position_dodge geom_text coord_flip labs theme scale_color_manual scale_fill_manual
#' @importFrom ggtext element_markdown
#' @importFrom glue glue
#' @importFrom gouvdown scale_fill_gouv_discrete
#' @importFrom lubridate year
#' @importFrom tidyr gather
#' @importFrom COGiter filtrer_cog
#'
#' @export
#'
#' @examples
#' creer_graphe_3_4(millesime_stock_artif = 2019,
#' millesime_population = 2018, code_reg = '52')

creer_graphe_3_4 <- function(millesime_stock_artif = NULL,millesime_population = NULL, code_reg = NULL){

  attempt::stop_if(millesime_stock_artif, is.null, msg = "millesime_stock_artif n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_stock_artif, is.numeric, msg = "millesime_stock_artif n\'est pas un nombre")
  attempt::stop_if(millesime_population, is.null, msg = "millesime_population n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_population, is.numeric, msg = "millesime_population n\'est pas un nombre")
  attempt::stop_if(code_reg, is.null, msg = "code_reg n\'est pas renseign\u00e9")
  attempt::stop_if_not(code_reg, ~ .x %in% levels(COGiter::regions$REG), msg = "code_reg n\'est pas un code r\u00e9gion valide")

  # # calcul des millesimes utilises
  millesime_debut <- 2016
  millesime_debut_population <- millesime_population - (millesime_stock_artif - millesime_debut)
  millesime_fin_population <- millesime_population

  if (is.numeric(code_reg)) {
    code_reg = as.character(code_reg)
  }

  # creation de la fonction inverse
  `%notin%` <- Negate(`%in%`)

  if (code_reg %in% c('52')) {

  # preparation donnees du stock
  stock_depart <- stock_artificialise %>%
    dplyr::mutate(date=lubridate::year(.data$date)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(((.data$TypeZone == "D\u00e9partements") | (.data$TypeZone == "R\u00e9gions")),
                  .data$date == millesime_debut) %>%
    dplyr::mutate(surface_artificialisee = .data$surface_artificialisee) %>%
    dplyr::select (-.data$surf_cadastree,-.data$date) %>%
    dplyr::rename ("stock_depart"="surface_artificialisee")

  # preparation des flux
  flux <- stock_artificialise %>%
    dplyr::mutate(date=lubridate::year(.data$date)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(((.data$TypeZone == "D\u00e9partements") | (.data$TypeZone == "R\u00e9gions")),
                  (.data$date == millesime_debut) |(.data$date == millesime_stock_artif)) %>%
    dplyr::select (-.data$surf_cadastree) %>%
    dplyr::arrange(.data$TypeZone, .data$Zone, .data$CodeZone, .data$date) %>%
    dplyr::group_by(.data$TypeZone, .data$Zone, .data$CodeZone) %>%
    dplyr::mutate(flux_artificialisation = .data$surface_artificialisee - dplyr::lag(.data$surface_artificialisee) ) %>%
    dplyr::ungroup()  %>%
    dplyr::filter(.data$date == millesime_stock_artif) %>%
    dplyr::select (-.data$date , -.data$surface_artificialisee)


  # preparation des donnees population
  evol_popul <- population_legale %>%
    dplyr::mutate(date=lubridate::year(.data$date)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(((.data$TypeZone == "D\u00e9partements") | (.data$TypeZone == "R\u00e9gions")),
                  .data$date == millesime_debut_population | .data$date == millesime_fin_population)
  popul_debut <- evol_popul %>%
    dplyr::filter(.data$date == millesime_debut_population) %>%
    dplyr::rename ("population_debut"="population_municipale") %>%
    dplyr::select (-.data$date)
  evol_popul <- evol_popul %>%
    dplyr::filter(.data$date %in% c(millesime_debut_population,millesime_fin_population)) %>%
    dplyr::arrange(.data$date) %>%
    dplyr::group_by(.data$TypeZone,.data$Zone,.data$CodeZone) %>%
    dplyr::mutate(evolution_population=.data$population_municipale - dplyr::lag(.data$population_municipale))  %>%
    dplyr::ungroup() %>%
    dplyr::filter(.data$date %in% c(millesime_fin_population)) %>%
    dplyr::select(.data$TypeZone,.data$Zone,.data$CodeZone,.data$evolution_population)

  data <-  stock_depart %>%
    dplyr::left_join(flux) %>%
    dplyr::left_join(evol_popul) %>%
    dplyr::left_join(popul_debut) %>%
    dplyr::mutate(Zone = as.factor(.data$Zone)) %>%
    dplyr::mutate(surf_artif_par_hab_an_x=.data$stock_depart / .data$population_debut *10000 ) %>%
    dplyr::mutate(surf_artif_par_nouv_hab_entre_x_y =.data$flux_artificialisation / .data$evolution_population *10000) %>%
    dplyr::select (.data$TypeZone,.data$Zone,.data$surf_artif_par_hab_an_x,.data$surf_artif_par_nouv_hab_entre_x_y) %>%
    tidyr::gather(variable, valeur, 3:4) %>%
    dplyr::arrange(desc(.data$TypeZone),desc(.data$Zone)) %>%
    dplyr::mutate(variable=forcats::fct_relevel(.data$variable,"surf_artif_par_nouv_hab_entre_x_y","surf_artif_par_hab_an_x")) %>%
    dplyr::mutate(Zone = forcats::fct_drop(.data$Zone) %>%
                    forcats::fct_inorder()) %>%
    dplyr::mutate(valeur = ifelse(.data$valeur<0, 0, valeur))

  # creation du graphique
  valeur_max_graphique <- max(data$valeur,na.rm=T)+500

  graph_3_4 <- data %>%
    ggplot2::ggplot(ggplot2::aes(x = .data$Zone, y = .data$valeur,fill = .data$variable), width = 0.9) +
    ggplot2::geom_bar(stat = "identity", position = ggplot2::position_dodge()) +
    ggplot2::coord_flip() +
    ggplot2::geom_text(ggplot2::aes(label = ifelse(.data$valeur>0, format(round(.data$valeur,0), big.mark = " "), "population en baisse"),
                                    color = .data$variable),
                       position = ggplot2::position_dodge(width = 1),
                       hjust = -0.1) +
    ggplot2::scale_color_manual(values = c("#82534b","#FF8D7E")) +
    ggplot2::scale_fill_manual(values = c("#82534b","#FF8D7E")) +
    ggplot2::labs(
      title = glue::glue("Surfaces cadastr\u00e9es consomm\u00e9es (en m\u00b2)", width = 60),
      subtitle = glue::glue("<span style = \'color:#FF8D7E\'>par habitant pr\u00e9sent en {millesime_debut_population}</span> et <span style = \'color:#82534b\'>par nouvel habitant entre {millesime_debut_population} et {millesime_fin_population}</span> "),
      x = "",
      y = "",
      fill = "",
      caption = glue::glue("Source : Cerema Fichiers fonciers / Insee RP {millesime_population}")
    ) +
    ggplot2::scale_y_continuous(
      labels = scales::label_number(big.mark = " ", decimal.mark = ",")
    ) +
    ggplot2::theme(
      legend.position = "none",
      plot.subtitle = ggtext::element_markdown(size = 12, lineheight = 1.2)
    )

  return(graph_3_4)
  }
  if (code_reg  %notin% c('52')) {
    return(invisible(NULL))
  }

}
