#' Création du graphique en nuage de points de l'étalement urbain par type de territoire selon le zonage INSEE.
#' @description Graphique en nuage de points de l'étalement urbain par type de territoire selon le zonage INSEE.
#'
#' @param millesime_stock_artif une année parmi les millésimes sélectionnables par l'utilisateur, au format numerique.
#' @param millesime_population une année parmi les millésimes sélectionnables par l'utilisateur, au format numerique.
#' @param code_reg code de la région sélectionnable par l'utilisateur.
#' @param police police des titres et sous titre
#'
#' @return Un graphique nuage de points
#'
#' @importFrom COGiter filtrer_cog charger_zonage ajouter_typologie
#' @importFrom dplyr mutate filter select arrange group_by lag ungroup case_when
#' @importFrom forcats fct_drop fct_inorder
#' @importFrom ggplot2 ggplot aes geom_point scale_x_continuous scale_y_continuous labs geom_hline theme element_rect
#' @importFrom glue glue
#' @importFrom lubridate year
#' @importFrom scales number_format
#' @importFrom stringr str_detect str_split_fixed
#' @export
#'
#' @examples
#' creer_graphe_3_3(millesime_stock_artif = 2020, millesime_population = 2019, code_reg = '52')

creer_graphe_3_3 <- function(millesime_stock_artif = NULL, millesime_population = NULL, code_reg = NULL, police = "sans") {

  attempt::stop_if(millesime_stock_artif, is.null, msg = "millesime_stock_artif n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_stock_artif, is.numeric, msg = "millesime_stock_artif n\'est pas un nombre")
  attempt::stop_if(millesime_population, is.null, msg = "millesime_population n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_population, is.numeric, msg = "millesime_population n\'est pas un nombre")
  attempt::stop_if(code_reg, is.null, msg = "code_reg n\'est pas renseign\u00e9")
  attempt::stop_if_not(code_reg, ~ .x %in% levels(COGiter::regions$REG), msg = "code_reg n\'est pas un code r\u00e9gion valide")

  millesime_debut <- 2016
  millesime_population_debut <- millesime_population - (millesime_stock_artif - millesime_debut)

  if (is.numeric(code_reg)) {
    code_reg = as.character(code_reg)
  }

  # creation de la fonction inverse
  `%notin%` <- Negate(`%in%`)

  if (code_reg %in% c('52')) {

  # seuils population
  pop_3 <- 40000
  pop_2 <- 10000
  pop_1 <- 2000

  # population du millesime
  population <- population_legale %>%
    dplyr::mutate(date=lubridate::year(.data$date)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$date == millesime_population,
                  .data$TypeZone =="Communes") %>%
    dplyr::mutate(seuil_pop = dplyr::case_when(
      .data$population_municipale > pop_3  ~ glue::glue("communes de plus\nde ",pop_3," habitants"),
      .data$population_municipale > pop_2  ~ glue::glue("communes entre\n",pop_2," et ",pop_3, " habitants"),
      .data$population_municipale > pop_1  ~ glue::glue("communes entre\n",pop_1," et ",pop_2, " habitants"),
      .data$population_municipale > 0  ~ glue::glue("communes de moins\nde ",pop_1," habitants"),
      TRUE ~ "")
      ) %>%
    dplyr::mutate(seuil_code = dplyr::case_when(
      .data$population_municipale > pop_3  ~ "D",
      .data$population_municipale > pop_2  ~ "C",
      .data$population_municipale > pop_1  ~ "B",
      .data$population_municipale > 0  ~ "A",
      TRUE ~ "")
      ) %>%
    dplyr::rename ("population_n"="population_municipale") %>%
    dplyr::select (-.data$date)

  # table des seuils
  seuil_population <- population %>%
    dplyr::select (.data$CodeZone,.data$seuil_pop)

  # population du millesime
  population_n <- population %>%
    dplyr::select (.data$seuil_pop,.data$seuil_code, .data$population_n) %>%
    dplyr::group_by(.data$seuil_pop,.data$seuil_code) %>%
    dplyr::summarise(population_n = sum(.data$population_n, na.rm = T)) %>%
    dplyr::ungroup()


# preparation des donnees evolution_artificialisation

evol_artif <- stock_artificialise %>%
    dplyr::mutate(date = lubridate::year(.data$date)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$TypeZone == "Communes",
                .data$date == millesime_stock_artif | .data$date == millesime_debut) %>%
    dplyr::select (-.data$surf_cadastree)

evol_artif1 <- evol_artif %>%
  dplyr::left_join(seuil_population) %>%
  dplyr::mutate(dep = substr(.data$CodeZone,1,2)) %>%
  dplyr::filter (.data$date == millesime_stock_artif) %>%
  dplyr::group_by(.data$dep, .data$seuil_pop, .data$date) %>%
  dplyr::summarise(surface_artificialisee = sum(.data$surface_artificialisee, na.rm = T)) %>%
  dplyr::ungroup() %>%
  dplyr::left_join(COGiter::departements %>%
                     dplyr::filter (.data$REG == code_reg) %>%
                     dplyr::select(.data$DEP, .data$NOM_DEP),
                   by=c("dep"= "DEP")) %>%
  dplyr::rename("Zone" = "NOM_DEP") %>%
  dplyr::select(-.data$dep)

evol_artif1_2 <- evol_artif1 %>%
  dplyr::group_by(.data$seuil_pop,.data$date) %>%
  dplyr::summarise(surface_artificialisee = sum(.data$surface_artificialisee, na.rm = T)) %>%
  dplyr::ungroup() %>%
  dplyr::mutate(Zone = COGiter::regions %>%
                  dplyr::filter (.data$REG == code_reg) %>%
                  dplyr::pull(.data$NOM_REG))

evol_artif1 <-dplyr::bind_rows(evol_artif1,evol_artif1_2) %>%
  dplyr::select(.data$seuil_pop, .data$Zone, .data$date, .data$surface_artificialisee)

evol_artif2 <- evol_artif %>%
  dplyr::left_join(seuil_population) %>%
  dplyr::mutate(dep = substr(.data$CodeZone,1,2)) %>%
  dplyr::filter (.data$date == millesime_debut) %>%
  dplyr::group_by(.data$dep, .data$seuil_pop, .data$date) %>%
  dplyr::summarise(surface_artificialisee = sum(.data$surface_artificialisee, na.rm = T)) %>%
  dplyr::ungroup() %>%
  dplyr::left_join(COGiter::departements %>%
                     dplyr::filter (.data$REG == code_reg) %>%
                     dplyr::select(.data$DEP, .data$NOM_DEP),
                   by=c("dep"= "DEP")) %>%
  dplyr::rename("Zone" = "NOM_DEP") %>%
  dplyr::select(-.data$dep)

evol_artif2_2 <- evol_artif2 %>%
  dplyr::group_by(.data$seuil_pop, .data$date) %>%
  dplyr::summarise(surface_artificialisee = sum(.data$surface_artificialisee, na.rm = T)) %>%
  dplyr::ungroup() %>%
  dplyr::mutate(Zone = COGiter::regions %>%
                  dplyr::filter (.data$REG == code_reg) %>%
                  dplyr::pull(.data$NOM_REG))

evol_artif2 <-dplyr::bind_rows(evol_artif2,evol_artif2_2) %>%
  dplyr::select(.data$seuil_pop, .data$Zone, .data$date, .data$surface_artificialisee)

evol_artif3 <- rbind(evol_artif1,evol_artif2) %>%  #regroupement des annees
  dplyr::arrange(.data$seuil_pop, .data$Zone, .data$date) %>%
  dplyr::group_by(.data$seuil_pop, .data$Zone) %>%
  dplyr::mutate(evolution_artificialisation = round(.data$surface_artificialisee * 100 / dplyr::lag(.data$surface_artificialisee) - 100, 1)) %>%
  dplyr::ungroup() %>%
  dplyr::filter(.data$date == millesime_stock_artif) %>%
  dplyr::select(.data$seuil_pop, .data$Zone, .data$evolution_artificialisation)

# preparation des donnees evol_population

evol_popul <- population_legale %>%
  dplyr::mutate(date = lubridate::year(.data$date)) %>%
  COGiter::filtrer_cog(reg = code_reg) %>%
  dplyr::filter(.data$TypeZone == "Communes",
                .data$date == millesime_population | .data$date == millesime_population_debut)

evol_popul1 <- evol_popul %>%
  dplyr::left_join(seuil_population) %>%
  dplyr::mutate(dep = substr(.data$CodeZone,1,2)) %>%
  dplyr::filter (.data$date == millesime_population) %>%
  dplyr::group_by(.data$dep, .data$seuil_pop, .data$date) %>%
  dplyr::summarise(population_municipale = sum(.data$population_municipale, na.rm = T)) %>%
  dplyr::ungroup() %>%
  dplyr::left_join(COGiter::departements %>%
                     dplyr::filter (.data$REG == code_reg) %>%
                     dplyr::select(.data$DEP, .data$NOM_DEP),
                   by=c("dep"= "DEP")) %>%
  dplyr::rename("Zone" = "NOM_DEP") %>%
  dplyr::select(-.data$dep)

evol_popul1_2 <- evol_popul1 %>%
  dplyr::group_by(.data$seuil_pop, .data$date) %>%
  dplyr::summarise(population_municipale = sum(.data$population_municipale, na.rm = T)) %>%
  dplyr::ungroup() %>%
  dplyr::mutate(Zone = COGiter::regions %>%
                  dplyr::filter (.data$REG == code_reg) %>%
                  dplyr::pull(.data$NOM_REG))

evol_popul1 <-dplyr::bind_rows(evol_popul1,evol_popul1_2) %>%
  dplyr::select(.data$seuil_pop, .data$Zone, .data$date, .data$population_municipale)

evol_popul2 <- evol_popul %>%
  dplyr::left_join(seuil_population) %>%
  dplyr::mutate(dep = substr(.data$CodeZone,1,2)) %>%
  dplyr::filter (.data$date == millesime_population_debut) %>%
  dplyr::group_by(.data$dep, .data$seuil_pop, .data$date) %>%
  dplyr::summarise(population_municipale = sum(.data$population_municipale, na.rm = T)) %>%
  dplyr::ungroup() %>%
  dplyr::left_join(COGiter::departements %>%
                     dplyr::filter (.data$REG == code_reg) %>%
                     dplyr::select(.data$DEP, .data$NOM_DEP),
                   by=c("dep"= "DEP")) %>%
  dplyr::rename("Zone" = "NOM_DEP") %>%
  dplyr::select(-.data$dep)

evol_popul2_2 <- evol_popul2 %>%
  dplyr::group_by(.data$seuil_pop, .data$date) %>%
  dplyr::summarise(population_municipale = sum(.data$population_municipale, na.rm = T)) %>%
  dplyr::ungroup() %>%
  dplyr::mutate(Zone = COGiter::regions %>%
                  dplyr::filter (.data$REG == code_reg) %>%
                  dplyr::pull(.data$NOM_REG))

evol_popul2 <-dplyr::bind_rows(evol_popul2,evol_popul2_2) %>%
  dplyr::select(.data$seuil_pop, .data$Zone, .data$date, .data$population_municipale)

evol_popul3 <- rbind(evol_popul1,evol_popul2) %>%  #regroupement des annees
  dplyr::arrange(.data$seuil_pop, .data$Zone, .data$date) %>%
  dplyr::group_by(.data$seuil_pop, .data$Zone) %>%
  dplyr::mutate(evolution_population = round(.data$population_municipale * 100 / dplyr::lag(.data$population_municipale) - 100, 1)) %>%
  dplyr::ungroup() %>%
  dplyr::filter(.data$date == millesime_population) %>%
  dplyr::select(.data$seuil_pop, .data$Zone, .data$evolution_population)

data_final <- evol_artif3 %>%
  dplyr::full_join(evol_popul3 %>% dplyr::select(.data$seuil_pop, .data$Zone, .data$evolution_population)) %>%
  dplyr::mutate(`Seuil de population` = factor(.data$seuil_pop, levels = c(glue::glue("communes de moins\nde ",pop_1," habitants"),
                                                               glue::glue("communes entre\n",pop_1," et ",pop_2, " habitants"),
                                                               glue::glue("communes entre\n",pop_2," et ",pop_3, " habitants"),
                                                               glue::glue("communes de plus\nde ",pop_3," habitants"))))


#creation du graphique
graphe_3_3 <- data_final %>%
  ggplot2::ggplot(ggplot2::aes(x = .data$evolution_artificialisation, y = .data$evolution_population, color = .data$Zone, shape = .data$`Seuil de population`)) +
  ggplot2::geom_point(size = 5)+
  ggplot2::scale_x_continuous(name = "Taux de croissance de l\'artificialisation", labels = scales::number_format(suffix = " %", accuracy = 1)) +
  ggplot2::scale_y_continuous(name = "Taux de croissance de la population", labels = scales::number_format(suffix = " %", accuracy = 1)) +
  gouvdown::scale_color_gouv_discrete(palette = "pal_gouv_qual2") +
  ggplot2::labs(title=glue::glue("Croissance compar\u00e9e de la population et de  la consommation d\u2019espaces cadastr\u00e9s par d\u00e9partement\net selon la taille des communes"),
                subtitle= glue::glue("Entre {millesime_debut} et {millesime_stock_artif}"),
                caption = glue::glue("Source : Cerema Fichiers fonciers / Insee RP {millesime_population}")
                # x = "",
                # fill = ""
                ) +
  ggplot2::geom_hline(yintercept = 0, color = "grey", size=0.8) +
  ggplot2::theme(panel.background = ggplot2::element_rect(fill = "white",
                                                          colour = "grey",
                                                          size = 0.5,
                                                          linetype = "solid"))

return(graphe_3_3)
}
if (code_reg  %notin% c('52')) {
  return(invisible(NULL))
  }

  }
