#' Création du graphique du taux de croissance comparee des surfaces artificialisees et de la population par département.
#' @description Graphique du taux de croissance comparee des surfaces artificialisees et de la population par département.
#'
#' @param millesime_stock_artif une année parmi les millésimes sélectionnables par l'utilisateur, au format numerique.
#' @param millesime_population une année parmi les millésimes sélectionnables par l'utilisateur, au format numerique.
#' @param code_reg code insee de la région sur laquelle construire le graphique
#' @param police police des titres et sous titre
#' @param code_reg code de la région sélectionnable par l'utilisateur.
#'
#' @return Un graphique lollipop
#'
#' @importFrom dplyr mutate filter select arrange group_by ungroup full_join lag
#' @importFrom forcats fct_drop fct_inorder
#' @importFrom ggplot2 ggplot aes geom_point position_dodge geom_linerange geom_label coord_flip scale_y_continuous labs theme element_blank element_text scale_color_manual
#' @importFrom ggtext element_markdown element_textbox_simple
#' @importFrom glue glue
#' @importFrom lubridate year
#' @importFrom mapfactory format_fr_pct
#' @importFrom scales label_number
#' @importFrom tidyr pivot_longer
#' @importFrom attempt stop_if stop_if_not
#' @export
#'
#' @examples
#' creer_graphe_3_1(millesime_stock_artif=2020, millesime_population = 2018, code_reg = '52')
creer_graphe_3_1 <- function(millesime_stock_artif = NULL,
                             millesime_population = NULL,
                             code_reg = NULL,
                             police = "sans") {

  attempt::stop_if(millesime_stock_artif, is.null, msg = "millesime_stock_artif n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_stock_artif, is.numeric, msg = "millesime_stock_artif n\'est pas un nombre")
  attempt::stop_if(millesime_population, is.null, msg = "millesime_population n\'est pas renseign\u00e9")
  attempt::stop_if_not(millesime_population, is.numeric, msg = "millesime_population n\'est pas un nombre")
  attempt::stop_if(code_reg, is.null, msg = "code_reg n\'est pas renseign\u00e9")
  attempt::stop_if_not(code_reg, ~ .x %in% levels(COGiter::regions$REG), msg = "code_reg n\'est pas un code r\u00e9gion valide")

    millesime_debut <- 2016
    millesime_population_debut <- millesime_population - (millesime_stock_artif - millesime_debut)

  if (is.numeric(code_reg)) {
    code_reg = as.character(code_reg)
  }

    # creation de la fonction inverse
    `%notin%` <- Negate(`%in%`)

    if (code_reg %in% c('52')) {

  # Creation de la table utile a la production du graphique
  evol_artif <- stock_artificialise %>%
    dplyr::mutate(date = lubridate::year(.data$date)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$TypeZone %in% c("France","R\u00e9gions", "D\u00e9partements"),
           .data$date == millesime_stock_artif | .data$date == millesime_debut) %>%
    dplyr::select (-.data$surf_cadastree) %>%
    dplyr::arrange(.data$TypeZone, .data$Zone, .data$CodeZone, .data$date) %>%
    dplyr::group_by(.data$TypeZone, .data$Zone, .data$CodeZone) %>%
    dplyr::mutate(evolution_artificialisation = round(.data$surface_artificialisee * 100 / dplyr::lag(.data$surface_artificialisee) - 100, 1)) %>%
    dplyr::ungroup() %>%
    dplyr::filter(.data$date == millesime_stock_artif) %>%
    dplyr::select(.data$TypeZone, .data$CodeZone, .data$Zone, .data$evolution_artificialisation)


  evol_popul <- population_legale %>%
    dplyr::mutate(date = lubridate::year(.data$date)) %>%
    COGiter::filtrer_cog(reg = code_reg) %>%
    dplyr::filter(.data$TypeZone %in% c("R\u00e9gions", "D\u00e9partements"),
           .data$date == millesime_population | .data$date == millesime_population_debut) %>%
    dplyr::arrange(.data$TypeZone, .data$Zone, .data$CodeZone, .data$date) %>%
    dplyr::group_by(.data$TypeZone, .data$Zone, .data$CodeZone) %>%
    dplyr::mutate(evolution_population = round(.data$population_municipale * 100 / dplyr::lag(.data$population_municipale) - 100, 1)) %>%
    dplyr::ungroup() %>%
    dplyr::filter(.data$date == millesime_population) %>%
    dplyr::select(.data$TypeZone, .data$CodeZone, .data$Zone, .data$evolution_population)


  data <- dplyr::full_join(evol_artif, evol_popul) %>%
    tidyr::pivot_longer(cols = .data$evolution_artificialisation:.data$evolution_population, names_to = "indicateur", values_to = "valeur")

  y_min <- min(data$valeur) * 1.2

  gg <- data %>%
    dplyr::arrange(.data$TypeZone, desc(.data$Zone)) %>%
    dplyr::mutate(Zone = forcats::fct_drop(.data$Zone) %>%
      forcats::fct_inorder()) %>%
    ggplot2::ggplot() +
    ggplot2::aes(x = .data$Zone, y = .data$valeur, group = .data$indicateur, color = .data$indicateur) +
    ggplot2::geom_point(size = 2, position = ggplot2::position_dodge(width = 0.5)) +
    ggplot2::aes(x = .data$Zone, ymin = 0, ymax = .data$valeur, group = .data$indicateur, color = .data$indicateur) +
    ggplot2::geom_linerange(size = 1.2, position = ggplot2::position_dodge(width = 0.5)) +
    ggplot2::geom_hline(yintercept = 0, color = "grey", size = 0.8) +
    ggplot2::aes(label = mapfactory::format_fr_pct_2(.data$valeur, accuracy = 0.1)) +
    ggplot2::geom_label(
      position = ggplot2::position_dodge(width = 0.5),
      vjust = 0.5, hjust = -0.5
    ) +
    ggplot2::coord_flip() +
    ggplot2::scale_y_continuous(
      breaks = seq(y_min, max(data$valeur) * 1.2, by = 1.5),
      minor_breaks = NULL,
      limits = c(y_min, max(data$valeur) * 1.2),
      labels = scales::label_number(big.mark = " ", decimal.mark = ",", suffix = " %"),
      expand = c(0,0)
    ) +
    ggplot2::labs(
      title = "Taux de croissance compar\u00e9s des <span style = \'color:#FF8D7E\'>cadastr\u00e9es consomm\u00e9es</span> <br>et de la <span style = \'color:#5770BE\'>population municipale</span>",
      subtitle = glue::glue("Entre {millesime_debut} et {millesime_stock_artif} (en %)"),
      caption = glue::glue("Source : Cerema Fichiers fonciers / Insee RP {millesime_population}"),
      x = NULL, y = NULL
    ) +
    ggplot2::theme(
      panel.grid.major.y = ggplot2::element_blank(),
      legend.position = "none",
      plot.title = ggtext::element_textbox_simple(size = 14, lineheight = 1, hjust = 0, family = police),
      plot.subtitle = ggplot2::element_text(size = 12, family = police),
      plot.caption = ggplot2::element_text(size = 10, family = police),
    ) +
    ggplot2::scale_color_manual(values = c("#FF8D7E", "#5770BE"))
  return(gg)}

  if (code_reg  %notin% c('52')) {
    return(invisible(NULL))
  }

}
